package monalipse.server.giko;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import monalipse.MonalipsePlugin;
import monalipse.editors.ThreadViewerEditor;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.IBBSReference;
import monalipse.server.ILinkedLineFragment;
import monalipse.server.IResponseEnumeration;
import monalipse.server.IThreadContentProvider;
import monalipse.server.LinkedObject;
import monalipse.server.Response;
import monalipse.utils.CancelableRunner;
import monalipse.widgets.ColoredText;
import org.apache.commons.httpclient.Cookie;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

class ThreadContentProvider implements IAdaptable, IThreadContentProvider, IBBSReference
{
	private static final Logger logger = MonalipsePlugin.getLogger();

	private static final HttpClient SUBMIT_HTTP_CLIENT = GikoServer.createHttpClient();

	private Board board;
	private URL boardURL;
	private String urlHint;
	private String id;
	private int index;
	private String name;
	private int responses;

	public ThreadContentProvider(Board board, URL boardURL, String id, int index, String name, int responses)
	{
		this.board = board;
		this.boardURL = boardURL;
		this.id = id;
		this.index = index;
		this.name = name;
		this.responses = responses;
	}
	
	public boolean isActive()
	{
		return board.containsID(id);
	}
	
	private String getURLHint()
	{
		if(urlHint == null)
		{
			String baseURL = boardURL.toExternalForm();
			int ss = baseURL.indexOf("//");
			if(ss != -1 && id.endsWith(".dat"))
			{
				int s = baseURL.indexOf('/', ss + 2);
				urlHint = baseURL.substring(0, s) + "/test/read.cgi" +
							baseURL.substring(s, baseURL.length()) +
							id.substring(0, id.length() - 4) + "/";
			}
			else
			{
				urlHint = "";
			}
		}
		return urlHint;
	}
	
	public URL getURL()
	{
		try
		{
			return new URL(getURLHint() + "l50");
		}
		catch (MalformedURLException e)
		{
			throw new RuntimeException(e);
		}
	}

	public IFolder getLogFolder()
	{
		return board.getLogFolder();
	}
	
	public IFile getLogFile()
	{
		return getLogFolder().getFile(id);
	}

	public IBBSBoard getBoard()
	{
		return board;
	}

	public String getID()
	{
		return id;
	}
	
	public int getIndex()
	{
		return index;
	}

	public String getName()
	{
		if(name == null)
			name = getResponses((List)null, 0, 0);
		if(name == null)
			return "";
		else
			return name;
	}
	
	public int getResponseNumber()
	{
		return -1;
	}
	
	public int getResponseCountHint()
	{
		return responses;
	}
	
	public boolean hasNewResponses()
	{
		return 0 < getCachedCount() && getCachedCount() < getResponseCountHint();
	}
	
	public boolean threadChanged(IResourceChangeEvent event)
	{
		return MonalipsePlugin.resourceModified(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED, event.getDelta(), getLogFile());
	}
	
	public Object getAdapter(Class adapter)
	{
		return null;
	}
	
	public int hashCode()
	{
		return id.hashCode();
	}
	
	public boolean equals(Object obj)
	{
		if(obj instanceof ThreadContentProvider)
		{
			ThreadContentProvider thread = (ThreadContentProvider) obj;
			return thread.getLogFolder().equals(getLogFolder()) && thread.getID().equals(getID());
		}
		return false;
	}
	
	public LinkedObject[] getLinkedObjects()
	{
		List list = new ArrayList();

		LogFile log = null;
		try
		{
			IFile file = getLogFile();
			MonalipsePlugin.ensureSynchronized(file);
			if(file.exists())
			{
				log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
				if(log != null)
				{
					for(int i = 0; i < log.responseCount; i++)
						log.readLinkedObject(this, list);
				}
			}
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
		if(log != null)
			log.close();

		LinkedObject[] objects = new LinkedObject[list.size()];
		list.toArray(objects);
		return objects;
	}
	
	public boolean submitResponse(CancelableRunner.ICancelableProgressMonitor monitor, String name, String mail, String body)
	{
		synchronized(SUBMIT_HTTP_CLIENT)
		{
			monitor.beginTask("submitting...", 6);
			try
			{
				monitor.worked(1);

				URL url = new URL(getBoard().getURL(), "../test/bbs.cgi");
				String cookiePath = url.getFile();
				cookiePath = cookiePath.substring(0, cookiePath.lastIndexOf('/'));

				SUBMIT_HTTP_CLIENT.getState().addCookie(new Cookie(url.getHost(), "NAME", name, cookiePath, null, false));
				SUBMIT_HTTP_CLIENT.getState().addCookie(new Cookie(url.getHost(), "MAIL", mail, cookiePath, null, false));
				
				String success = "\u66f8\u304d\u3053\u307f\u307e\u3057\u305f";
				String retry = "\u78ba\u8a8d";
				
				BBSServerManager.configureHttpClient(SUBMIT_HTTP_CLIENT, url);
				
				monitor.subTask("POST");
				PostMethod post = createResponsePostData(monitor, url, name, mail, body);

				try
				{
					SUBMIT_HTTP_CLIENT.executeMethod(post);
					MonalipsePlugin.log(logger, post);
					System.err.println(post.getResponseBodyAsString());

					if(post.getResponseBodyAsString().indexOf(retry) != -1)
					{
						monitor.subTask("retry POST with sid");
						post.releaseConnection();
						post = createResponsePostData(monitor, url, name, mail, body);
						BBSServerManager.configureHttpClient(SUBMIT_HTTP_CLIENT, url);
						SUBMIT_HTTP_CLIENT.executeMethod(post);
						MonalipsePlugin.log(logger, post);
						System.err.println(post.getResponseBodyAsString());
					}
	
					return post.getResponseBodyAsString().indexOf(success) != -1;
				}
				finally
				{
					post.releaseConnection();
				}
			}
			catch (MalformedURLException e)
			{
				e.printStackTrace();
			}
			catch (UnsupportedEncodingException e)
			{
				e.printStackTrace();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
			finally
			{
				monitor.done();
			}
			
			return false;
		}
	}
	
	private PostMethod createResponsePostData(CancelableRunner.ICancelableProgressMonitor monitor, URL url, String name, String mail, String body)
	{
		PostMethod post = GikoServer.createPostMethod();
		post.setPath(url.getFile());

		post.setRequestHeader("Referer", getBoard().getURL().toExternalForm());

		String bbs = getBoard().getURL().getFile();
		post.addParameter("submit", "\u66f8\u304d\u8fbc\u3080");
		post.addParameter("bbs", bbs.substring(1, bbs.length() - 1));
		post.addParameter("key", id.substring(0, id.lastIndexOf('.')));
		post.addParameter("time", String.valueOf(System.currentTimeMillis() / 1000));
		post.addParameter("FROM", name);
		post.addParameter("mail", mail);
		post.addParameter("MESSAGE", body);
		
//		String sid = GikoServer.getOysterSessionID(monitor.getRunner().getSubProgressMonitor(monitor, 2));
//		if(sid != null)
//			post.addParameter("sid", sid);

		return post;	
	}
	
	public IResponseEnumeration getResponses(CancelableRunner.ICancelableProgressMonitor monitor, int sequence, int rangeStart)
	{
		LogFile log = null;
		try
		{
			IFile file = getLogFile();
			MonalipsePlugin.ensureSynchronized(file);
			if(file.exists())
			{
				log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
				if(log != null)
				{
					boolean partial = log.sequence == sequence;
					if(partial)
					{
						try
						{
							if(log.responseCount < rangeStart)
								throw new IOException();
							for(int i = 0; i < rangeStart; i++)
								log.skipResponse();
						}
						catch(IOException e)
						{
							log.close();
							log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
							return new ThreadLogReader(monitor, log.title, log, false, log.sequence, log.isActive);
						}
					}
					return new ThreadLogReader(monitor, log.title, log, partial, log.sequence, log.isActive);
				}
			}
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
		if(log != null)
			log.close();

		return new NullResponseEnumeration(getURL().toExternalForm(), sequence);
	}
	
	public String getResponses(List responseReceiver, int start, int end)
	{
		LogFile log = null;
		String title = null;
		try
		{
			IFile file = getLogFile();
			MonalipsePlugin.ensureSynchronized(file);
			if(file.exists())
			{
				log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
				if(log != null)
				{
					title = log.title;
					for(int i = 1; i < start && i < log.responseCount + 1; i++)
						log.skipResponse();
					for(int i = start; 1 <= i && i <= end && i < log.responseCount + 1; i++)
						responseReceiver.add(log.readResponse());
				}
			}
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		finally
		{
			if(log != null)
				log.close();
		}
		
		return title;
	}
	
	public IResponseEnumeration updateResponses(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart)
	{
		AccessContext context = new AccessContext();
		context.progressRemaining = 100;
		context.title = getURL().toExternalForm();

		monitor.beginTask("downloading...", context.progressRemaining);
		
		IResponseEnumeration e;
		
		e = getPartialContent(monitor, workbenchWindow, sequence, rangeStart, context);
		if(e != null)
			return e;
		
		sequence++;

		if(!context.notFound)
		{
			e = getFullContent(monitor, workbenchWindow, sequence, rangeStart, context);
			if(e != null)
				return e;
		}
		
		List boardURLs = new ArrayList();
		boardURLs.add(getBoard().getURL().toExternalForm());
		if(!boardURLs.contains(boardURL.toExternalForm()))
			boardURLs.add(boardURL.toExternalForm());
		String[] obs = board.getObsoleteURLs();
		for(int i = 0; i < obs.length; i++)
		{
			if(!boardURLs.contains(obs[i]))
				boardURLs.add(obs[i]);
		}
			
		for(int i = 0; i < boardURLs.size(); i++)
		{
			try
			{
				URL url = new URL((String)boardURLs.get(i));
				
				e = getPublicLogContent(monitor, workbenchWindow, sequence, rangeStart, context, url);
				if(e != null)
					return e;
				
				e = getOysterLogContent(monitor, workbenchWindow, sequence, rangeStart, context, url);
				if(e != null)
					return e;
			}
			catch (MalformedURLException ex)
			{
			}
		}

		return new NullResponseEnumeration(context.title, sequence);
	}
	
	private static class AccessContext
	{
		public int progressRemaining;
		public boolean notFound;
		public String lastModified;
		public String title;
	}
	
	private static String getResponseHeaderString(Header header)
	{
		if(header == null)
			return null;
		else
			return header.getValue();
	}

	private IResponseEnumeration getPartialContent(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart, AccessContext context)
	{
		IFile file = getLogFile();
		
		try
		{
			MonalipsePlugin.ensureSynchronized(file);
			if(file.exists())
			{
				LogFile log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
				try
				{
					if(log != null)
					{
						context.title = log.title;
						context.lastModified = log.lastModifierd;
						boolean partial = log.sequence == sequence;
						if(partial)
						{
							List logs = new ArrayList();
							Response lastResponse = null;

							if(rangeStart <= log.responseCount)
							{
								while(log.available())
								{
									lastResponse = log.readResponse();
									logs.add(lastResponse);
								}
							}

							if(lastResponse != null)
							{
								HttpClient hc = GikoServer.createHttpClient();

								URL url = new URL(board.getURL(), "dat/" + id);
								BBSServerManager.configureHttpClient(hc, url);

								GetMethod get = GikoServer.createGetMethod();

								try
								{
									get.setPath(url.getFile());
									get.setRequestHeader("Range", "bytes=" + log.httpRangeStart + "-");
									get.setRequestHeader("If-Modified-Since", context.lastModified);
									context.progressRemaining -= 10;
									monitor.worked(10);
									monitor.subTask("GET " + url);
	
									logger.finest("partial " + url);
									hc.executeMethod(get);
									MonalipsePlugin.log(logger, get);
									if(get.getStatusLine().getStatusCode() == 206)
									{
										RangeAnalyzeInputStream cin = new RangeAnalyzeInputStream(log.httpRangeStart, get.getResponseBodyAsStream());
										BufferedReader r = new BufferedReader(new InputStreamReader(cin, "Windows-31J"));
										int contentLength = Integer.parseInt(getResponseHeaderString(get.getResponseHeader("Content-Length")));
										Response resp = LogFile.parseResponse(r, logs.size(), getURLHint(), null);
										if(resp != null && lastResponse.equals(resp))
										{
											IResponseEnumeration e = new ThreadDownloader(monitor.getRunner().getSubProgressMonitor(monitor, context.progressRemaining), workbenchWindow, log.title, get, r, cin, true, log.sequence, getResponseHeaderString(get.getResponseHeader("Last-Modified")), logs, rangeStart, contentLength, file, true);
											get = null;
											return e;
										}
										else
										{
											r.close();
										}
									}
									else if(get.getStatusLine().getStatusCode() == 304)
									{
										return new ArrayThreadReader(log.title, true, log.sequence, logs, rangeStart, true);
									}
									else if(get.getStatusLine().getStatusCode() == 404 || get.getStatusLine().getStatusCode() == 302)
									{
										context.notFound = true;
									}
								}
								finally
								{
									if(get != null)
										get.releaseConnection();
								}
							}
						}
					}
				}
				finally
				{
					if(log != null)
						log.close();
				}
			}
		}
		catch (MalformedURLException e)
		{
			e.printStackTrace();
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
			context.lastModified= null;
		}
		
		return null;
	}
	
	private IResponseEnumeration getFullContent(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart, AccessContext context)
	{
		context.progressRemaining -= 10;
		monitor.worked(10);
		try
		{
			URL url = new URL(board.getURL(), "dat/" + id);
			monitor.subTask("GET " + url);
			logger.finest("full " + url);
			return getResponse(url, GikoServer.createGetMethod(), context.lastModified, monitor.getRunner().getSubProgressMonitor(monitor, context.progressRemaining), workbenchWindow, context.title, sequence, getLogFile(), false, true);
		}
		catch (MalformedURLException ex)
		{
		}
		return null;
	}
	
	private IResponseEnumeration getPublicLogContent(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart, AccessContext context, URL boardURL)
	{
		context.progressRemaining -= 10;
		monitor.worked(10);
		try
		{
			URL url;
			if(14 <= id.length())
				url = new URL(boardURL, "kako/" + id.substring(0, 4) + "/" + id.substring(0, 5) + "/" + id + ".gz");
			else if(id.length() == 13)
				url = new URL(boardURL, "kako/" + id.substring(0, 3) + "/" + id + ".gz");
			else
				return null;
			
			monitor.subTask("GET " + url);
			logger.finest("log " + url);
			return getResponse(url, GikoServer.createGetMethod(), context.lastModified, monitor.getRunner().getSubProgressMonitor(monitor, context.progressRemaining), workbenchWindow, context.title, sequence, getLogFile(), false, false);
		}
		catch (MalformedURLException e)
		{
		}

		return null;
	}
	
	private IResponseEnumeration getOysterLogContent(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, int sequence, int rangeStart, AccessContext context, URL boardURL)
	{
		String sid = GikoServer.getOysterSessionID(monitor.getRunner().getSubProgressMonitor(monitor, 10));
		context.progressRemaining -= 10;
		if(sid != null)
		{
			context.progressRemaining -= 10;
			monitor.worked(10);
			try
			{
				URL url = new URL(boardURL, "/test/offlaw.cgi");

				GetMethod get = GikoServer.createGetMethod();
				String bbs = boardURL.getFile();
				get.setQueryString(new NameValuePair[]
					{
						new NameValuePair("bbs", bbs.substring(1, bbs.length() - 1)),
						new NameValuePair("key", id.substring(0, id.lastIndexOf('.'))),
						new NameValuePair("sid", sid),
						new NameValuePair("raw", "0.0"),
					});
				monitor.subTask("GET " + url);
				logger.finest("oyster " + url);
				return getResponse(url, get, context.lastModified, monitor.getRunner().getSubProgressMonitor(monitor, context.progressRemaining), workbenchWindow, context.title, sequence, getLogFile(), true, false);
			}
			catch (MalformedURLException e)
			{
			}
		}
		
		return null;
	}
	
	private IResponseEnumeration getResponse(URL url, HttpMethod method, String lastModified, CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, String title, int sequence, IFile file, boolean oyster, boolean active)
	{
		try
		{
			if(lastModified != null)
				method.setRequestHeader("If-Modified-Since", lastModified);

			HttpClient hc = GikoServer.createHttpClient();
			try
			{
				method.setPath(url.getFile());
				method.setRequestHeader("Accept-Encoding", "gzip");
				BBSServerManager.configureHttpClient(hc, url);
				hc.executeMethod(method);
				MonalipsePlugin.log(logger, method);
				
				if(method.getStatusCode() == 200)
				{
					RangeAnalyzeInputStream cin = new RangeAnalyzeInputStream(0, GikoServer.getResponseBodyAsStream(method));
					BufferedReader r = new BufferedReader(new InputStreamReader(cin, "Windows-31J"));
					int contentLength = 0;
					String enc = getResponseHeaderString(method.getResponseHeader("Content-Encoding"));
					String len = getResponseHeaderString(method.getResponseHeader("Content-Length"));
					if(enc != null && enc.equals("gzip") && oyster)
						contentLength = readContentLength(r);
					else if(len != null)
						contentLength = Integer.parseInt(len);
					if(0 < contentLength)
					{
						IResponseEnumeration e = new ThreadDownloader(monitor, workbenchWindow, title, method, r, cin, false, sequence, getResponseHeaderString(method.getResponseHeader("Last-Modified")), new ArrayList(), 0, contentLength, file, active);
						method = null;
						return e;
					}
				}
				else if(method.getStatusCode() == 304)
				{
					return new NullResponseEnumeration(getURL().toExternalForm(), sequence);
				}
			}
			finally
			{
				if(method != null)
					method.releaseConnection();
			}
		}
		catch (MalformedURLException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
		return null;
	}

	private int readContentLength(Reader r) throws IOException
	{
		if(r.read() != '+' || r.read() != 'O' || r.read() != 'K' || r.read() != ' ')
			throw new IOException();
		int ch = r.read();
		StringBuffer buf = new StringBuffer();
		while(ch != '/')
		{
			buf.append((char)ch);
			ch = r.read();
		}
		ch = r.read();
		while(ch != '/')
			ch = r.read();
		r.read();
		return Integer.parseInt(buf.toString());
	}
	
	public int getCachedCount()
	{
		try
		{
			String cached = getLogFile().getPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), "cached"));
			if(cached != null)
				return Integer.parseInt(cached);
		}
		catch(CoreException e)
		{
		}
		catch(NumberFormatException e)
		{
		}

		try
		{
			IFile file = getLogFile();
			if(file.exists())
			{
				LogFile log = LogFile.of(new DataInputStream(new BufferedInputStream(file.getContents())));
				try
				{
					if(log != null)
						return log.responseCount;
				}
				finally
				{
					if(log != null)
						log.close();
				}
			}
		}
		catch (CoreException e)
		{
		}
		catch (IOException e)
		{
		}
		
		return 0;
	}
	
	public void prefetchToolTip(ColoredText.ToolTipTarget target)
	{
		if(target instanceof HoverLineFragment)
			((HoverLineFragment)target).prefetchToolTip();
	}
	
	public Point fillToolTip(Composite parent, ColoredText sourceText, int maxWidth, ColoredText.ToolTipTarget target, String sourceTitle)
	{
		if(target instanceof HoverLineFragment)
			return ((HoverLineFragment)target).fillToolTip(parent, sourceText, maxWidth, sourceTitle);
		else
			return null;
	}
	
	public Point fillToolTip(Composite parent, ColoredText sourceText, int maxWidth, String selection, String sourceTitle)
	{
		return HoverLineFragment.fillToolTip(parent, sourceText, maxWidth, selection, sourceTitle);
	}
	
	private class ThreadDownloader implements IResponseEnumeration, CancelableRunner.ICancelableRunnableWithProgress
	{
		private CancelableRunner.ICancelableProgressMonitor monitor;
		private IWorkbenchWindow workbenchWindow;
		private String title;
		private BufferedReader r;
		private RangeAnalyzeInputStream cin;
		private boolean partial;
		private int sequence;
		private String lastModified;
		private List log;
		private IFile logFile;
		private int position;
		private int contentRange;
		private int contentLength;
		private boolean closed;
		private boolean active;
		private HttpMethod method;
		private boolean getLog;

		public ThreadDownloader(CancelableRunner.ICancelableProgressMonitor monitor, IWorkbenchWindow workbenchWindow, String title, HttpMethod method, BufferedReader r, RangeAnalyzeInputStream cin, boolean partial, int sequence, String lastModified, List log, int position, int contentLength, IFile logFile, boolean active)
		{
			this.monitor = monitor;
			this.workbenchWindow = workbenchWindow;
			this.title = title;
			this.method = method;
			this.r = r;
			this.cin = cin;
			this.partial = partial;
			this.sequence = sequence;
			this.lastModified = lastModified;
			this.log = log;
			this.position = position;
			this.contentLength = contentLength;
			this.logFile = logFile;
			this.active = active;
			contentRange = cin.getRange();
			monitor.getRunner().run(null, this);
		}

		public void run(CancelableRunner.ICancelableProgressMonitor mon)
		{
			try
			{
				try
				{
					monitor.beginTask("", 100);
					int worked = 0;
					String delim = null;
					while(!mon.isCanceled())
					{
						Response response = LogFile.parseResponse(r, log.size() + 1, getURLHint(), delim);
						if(response == null)
							break;
						getLog = true;
						log.add(response);
						delim = response.getDelimiter();
						if(log.size() == 1)
							title = response.getTitle();

						synchronized(this)
						{
							notifyAll();
						}
						Thread.yield();
						
						int w;
						if(contentLength == 0)
							w = 100;
						else
							w = Math.min((cin.getRange() - contentRange) * 100 / contentLength, 100);
						if(worked != w)
							monitor.worked(w - worked);
						worked = w;
					}
					monitor.done();
				}
				catch (IOException e)
				{
				}
			}
			finally
			{
				synchronized(this)
				{
					closed = true;
					notifyAll();
					try
					{
						r.close();
						method.releaseConnection();
					}
					catch (IOException e)
					{
					}
				}
			}
		}

		private void setLog(int pos) throws IOException
		{
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			DataOutputStream dout = new DataOutputStream(bout);
			LogFile.writeLogFile(dout, sequence, title, lastModified, active, pos, log);
			dout.close();
			final byte[] bytes = bout.toByteArray();
			MonalipsePlugin.asyncExec(workbenchWindow, new WorkspaceModifyOperation()
				{
					protected void execute(IProgressMonitor monitor) throws InvocationTargetException
					{
						try
						{
							board.createLogFolder(workbenchWindow, monitor);
							IFile cache = getLogFile();
							MonalipsePlugin.ensureSynchronized(cache);
							if(cache.exists())
								cache.setContents(new ByteArrayInputStream(bytes), false, false, monitor);
							else
								cache.create(new ByteArrayInputStream(bytes), false, monitor);
						}
						catch (CoreException e)
						{
							throw new InvocationTargetException(e);
						}
					}
				});
		}
		
		public int getSequenceNumber()
		{
			return sequence;
		}
		
		public String getTitle()
		{
			return title;
		}
	
		public boolean isPartialContent()
		{
			return partial;
		}

		public boolean isReady()
		{
			return !closed && position < log.size();
		}

		public boolean hasNextResponse()
		{
			return !closed || position < log.size();
		}

		public Response getNextResponse() throws InterruptedException
		{
			synchronized(this)
			{
				while(!closed && position == log.size())
					wait();
			}
			if(position < log.size())
				return (Response)log.get(position++);
			else
				return null;
		}
		
		public void close()
		{
			try
			{
				if(getLog)
				{
					try
					{
						getLogFile().setPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), "cached"), String.valueOf(log.size()));
					}
					catch (CoreException e)
					{
					}
					setLog(cin.getRange());
				}
			}
			catch (IOException e)
			{
			}
		}
		
		public boolean isWritable()
		{
			return active;
		}
	}
	
	private class ArrayThreadReader implements IResponseEnumeration
	{
		private String title;
		private boolean partial;
		private int sequence;
		private List log;
		private int position;
		private boolean active;

		public ArrayThreadReader(String title, boolean partial, int sequence, List log, int position, boolean active)
		{
			this.title = title;
			this.partial = partial;
			this.sequence = sequence;
			this.log = log;
			this.position = position;
			this.active = active;
		}
		
		public int getSequenceNumber()
		{
			return sequence;
		}
		
		public String getTitle()
		{
			return title;
		}
	
		public boolean isPartialContent()
		{
			return partial;
		}

		public boolean isReady()
		{
			return position < log.size();
		}

		public boolean hasNextResponse()
		{
			return position < log.size();
		}

		public Response getNextResponse() throws InterruptedException
		{
			if(position < log.size())
				return (Response)log.get(position++);
			else
				return null;
		}
		
		public void close()
		{
		}
		
		public boolean isWritable()
		{
			return active;
		}
	}
	
	private static class ThreadLogReader implements IResponseEnumeration
	{
		private CancelableRunner.ICancelableProgressMonitor monitor;
		private String title;
		private LogFile log;
		private int sequence;
		private boolean partial;
		private boolean active;
		
		public ThreadLogReader(CancelableRunner.ICancelableProgressMonitor monitor, String title, LogFile log, boolean partial, int sequence, boolean active)
		{
			this.monitor = monitor;
			this.title = title;
			this.log = log;
			this.sequence = sequence;
			this.partial = partial;
			this.active = active;
			monitor.beginTask("rendering...", log.responseCount);
		}
		
		public int getSequenceNumber()
		{
			return sequence;
		}
		
		public String getTitle()
		{
			return title;
		}
		
		public boolean isPartialContent()
		{
			return partial;
		}

		public boolean isReady()
		{
			return hasNextResponse();
		}

		public boolean hasNextResponse()
		{
			return log != null && log.available();
		}

		public Response getNextResponse()
		{
			try
			{
				monitor.worked(1);
				return log.readResponse();
			}
			catch (IOException e)
			{
				close();
				return null;
			}
		}
		
		public void close()
		{
			if(log != null)
				log.close();
			log = null;
			if(monitor != null)
				monitor.done();
			monitor = null;
		}
		
		public boolean isWritable()
		{
			return active;
		}
	}
	
	private static class NullResponseEnumeration implements IResponseEnumeration
	{
		private String title;
		private int sequence;

		public NullResponseEnumeration(String title, int sequence)
		{
			this.title = title;
			this.sequence = sequence;
		}

		public int getSequenceNumber()
		{
			return sequence;
		}
		
		public String getTitle()
		{
			return title;
		}

		public boolean isPartialContent()
		{
			return true;
		}

		public boolean isReady()
		{
			return false;
		}

		public boolean hasNextResponse()
		{
			return false;
		}

		public Response getNextResponse()
		{
			return null;
		}
		
		public void close()
		{
		}
		
		public boolean isWritable()
		{
			return false;
		}
	}
	
	// int version
	// int sequence (for "a bone")
	// UTF title
	// UTF Last-Modified
	// int active (writable)
	// int HTTP range start
	// int <responseCount>
	// {
	//  UTF text
	//  @fragments name
	//  @fragments mail
	//  @fragments date
	//  int <lineCount>
	//  {
	//   @fragments line
	//  }*
	// }*
	//
	//
	// #fragments
	// int <fragmentCount>
	// {
	//  int offset
	//  int length
	//  int style
	//  UTF linkText
	//  int color
	//  int font
	// }*
	private static class LogFile
	{
		private static final int LOG_FILE_VERSION = 15;
		private static final int RESPONSE_BODY_INDENT = 32;
		private static final int STYLE_UNDERLINE = 0x01;
		private static final int STYLE_LINK = 0x02;

		private DataInputStream din;
		public int version;
		public int sequence;
		public String title;
		public String lastModifierd;
		public boolean isActive;
		public int httpRangeStart;
		public int responseCount;
		private int responseRead;
		
		public static LogFile of(DataInputStream din) throws IOException
		{
			LogFile res = null;
			try
			{
				int version = din.readInt();
				if(version == LOG_FILE_VERSION)
					res = new LogFile(din);
			}
			finally
			{
				if(res == null)
					din.close();
			}
			return res;
		}
		
		private LogFile(DataInputStream din) throws IOException
		{
			this.din = din;
			
			version = LOG_FILE_VERSION;
			sequence = din.readInt();
			title = din.readUTF();
			lastModifierd = din.readUTF();
			isActive = din.readInt() != 0;
			httpRangeStart = din.readInt();
			responseCount = din.readInt();
		}
		
		public void close()
		{
			try
			{
				if(din != null)
					din.close();
				din = null;
			}
			catch (IOException e)
			{
			}
		}
		
		public boolean available()
		{
			return responseRead < responseCount;
		}
		
		public void skipResponse() throws IOException
		{
			responseRead++;
			din.skipBytes(din.readShort());
			skipFragments();
			skipFragments();
			skipFragments();
			int lineCount = din.readInt();
			for(int i = 0; i < lineCount; i++)
				skipFragments();
		}
		
		private void skipFragments() throws IOException
		{
			int fragmentCount = din.readInt();
			for(int j= 0; j < fragmentCount; j++)
			{
				din.skipBytes(12);
				din.skipBytes(din.readShort() + 8);
			}
		}
		
		public Response readResponse() throws IOException
		{
			responseRead++;
			String text = din.readUTF();
			ColoredText.LineFragment[] name = readLineFragments(text);
			ColoredText.LineFragment[] mail = readLineFragments(text);
			ColoredText.LineFragment[] date = readLineFragments(text);
			ColoredText.Line[] body = new ColoredText.Line[din.readInt()];
			for(int i = 0; i < body.length; i++)
			{
				ColoredText.Line line = new ColoredText.Line(RESPONSE_BODY_INDENT);
				int fragmentCount = din.readInt();
				for(int j= 0; j < fragmentCount; j++)
					line.addLineFragment(readLineFragment(text));
				body[i] = line;
			}
			return new Response(responseRead, name, mail, date, body, null, null);
		}
		
		private ColoredText.LineFragment[] readLineFragments(String text) throws IOException
		{
			ColoredText.LineFragment[] fragments = new ColoredText.LineFragment[din.readInt()];
			for(int i = 0; i < fragments.length; i++)
				fragments[i] = readLineFragment(text);
			return fragments;
		}
		
		private ColoredText.LineFragment readLineFragment(String text) throws IOException
		{
			int offset = din.readInt();
			int length = din.readInt();
			int style = din.readInt();
			boolean isLink = (style & STYLE_LINK) != 0;
			String link = "";
			if(isLink)
				link = din.readUTF();
			else
				din.skipBytes(din.readShort());
			int color = din.readInt();
			int font = din.readInt();

			if(isLink)
			{
				if(color == ThreadViewerEditor.ThreadViewerAttributes.COLOR_LINK)
					return new LinkedLineFragment(text, offset, length, color, font, (style & STYLE_UNDERLINE) != 0, false, link);
				else
					return new HoverLineFragment(text, offset, length, color, font, (style & STYLE_UNDERLINE) != 0, false, link);
			}
			else
			{
				return new MarkedLineFragment(text, offset, length, color, font, (style & STYLE_UNDERLINE) != 0, false);
			}
		}
		
		public void readLinkedObject(ThreadContentProvider thread, List receiver) throws IOException
		{
			responseRead++;
			Map urls = new HashMap();
			din.skipBytes(din.readShort());
			readLinkedLineFragments(thread, receiver, urls);
			readLinkedLineFragments(thread, receiver, urls);
			readLinkedLineFragments(thread, receiver, urls);
			int lineCount = din.readInt();
			for(int i = 0; i < lineCount; i++)
				readLinkedLineFragments(thread, receiver, urls);
		}
		
		private void readLinkedLineFragments(ThreadContentProvider thread, List receiver, Map urls) throws IOException
		{
			int fragmentCount = din.readInt();
			for(int j= 0; j < fragmentCount; j++)
				readLinkedLineFragment(thread, receiver, urls);
		}
		
		private void readLinkedLineFragment(ThreadContentProvider thread, List receiver, Map urls) throws IOException
		{
			din.skipBytes(8);
			int style = din.readInt();
			boolean isLink = (style & STYLE_LINK) != 0;
			if(isLink)
			{
				String link = din.readUTF();
				try
				{
					if(!urls.containsKey(link) && LinkedObject.matches(link))
					{
						receiver.add(new LinkedObject(thread, responseRead, new URL(link)));
						urls.put(link, link);
					}
				}
				catch (MalformedURLException e)
				{
				}
			}
			else
			{
				din.skipBytes(din.readShort());
			}
			din.skipBytes(8);
		}
		
		private static int skipWhitespace(Reader r) throws IOException
		{
			int ch = r.read();
			while(ch != -1 && Character.isWhitespace((char)ch))
				ch = r.read();
			return ch;
		}
		
		private static void convertMatches(String text, List receiver, int color, int font, PatternAction[] patterns, boolean newMark, String urlBase)
		{
			int done = 0;
			
			Matcher[] matchers = new Matcher[patterns.length];
			for(int i = 0; i < matchers.length; i++)
				matchers[i] = patterns[i].matcher(text);
			
			boolean[] matchs = new boolean[matchers.length];
			for(int i = 0; i < matchs.length; i++)
				matchs[i] = patterns[i].next(matchers[i]);
			
			while(true)
			{
				int start = Integer.MAX_VALUE;
				int first = -1;

				for(int i = 0; i < matchs.length; i++)
				{
					if(matchs[i] && matchers[i].start() < start)
					{
						start = matchers[i].start();
						first = i;
					}
				}
				
				if(first == -1)
					break;
				
				int end = matchers[first].end();
				
				if(done <= start)
				{
					String link = patterns[first].getLink(urlBase, matchers[first]);
					
					receiver.add(new MarkedLineFragment(text, done, start - done, color, font, false, newMark));
					if(start < end)
					{
						if(patterns[first].isLink())
							receiver.add(new LinkedLineFragment(text, start, end - start, ThreadViewerEditor.ThreadViewerAttributes.COLOR_LINK, font, true, newMark, link));
						else
							receiver.add(new HoverLineFragment(text, start, end - start, color, font, true, newMark, link));
					}
	
					done = end;
				}
				
				matchs[first] = patterns[first].next(matchers[first]);
				if(matchs[first] && matchers[first].start() == start)
					matchs[first] = false;
			}

			receiver.add(new MarkedLineFragment(text, done, text.length() - done, color, font, false, newMark));
		}
		
		private static void addLine(List lineReceiver, List fragments)
		{
			ColoredText.Line line = new ColoredText.Line(RESPONSE_BODY_INDENT);
			for(int j = 0; j < fragments.size(); j++)
				line.addLineFragment((ColoredText.LineFragment)fragments.get(j));
			lineReceiver.add(line);
		}
		
		private static ColoredText.LineFragment[] readToken(String text, int start, int end, List lineReceiver, int color, int font, PatternAction[] patterns, boolean newMark, String urlBase, boolean escapeComma) throws IOException
		{
			List fragments = new ArrayList();

			StringBuffer buf = new StringBuffer(128);
			
			while(start < end && text.charAt(start) == ' ')
				start++;

			for(int i = start; i < end; i++)
			{
				char ch = text.charAt(i);
	
				if(ch == '<')
				{
					int e = text.indexOf('>', i + 1);
					if(end <= e || e == -1)
					{
						buf.append('<');
						continue;
					}
					
					String tag = text.substring(i + 1, e).toLowerCase();
					i = e;

					if(0 < buf.length() && buf.charAt(buf.length() - 1) == ' ' && tag.equals("br"))
						buf.deleteCharAt(buf.length() - 1);

					if(0 < buf.length())
					{
						if(patterns == null)
							fragments.add(new MarkedLineFragment(buf.toString(), 0, buf.length(), color, font, false, newMark));
						else
							convertMatches(buf.toString(), fragments, color, font, patterns, newMark, urlBase);
						buf = new StringBuffer(128);
					}

					if(tag.equals("b"))
					{
						font = ThreadViewerEditor.ThreadViewerAttributes.FONT_BOLD;
					}
					else if(tag.equals("/b"))
					{
						font = ThreadViewerEditor.ThreadViewerAttributes.FONT_NORMAL;
					}
					else if(tag.startsWith("font") && tag.indexOf("red") != -1)
					{
						color = ThreadViewerEditor.ThreadViewerAttributes.COLOR_MARKING;
					}
					else if(tag.startsWith("font") && tag.indexOf("forestgreen") != -1)
					{
						color = ThreadViewerEditor.ThreadViewerAttributes.COLOR_NAME;
					}
					else if(lineReceiver != null && tag.equals("br"))
					{
						addLine(lineReceiver, fragments);
						fragments.clear();
						while(i + 1 < end && text.charAt(i + 1) == ' ')
							i++;
					}
					else if(tag.startsWith("a "))
					{
					}
				}
				else if(ch == '&' && i + 1 < end)
				{
					i++;
					if(text.charAt(i) == '#' && i + 1 < end)
					{
						i++;
						int radix = 10;
						if(text.charAt(i) == 'x' && i + 1 < end)
						{
							i++;
							radix = 16;
						}
						
						int e = i;
						while(e < end && Character.isDigit(text.charAt(e)) && e - i < 5)
							e++;
						
						try
						{
							buf.append((char)Integer.parseInt(text.substring(i, e), radix));

							if(e < end && text.charAt(e) == ';')
								i = e;
							else
								i = e - 1;
						}
						catch (NumberFormatException ex)
						{
							buf.append("&#");
							if(radix == 16)
								buf.append('x');
							i--;
						}
					}
					else
					{
						int e = i;
						while(e < end && Character.isLetter(text.charAt(e)) && e - i < 6)
							e++;
						
						String ref = text.substring(i, e);
						if(ref.equals("lt"))
							buf.append('<');
						else if(ref.equals("gt"))
							buf.append('>');
						else if(ref.equals("amp"))
							buf.append('&');
						else if(ref.equals("apos"))
							buf.append('\'');
						else if(ref.equals("quot"))
							buf.append('"');
						else if(ref.equals("hearts"))
							buf.append('\u2665');
						else if(ref.equals("thinsp"))
							buf.append('\u2009');
						else if(ref.equals("nbsp"))
							buf.append(' ');
						else if(ref.equals("ensp"))
							buf.append('\u2002');
						else if(ref.equals("emsp"))
							buf.append('\u2003');
						else
							buf.append('&').append(ref + (text.charAt(e) == ';' ? ";" : ""));
						
						if(e < end && text.charAt(e) == ';')
							i = e;
						else
							i = e - 1;
					}
				}
				else
				{
					if(escapeComma && ch == '\uff20' && i + 1 < end && text.charAt(i + 1) == '\uff40')
					{
						i++;
						ch = ',';
					}

					buf.append((char)ch);

					if(ch == ' ')
					{
						while(i + 1 < end && text.charAt(i + 1) == ' ')
							i++;
					}
				}
			}

			if(0 < buf.length() && buf.charAt(buf.length() - 1) == ' ' && lineReceiver != null)
				buf.deleteCharAt(buf.length() - 1);

			if(0 < buf.length())
			{
				if(patterns == null)
					fragments.add(new MarkedLineFragment(buf.toString(), 0, buf.length(), color, font, false, newMark));
				else
					convertMatches(buf.toString(), fragments, color, font, patterns, newMark, urlBase);
			}
			
			if(lineReceiver != null)
			{
				addLine(lineReceiver, fragments);
				fragments.clear();
			}
			
			ColoredText.LineFragment[] f = new ColoredText.LineFragment[fragments.size()];
			fragments.toArray(f);
			return f;
		}
		
		public static String getDelimiterOf(String line)
		{
			if(line == null)
				return null;
			else if(line.indexOf("<>") != -1)
				return "<>";
			else
				return ",";
		}

		public static Response parseResponse(BufferedReader r, int responseNumber, String urlBase, String delim) throws IOException
		{
			try
			{
				String line = r.readLine();
				if(line == null)
					return null;
				
				if(delim == null)
					delim = getDelimiterOf(line);
				
				int nameEnd = line.indexOf(delim);
				int mailEnd = line.indexOf(delim, nameEnd + delim.length());
				int dateEnd = line.indexOf(delim, mailEnd + delim.length());
				int bodyEnd = line.lastIndexOf(delim);
				
				boolean escapeComma = delim.equals(",");
				
				if(nameEnd == -1 || mailEnd == -1 || dateEnd == -1 || bodyEnd == -1)
				{
					String error = "(error)";
					ColoredText.LineFragment[] name = readToken(error, 0, error.length(), null, ThreadViewerEditor.ThreadViewerAttributes.COLOR_NAME, ThreadViewerEditor.ThreadViewerAttributes.FONT_BOLD, NAME_PATTERN_SET, false, urlBase, escapeComma);
					ColoredText.LineFragment[] mail = readToken(error, 0, error.length(), null, ThreadViewerEditor.ThreadViewerAttributes.COLOR_MAIL, ThreadViewerEditor.ThreadViewerAttributes.FONT_BOLD, MAIL_PATTERN_SET, false, urlBase, escapeComma);
					ColoredText.LineFragment[] date = readToken(error, 0, error.length(), null, ThreadViewerEditor.ThreadViewerAttributes.COLOR_BODY, ThreadViewerEditor.ThreadViewerAttributes.FONT_NORMAL, DATE_PATTERN_SET, true, urlBase, escapeComma);
					List lineReceiver = new ArrayList();
					readToken(error, 0, error.length(), lineReceiver, ThreadViewerEditor.ThreadViewerAttributes.COLOR_BODY, ThreadViewerEditor.ThreadViewerAttributes.FONT_NORMAL, BODY_PATTERN_SET, false, urlBase, escapeComma);
					ColoredText.Line[] body = new ColoredText.Line[lineReceiver.size()];
					lineReceiver.toArray(body);
					return new Response(responseNumber, name, mail, date, body, delim, null);
				}
				else
				{
					ColoredText.LineFragment[] name = readToken(line, 0, nameEnd, null, ThreadViewerEditor.ThreadViewerAttributes.COLOR_NAME, ThreadViewerEditor.ThreadViewerAttributes.FONT_BOLD, NAME_PATTERN_SET, false, urlBase, escapeComma);
					ColoredText.LineFragment[] mail = readToken(line, nameEnd + delim.length(), mailEnd, null, ThreadViewerEditor.ThreadViewerAttributes.COLOR_MAIL, ThreadViewerEditor.ThreadViewerAttributes.FONT_BOLD, MAIL_PATTERN_SET, false, urlBase, escapeComma);
					ColoredText.LineFragment[] date = readToken(line, mailEnd + delim.length(), dateEnd, null, ThreadViewerEditor.ThreadViewerAttributes.COLOR_BODY, ThreadViewerEditor.ThreadViewerAttributes.FONT_NORMAL, DATE_PATTERN_SET, true, urlBase, escapeComma);
					List lineReceiver = new ArrayList();
					readToken(line, dateEnd + delim.length(), bodyEnd, lineReceiver, ThreadViewerEditor.ThreadViewerAttributes.COLOR_BODY, ThreadViewerEditor.ThreadViewerAttributes.FONT_NORMAL, BODY_PATTERN_SET, false, urlBase, escapeComma);
					ColoredText.Line[] body = new ColoredText.Line[lineReceiver.size()];
					lineReceiver.toArray(body);
					String title = null;
					if(bodyEnd + delim.length() < line.length())
						title = line.substring(bodyEnd + delim.length(), line.length());
					return new Response(responseNumber, name, mail, date, body, delim, title);
				}
			}
			catch(RuntimeException e)
			{
				return null;
			}
		}
		
		private static void writeLogFile(DataOutputStream dout, int sequence, String title, String lastModified, boolean isActive, int httpRangeStart, List logs) throws IOException
		{
			dout.writeInt(LOG_FILE_VERSION);
			dout.writeInt(sequence);
			dout.writeUTF(title);
			dout.writeUTF(lastModified);
			dout.writeInt(isActive ? 1 : 0);
			dout.writeInt(httpRangeStart);
			dout.writeInt(logs.size());
			for(int i = 0; i < logs.size(); i++)
			{
				Response resp = (Response)logs.get(i);
				ColoredText.LineFragment[] name = resp.getName();
				ColoredText.LineFragment[] mail = resp.getMail();
				ColoredText.LineFragment[] date = resp.getDate();
				ColoredText.Line[] body = resp.getBody();

				StringBuffer buf = new StringBuffer();
				appendLineFragments(buf, name);
				appendLineFragments(buf, mail);
				appendLineFragments(buf, date);
				for(int j = 0; j < body.length; j++)
					buf.append(body[j].getText());
				dout.writeUTF(buf.toString());

				int textOffset = 0;
				
				textOffset = writeLineFragments(dout, textOffset, name);
				textOffset = writeLineFragments(dout, textOffset, mail);
				textOffset = writeLineFragments(dout, textOffset, date);

				dout.writeInt(body.length);
				for(int j = 0; j < body.length; j++)
				{
					ColoredText.Line line = body[j];
					dout.writeInt(line.getLineFragmentCount());
					for(int k = 0; k < line.getLineFragmentCount(); k++)
						textOffset = writeLineFragment(dout, textOffset, line.getLineFragmentAt(k));
				}
			}
		}
		
		private static void appendLineFragments(StringBuffer buf, ColoredText.LineFragment[] fragments)
		{
			for(int i = 0; i < fragments.length; i++)
				buf.append(fragments[i].getText());
		}
		
		private static int writeLineFragments(DataOutputStream dout, int textOffset, ColoredText.LineFragment[] fragments) throws IOException
		{
			dout.writeInt(fragments.length);
			for(int i = 0; i < fragments.length; i++)
				textOffset = writeLineFragment(dout, textOffset, fragments[i]);
			return textOffset;
		}
		
		private static int writeLineFragment(DataOutputStream dout, int textOffset, ColoredText.LineFragment fragment) throws IOException
		{
			dout.writeInt(textOffset);
			dout.writeInt(fragment.getTextLength());
			int style = fragment.getUnderline() ? STYLE_UNDERLINE : 0;
			if(fragment instanceof HoverLineFragment)
			{
				dout.writeInt(style | STYLE_LINK);
				dout.writeUTF(((HoverLineFragment)fragment).getURL());
			}
			else
			{
				dout.writeInt(style);
				dout.writeUTF("");
			}
			dout.writeInt(fragment.getColor());
			dout.writeInt(fragment.getFont());
			return textOffset + fragment.getTextLength();
		}
	}
	
	private static String toASCIIDigits(String str)
	{
		StringBuffer buf = new StringBuffer();
		for(int i = 0; i < str.length(); i++)
		{
			char ch = str.charAt(i);
			if('\uff10' <= ch && ch <= '\uff19')
				ch = (char)('0' + (ch - '\uff10'));
			buf.append(ch);
		}
		return buf.toString();
	}

	private static final PatternAction NUM_REF_PATTERN = new PatternAction(Pattern.compile("(>|\uff1e)+(([0-9\uff10-\uff19]+)(->*([0-9\uff10-\uff19]+))?)"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				String start = toASCIIDigits(m.group(3));
				String end = m.group(5);
				if(end == null)
					end = start;
				else
					end = toASCIIDigits(end);
				return urlBase + start + "-" + end;
			}
		};
	private static final PatternAction NAME_NUM_REF_PATTERN = new PatternAction(Pattern.compile("[0-9\uff10-\uff19]+"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				String num = toASCIIDigits(m.group());
				return urlBase + num + "-" + num;
			}
			
			public boolean next(Matcher m)
			{
				return m.matches();
			}
			
			public boolean isLink()
			{
				return false;
			}
		};
	private static final PatternAction TRIP_DECL_PATTERN = new PatternAction(Pattern.compile("\u25C6([\\p{Alnum}\\./]{8,10})"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				return "trip:" + m.group(1);
			}
			
			public boolean next(Matcher m)
			{
				return m.find();
			}
			
			public boolean isLink()
			{
				return false;
			}
		};
	private static final PatternAction URL_REF_PATTERN = new PatternAction(Pattern.compile("(((h?t)?t)?p)?(s?://\\p{Alnum}[\\p{Alnum}\\.\\-_:]+(/[\\p{Alnum}!#%&'*+,-./:;=?@\\\\^_`\\|~]*)?)"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				return "http" + m.group(4);
			}
		};
	private static final PatternAction ID_DECL_PATTERN = new PatternAction(Pattern.compile("ID:(.{8})"))
		{
			public String getLink(String urlBase, Matcher m)
			{
				return "id:" + m.group(1).trim();
			}
			
			public boolean isLink()
			{
				return false;
			}
		};
	private static final PatternAction[] NAME_PATTERN_SET = new PatternAction[]{NUM_REF_PATTERN, NAME_NUM_REF_PATTERN, TRIP_DECL_PATTERN};
	private static final PatternAction[] MAIL_PATTERN_SET = new PatternAction[]{NUM_REF_PATTERN};
	private static final PatternAction[] DATE_PATTERN_SET = new PatternAction[]{ID_DECL_PATTERN};
	private static final PatternAction[] BODY_PATTERN_SET = new PatternAction[]{NUM_REF_PATTERN, URL_REF_PATTERN};
	
	private static abstract class PatternAction
	{
		private Pattern pattern;
		
		public PatternAction(Pattern pattern)
		{
			this.pattern = pattern;
		}
		
		public Matcher matcher(String text)
		{
			return pattern.matcher(text);
		}
		
		public boolean next(Matcher m)
		{
			return m.find();
		}
		
		public boolean isLink()
		{
			return true;
		}
		
		public abstract String getLink(String urlBase, Matcher m);
	}
	
	private static class LinkedLineFragment extends HoverLineFragment implements ColoredText.LinkTarget
	{
		public LinkedLineFragment(String text, int offset, int length, int color, int font, boolean underline, boolean newMark, String href)
		{
			super(text, offset, length, color, font, underline, newMark, href);
		}
		
		public int getTargetResponseNumber()
		{
			String href = super.getURL();
			String pos = href.substring(href.lastIndexOf('/') + 1, href.length());
			Pattern p = Pattern.compile("(\\p{Digit}+)(-(\\p{Digit}+)?)?");
			Matcher m = p.matcher(pos);
			if(m.lookingAt())
				return Integer.parseInt(m.group(1));
			else
				return -1;
		}
	}
	
	private static class HoverLineFragment extends MarkedLineFragment implements ColoredText.ToolTipTarget, ILinkedLineFragment
	{
		private String href;

		public HoverLineFragment(String text, int offset, int length, int color, int font, boolean underline, boolean newMark, String href)
		{
			super(text, offset, length, color, font, underline, newMark);
			this.href = href;
		}
		
		public void prefetchToolTip()
		{
		}
		
		public String getURL()
		{
			return href;
		}
		
		public int getTargetResponseNumber()
		{
			return -1;
		}
		
		public Point fillToolTip(Composite parent, ColoredText sourceText, int maxWidth, String sourceTitle)
		{
			List responses = new ArrayList();
			String toolTipTitle = null;

			if(href.startsWith("id:") || href.startsWith("trip:"))
			{
				for(int i = 0; i < sourceText.getLineCount(); i++)
				{
					ColoredText.Line line = sourceText.getLineAt(i);
					if(line instanceof Response.HeaderLine)
					{
						Response resp = ((Response.HeaderLine)line).getResponse();
						int fragment = href.startsWith("id:") ? Response.FRAGMENT_DATE : Response.FRAGMENT_NAME;
						if(resp.hasLink(fragment, href))
							responses.add(resp);
					}
				}
			}
			else if(href.startsWith("http"))
			{
				try
				{
					IThreadContentProvider cp = BBSServerManager.getThreadContentProviderOf(new URL(href));
					if(cp != null)
					{
						String pos = href.substring(href.lastIndexOf('/') + 1, href.length());
						Pattern p = Pattern.compile("(\\p{Digit}+)(-(\\p{Digit}+)?)?");
						Matcher m = p.matcher(pos);
						String targetTitle;
						if(m.lookingAt())
						{
							String s = m.group(1);
							int start = Integer.parseInt(s);

							int end = start;
							String e = m.group(3);
							if(e != null)
								end = Integer.parseInt(e);
							
							if(end < start)
							{
								int t = start;
								start = end;
								end = t;
							}
							
							targetTitle = cp.getResponses(responses, start, end);
						}
						else
						{
							targetTitle = cp.getResponses(responses, 1, 1);
						}
						
						if(targetTitle != null && sourceTitle != null && !sourceTitle.equals(targetTitle))
							toolTipTitle = targetTitle;
						
						if(responses.size() == 0)
						{
							if(0 < cp.getName().length())
								return ThreadViewerEditor.createColoredTextToolTip(parent, sourceText, maxWidth, cp.getName(), null);
						}
					}
				}
				catch (MalformedURLException e)
				{
				}
			}
			
			Response[] r = new Response[responses.size()];
			responses.toArray(r);
			return createToolTip(parent, sourceText, maxWidth, toolTipTitle, r);
		}
		
		public static Point fillToolTip(Composite parent, ColoredText sourceText, int maxWidth, String selection, String sourceTitle)
		{
			List responses = new ArrayList();

			try
			{
				int num = Integer.parseInt(selection.trim());

				boolean copy = false;
				
				for(int i = 0; i < sourceText.getLineCount(); i++)
				{
					ColoredText.Line line = sourceText.getLineAt(i);
					if(line instanceof Response.HeaderLine)
					{
						Response.HeaderLine rhl = (Response.HeaderLine)line;
						if(num == rhl.getReponseNumber())
							responses.add(rhl.getResponse());
					}
				}
			}
			catch(NumberFormatException e)
			{
			}
			
			if(responses.size() == 0)
			{
				Response.HeaderLine rhl = null;
				
				for(int i = 0; i < sourceText.getLineCount(); i++)
				{
					ColoredText.Line line = sourceText.getLineAt(i);
					if(line instanceof Response.HeaderLine)
						rhl = (Response.HeaderLine)line;

					if(rhl != null && line.toString().indexOf(selection) != -1)
					{
						responses.add(rhl.getResponse());
						rhl = null;
					}
				}
			}

			Response[] r = new Response[responses.size()];
			responses.toArray(r);
			return createToolTip(parent, sourceText, maxWidth, null, r);
		}

		private static Point createToolTip(Composite parent, ColoredText sourceText, int maxWidth, String title, Response[] responses)
		{
			if(0 < responses.length)
				return ThreadViewerEditor.createColoredTextToolTip(parent, sourceText, maxWidth, title, responses);
			else
				return null;
		}

	}
	
	private static class MarkedLineFragment extends ColoredText.LineFragment implements Response.INewResponseLineFragment
	{
		private int color;
		private boolean newMark;

		public MarkedLineFragment(String text, int offset, int length, int color, int font, boolean underline, boolean newMark)
		{
			super(text, offset, length, newMark ? ThreadViewerEditor.ThreadViewerAttributes.COLOR_MARKING : color, font, underline);
			this.color = color;
			this.newMark = newMark;
		}

		public void unmark()
		{
			if(newMark)
				setColor(color);
		}
		
		public int getColor()
		{
			return color;
		}
	}
	
	private static class RangeAnalyzeInputStream extends InputStream
	{
		private int range;
		private int lf;
		private int count;
		private InputStream in;

		public RangeAnalyzeInputStream(int range, InputStream in)
		{
			this.range = range;
			this.in = in;
			lf = range;
			count = range;
		}
		
		public int getRange()
		{
			return range;
		}
		
		public int read() throws IOException
		{
			int r = in.read();
			if(r == '\n')
			{
				range = lf;
				lf = count + 1;
			}
			count++;
			return r;
		}
		
		public int read(byte[] b) throws IOException
		{
			int r = in.read(b);
			for(int i = 0; i < r; i++)
			{
				if(b[i] == '\n')
				{
					range = lf;
					lf = count + i + 1;
				}
			}
			count += r;
			return r;
		}
		
		public int read(byte[] b, int off, int len) throws IOException
		{
			int r = in.read(b, off, len);
			int end = off + len;
			for(int i = off; i < end; i++)
			{
				if(b[i] == '\n')
				{
					range = lf;
					lf = count + i - off + 1;
				}
			}
			count += r;
			return r;
		}
	}

}
