<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * コントローラクラスです。
 * アクションの設定・起動を行います。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Controller
{
    /**
     * コンストラクタです。
     * 
     * @access public
     */
    function Moony_Controller()
    {}

    /**
     * 実際の処理を行います。アクションを起動します。
     * また、指定されたアクションに対応するクラスが存在しない場合は、
     * 無条件にアクション名に紐付くテンプレートを表示させます。
     * 
     * @access public
     * @param string $action_name アクション名称
     * @param array $path_info PATH_INFOの値の配列
     */
    function process($action_name, $path_info)
    {
        // フレームワーク情報をHTTPヘッダ出力
        header('X-Framework: Moony/' . MOONY_VERSION);

        // テンプレート名称初期値
        $template = Moony_Utils::buildPath('', $action_name, MOONY_SMARTY_TEMPLATE_EXT);

        // Session生成
        $session = new Moony_Session();
        if (MOONY_SESSION_AUTO_START) {
            // 設定に応じて開始
            $session->start();
        }

        // Request生成、値をセット
        $request = new Moony_Request();
        $request->session =& $session;
        $request->path_info = $path_info;

        // Response生成、デフォルトテンプレート名称セット
        $response = new Moony_Response();
        $response->setTemplate($template);

        // View生成、プラグイン登録
        $view = new Moony_View();
        Moony_SmartyPlugin::regist($view);

        // アクション実行
        $action = $this->_createAction($action_name);
        $this->_filterAction($action, $request);
        $this->_validateAction($action, $request, $view);
        $this->_executeAction($action, $request, $response, $view);

        // 画面表示
        $view->render($response);

        exit(0);
    }

    /**
     * アクションクラスのインスタンスを生成します。
     *
     * @access private
     * @param string $action_name アクション名
     * @return object アクションクラスのインスタンス（生成に失敗した場合null）
     */
    function _createAction($action_name)
    {
        $action = null;
        $action_class_name = $action_name . 'Action';
        $action_class_path = Moony_Utils::buildPath(MOONY_ACTION_DIR, $action_class_name, MOONY_ACTION_EXT);
        if (file_exists($action_class_path)) {
            include_once($action_class_path);
            if (class_exists($action_class_name)) {
                $action = new $action_class_name;
            }
        }
        return $action;
    }

    /**
     * 入力値の変換処理を行います。
     *
     * @access private
     * @param object $action アクションクラスのインスタンス
     * @param object $request Moony_Requestのインスタンス
     */
    function _filterAction(&$action, &$request)
    {
        if (!is_null($action)) {
            $action->filter(&$request);
        }
    }

    /**
     * 入力値の妥当性チェックを行います。
     * エラーが存在する場合、前回表示した画面を再表示して処理を終了します。
     *
     * @access private
     * @param object $action アクションクラスのインスタンス
     * @param object $request Moony_Requestのインスタンス
     * @param object $view Moony_Viewのインスタンス
     */
    function _validateAction(&$action, &$request, &$view)
    {
        $session = $request->getSession();
         if ($session->hasStarted() && !is_null($action)) {
            $errors =& new Moony_Messages();
            if (!$action->validate(&$request, &$errors)) {
                $this->_validationErrorProcess($request, $view, $errors);
            }
        }
    }

    /**
     * アクションの処理を実行します。
     *
     * @access private
     * @param object $action アクションクラスのインスタンス
     * @param object $request Moony_Requestのインスタンス
     * @param object $response Moony_Responseのインスタンス
     * @param object $view Moony_Viewのインスタンス
     */
    function _executeAction(&$action, &$request, &$response, &$view)
    {
        if (!is_null($action)) {

            // 実行
            $action->execute(&$request, &$response);

            // トランザクショントークン保存
            Moony_Token::save($request, $response);

            if (strlen($response->buffer) > 0) {
                // テンプレート表示を行わずにそのまま出力して終了
                if (!headers_sent()) {
                    header('Content-Type: text/plain');
                }
                echo $response->buffer;
                exit(0);
            }

            // セッションに格納しておく
            $session = $request->getSession();
            $session->setResponse($response);

        } else {
            $view->caching = 1;
        }
    }

    /**
     * 入力値の妥当性チェックでエラーになった場合の処理を行います。
     * 前回表示した画面を再表示して処理を終了します。
     *
     * @access private
     * @param object $request Moony_Requestのインスタンス
     * @param object $view Moony_Viewのインスタンス
     * @param object $errors Moony_messagesのインスタンス
     */
    function _validationErrorProcess(&$request, &$view, &$errors)
    {
        $session = $request->getSession();
        $response = $session->getResponse();
        // パラメータ設定し直し
        $params = $request->getAll();
        foreach ($params as $key => $value) {
            $response->set($key, $value);
        }
        $response->set(MOONY_VALIDATION_ERROR_KEY, $errors->getAll());
        $view->render($response);
        exit(0);
    }
}
?>
