<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

// ログ出力フォーマットの定義
define('MOONY_LOG_FORMAT_LINES', "%s %s %s - %s(L:%s)\n");
define('MOONY_LOG_FORMAT_CALLER', "%s %s %s - %s\n");
define('MOONY_LOG_FORMAT_NORMAL', "%s %s %s\n");

/**
 * ログ出力のためのクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Logger
{
    /**
     * 致命的エラーのログをテキストファイルに出力します。
     *
     * @access public
     * @static
     * @param string $message メッセージ
     * @param string $caller 呼び出し元に関する情報
     * @param integer $line 呼び出し元の行に関する情報
     */
    function fatal($message, $caller = null, $line = null)
    {
        Moony_Logger::log(MOONY_LOG_LEVEL_FATAL, $message, $caller, $line);
    }

    /**
     * エラーログをテキストファイルに出力します。
     *
     * @access public
     * @static
     * @param string $message メッセージ
     * @param string $caller 呼び出し元に関する情報
     * @param integer $line 呼び出し元の行に関する情報
     */
    function error($message, $caller = null, $line = null)
    {
        Moony_Logger::log(MOONY_LOG_LEVEL_ERROR, $message, $caller, $line);
    }

    /**
     * 警告ログをテキストファイルに出力します。
     *
     * @access public
     * @static
     * @param string $message メッセージ
     * @param string $caller 呼び出し元に関する情報
     * @param integer $line 呼び出し元の行に関する情報
     */
    function warn($message, $caller = null, $line = null)
    {
        Moony_Logger::log(MOONY_LOG_LEVEL_WARN, $message, $caller, $line);
    }

    /**
     * 通常レベルのログをテキストファイルに出力します。
     *
     * @access public
     * @static
     * @param string $message メッセージ
     * @param string $caller 呼び出し元に関する情報
     * @param integer $line 呼び出し元の行に関する情報
     */
    function info($message, $caller = null, $line = null)
    {
        Moony_Logger::log(MOONY_LOG_LEVEL_INFO, $message, $caller, $line);
    }

    /**
     * デバッグ用のログをテキストファイルに出力します。
     *
     * @access public
     * @static
     * @param string $message メッセージ
     * @param string $caller 呼び出し元に関する情報
     * @param integer $line 呼び出し元の行に関する情報
     */
    function debug($message, $caller = null, $line = null)
    {
        Moony_Logger::log(MOONY_LOG_LEVEL_DEBUG, $message, $caller, $line);
    }

    /**
     * テキストファイルにログを出力します。
     * 
     * @access public
     * @static
     * @param integer $log_level ログレベル
     * @param string $message メッセージ
     * @param string $caller 呼び出し元に関する情報（指定は任意）
     * @param integer $line 呼び出し元の行に関する情報（指定は任意）
     */
    function log($log_level, $message, $caller = null, $line = null)
    {
        static $level_flags = array(
            MOONY_LOG_LEVEL_FATAL => '[fatal]',
            MOONY_LOG_LEVEL_ERROR => '[error]',
            MOONY_LOG_LEVEL_WARN  => '[warn] ',
            MOONY_LOG_LEVEL_INFO  => '[info] ',
            MOONY_LOG_LEVEL_DEBUG => '[debug]'
        );

        if ($log_level == MOONY_LOG_LEVEL_NONE || $log_level < MOONY_LOG_CATEGORY) {
            return;
        }

        $log_file = Moony_Utils::buildPath(MOONY_LOG_DIR, MOONY_LOG_FILE);
        if (MOONY_LOG_ROTATE) {
            $log_file = Moony_Utils::buildPath('', $log_file, date('Ymd'));
        }

        $system_date = date(MOONY_LOG_DATE_FORMAT);

        if (strlen($caller) > 0) {
            if (strlen($line) > 0) {
                $message = sprintf(MOONY_LOG_FORMAT_LINES, $system_date, $level_flags[$log_level], $message, $caller, $line);
            } else {
                $message = sprintf(MOONY_LOG_FORMAT_CALLER, $system_date, $level_flags[$log_level], $message, $caller);
            }
        } else {
            $message = sprintf(MOONY_LOG_FORMAT_NORMAL, $system_date, $level_flags[$log_level], $message);
        }

        @error_log($message, 3, $log_file);
    }
}
?>
