<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * リクエスト情報を格納するクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Request
{
    /** @var array 実際のパラメータを格納する配列 */
    var $_params;

    /** @var array アップロードされるファイルに関する情報を格納する配列 */
    var $_files;

    /** @var object Moony_Sessionのインスタンス */
    var $session;

    /** @var array PATH_INFOの値の配列 */
    var $path_info;

    /**
     * コンストラクタです。
     * 
     * @access public
     */
    function Moony_Request()
    {
        $this->_params = ($_SERVER['REQUEST_METHOD'] == 'POST') ? $_POST : $_GET;
        $this->_params = $this->_filter($this->_params);
        $this->_files = $_FILES;

        // エンコーディング変換
        if (MOONY_REQUEST_PARAMETER_CONVERT_ENCODING &&
            MOONY_OUTPUT_ENCODING != MOONY_INTERNAL_ENCODING) {
            mb_convert_variables(MOONY_INTERNAL_ENCODING, MOONY_OUTPUT_ENCODING, &$this->_params);
            mb_convert_variables(MOONY_INTERNAL_ENCODING, MOONY_OUTPUT_ENCODING, &$this->_files);
        }
    }

    /**
     * パラメータ値を取得します。
     * 
     * @access public
     * @param string $name パラメータキー
     * @param mixed $alt パラメータが存在しない場合の代替値（（デフォルト<code>null</code>））
     * @return mixed パラメータ値
     */
    function get($name, $alt = null)
    {
        return Moony_Utils::getArrayValue($name, $this->_params, $alt);
    }

    /**
     * 全てのパラメータを連想配列として返します。
     *
     * @access public
     * @return array 全てのパラメータの連想配列
     */
    function getAll()
    {
        return $this->_params;
    }

    /**
     * アップロードされたファイルに関する情報を
     * 連想配列として返します。返される値は$_FILES[$name]と同等です。
     * ファイル名はエンコーディング変換されています。
     * 指定された名称のファイルがアップロード対象でない場合、無条件にNULLを返します。
     *
     * @access public
     * @param string $name 処理対象のファイルに付けられた名前
     * @return array アップロードされたファイルに関する情報
     */
    function getFile($name)
    {
        return Moony_Utils::getArrayValue($name, $this->_files);
    }

    /**
     * パラメータの値を指定された値に置き換えます。
     * 該当する名称のパラメータが存在しなかった場合、
     * 何も処理を行わずにfalseを返します。
     *
     * @access public
     * @param string $name パラメータ名称
     * @param mixed $value パラメータ値を置換する値
     * @return boolean 置換に成功したらtrue、失敗したらfalse
     */
    function replace($name, $value)
    {
        if ($this->exists($name)) {
            $this->_params[$name] = $value;
            return true;
        }
        return false;
    }

    /**
     * 指定された名称のリクエストパラメータが
     * 存在するかどうかを返します。
     *
     * @access public
     * @param string $name パラメータキー
     * @return boolean 存在する場合<code>true</code>
     */
    function exists($name)
    {
        return isset($this->_params[$name]);
    }

    /**
     * HTTPセッション情報を取得します。
     *
     * @access public
     * @return object HTTPセッション情報
     */
    function &getSession()
    {
        if (isset($this->session)) {
            return $this->session;
        }
        return null;
    }

    /**
     * 指定されたインデックスのPATH_INFOの値を返します。
     * 例えば、PATH_INFOが/foo/bar.htmlの場合、
     * getPathInfo(0)の戻り値は'foo'、
     * getPathInfo(1)の戻り値は'bar'になります。
     *
     * @access public
     * @param integer $index PATH_INFOの値の配列のインデックス
     * @param string $alt 該当のインデックスが存在しない場合の代替値
     */
    function getPathInfo($index, $alt = null)
    {
        return Moony_Utils::getArrayValue($index, $this->path_info, $alt);
    }

    /**
     * リクエストパラメータを処理します。
     * magic_quotes_gpcがONの場合、エスケープ文字を除去します。
     * また、NULLバイト文字列を除去します。
     *
     * @access private
     * @param mixed $data 処理対象のデータ
     * @return string 処理済の文字列
     */
    function _filter($data)
    {
        if (is_array($data)) {
            return array_map(array(&$this, '_filter'), $data);
        }
        if (get_magic_quotes_gpc()) {
            $data = stripslashes($data);
        }
        return $data;
    }
}
?>
