<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * レスポンス情報を格納するクラスです。
 * テンプレートに設定する値を格納します。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Response
{
    /** @var array テンプレートに設定する値 */
    var $_attributes = array();

    /** @var string テンプレートファイル名 */
    var $_template;

    /** @var string 出力バッファ */
    var $buffer;

    /**
     * コンストラクタです。
     *
     * @access public
     */
    function Moony_Response()
    {
        // 初期化
        $this->set(MOONY_TRANSACTION_TOKEN_NAME, '');
        $this->set(MOONY_VALIDATION_ERROR_KEY, array());
        $this->buffer = '';
    }

    /**
     * テンプレートに設定する値を設定します。
     * 割り当てる値がオブジェクトの場合、
     * エスケープ処理は行われません。
     *
     * @access public
     * @param string $key テンプレートの変数名
     * @param mixed $value 割り当てる値
     * @param boolean $escape 値をエスケープ処理するかどうか
     */
    function set($key, $value, $escape = true)
    {
        if ($escape) {
            if (!is_object($value)) {
                $value = Moony_Utils::escape($value);
            }
        }
        $this->_attributes[$key] = $value;
    }

    /**
     * テンプレートに設定する値を設定します。
     *
     * @access public
     * @param array $attributes テンプレートの変数名をキーに、割り当てる値を値に持つ連想配列
     * @param boolean $escape 値をエスケープ処理するかどうか
     */
    function setAll($attributes, $escape = true)
    {
        foreach ($attributes as $key => $value) {
            $this->set($key, $value, $escape);
        }
    }

    /**
     * テンプレートに設定する値を全て取得します。
     *
     * @access public
     * @return array テンプレートに設定する全ての値
     */
    function getAll()
    {
        return $this->_attributes;
    }

    /**
     * テンプレートファイル名を取得します。
     *
     * @access public
     * @return string テンプレートファイル名
     */
    function getTemplate()
    {
        return $this->_template;
    }

    /**
     * テンプレートファイル名を設定します。
     *
     * @access public
     * @param string $template テンプレートファイル名
     */
    function setTemplate($template)
    {
        $this->_template = $template;
    }

    /**
     * 指定されたステータスコードを送出します。
     *
     * @access public
     * @param integer $code ステータスコード
     */
    function sendStatus($code) {
        if (is_int($code)) {
            Moony_Utils::sendStatus($code);
        }
    }

    /**
     * 指定されたURLにリダイレクトします。
     *
     * @access public
     * @param string $url リダイレクト先のURL
     */
    function sendRedirect($url)
    {
        Moony_Utils::redirect($url);
    }

    /**
     * 指定されたファイルをHTTPレスポンスに添付します。
     * $nameが指定されない場合、元のファイルの名称がそのまま使用されます。
     *
     * @access public
     * @param string $path 添付するファイルのパス
     * @param string $name 添付するファイルにつける名前
     */
    function sendFile($path, $name = null)
    {
        Moony_Downloader::download($path, $name);
    }

    /**
     * 文字列を出力します。
     * $encodingが指定された場合、エンコーディング変換処理が行われます。
     * このメソッドは複数回実行することが可能です。
     *
     * @access public
     * @param string $content 出力する内容
     * @param string $encoding 出力する文字エンコーディング
     */
    function out($content, $encoding = null)
    {
        if (!is_null($encoding) && $encoding != MOONY_INTERNAL_ENCODING) {
            $content = mb_convert_encoding($content, $encoding, MOONY_INTERNAL_ENCODING);
        }
        $this->buffer .= $content;
    }
}
?>
