<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

define('MOONY_PREVIOUS_RESPONSE', '___moony_previous_response');
define('MOONY_ACCESS_TIME', '___moony_access_time');

/**
 * セッションを管理するクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Session
{
    /** @var boolean セッションが開始しているかどうか */
    var $_has_started = false;

    /**
     * コンストラクタです。
     * 
     * @access public
     */
    function Moony_Session()
    {
        session_name(MOONY_SESSION_NAME);
        session_cache_limiter('private, must-revalidate');
    }

    /**
     * セッションを開始します。
     * 
     * @access public
     */
    function start()
    {
        session_start();
        $this->_has_started = true;

        $now = time();
        if ($this->exists(MOONY_ACCESS_TIME)) {
            $access_time = $this->get(MOONY_ACCESS_TIME);
            if ($now - $access_time > MOONY_SESSION_REGENERATE_TIME) {
                // Config.phpで設定された間隔を過ぎた場合IDを再生成
                $this->regenerateId();
                $this->set(MOONY_ACCESS_TIME, $now);
            }
        } else {
            $this->set(MOONY_ACCESS_TIME, $now);
        }
    }

    /**
     * セッションIDを再生成します。
     *
     * @access public
     */
    function regenerateId()
    {
        // 値の退避
        $stored = $_SESSION;

        // 破棄
        $this->destroy();

        // 新セッションID生成
        session_id(md5(uniqid(rand(), true)));

        // 再開、セッション値復元
        session_start();
        $_SESSION = $stored;
    }

    /**
     * セッションを終了します。
     * 
     * @access public
     * @return boolean 正常に終了すれば<code>true</code>
     */
    function destroy()
    {
        // 初期化
        $_SESSION = array();

        // セッションクッキーの削除
        $name = session_name();
        if (isset($_COOKIE[$name])) {
            setcookie($name, '', 0, '/');
        }

        if (@session_destroy() === false) {
            // セッション破棄失敗
            return false;
        }
        $this->_has_started = false;

        return true;
    }

    /**
     * セッションが開始しているかどうかを返します。
     *
     * @access public
     * @return boolean セッションが開始しているかどうか
     */
    function hasStarted()
    {
        return $this->_has_started;
    }

    /**
     * セッションIDを返します。
     * セッションが開始されていない場合、
     * 無条件に<code>null</code>を返します。
     *
     * @access public
     * @return string セッションID
     */
    function getId()
    {
        if (!$this->_has_started) {
            return null;
        }
        return session_id();
    }

    /**
     * セッションに格納された値を取得します。
     * 指定されたキーに紐付く値が設定されていない場合、
     * 代替値を返します。
     * 
     * @access public
     * @param string $key 格納キー
     * @param mixed $alt 代替値
     * @return mixed 格納値
     */
    function get($key, $alt = null)
    {
        if (!$this->_has_started || !$this->exists($key)) {
            return $alt;
        }
        return $_SESSION[$key];
    }

    /**
     * セッションに値を設定します。
     * 
     * @access public
     * @param string $key 格納キー
     * @param mixed $value 格納値
     * @return boolean 正常に終了すれば<code>true</code>
     */
    function set($key, $value)
    {
        if (!$this->_has_started) {
            return false;
        }
        $_SESSION[$key] = $value;
        return true;
    }

    /**
     * 指定されたキーにセッション値が
     * 格納されているかどうかを返します。
     *
     * @access public
     * @param string $key セッションキー
     * @return boolean 存在する場合<code>true</code>
     */
    function exists($key)
    {
        return isset($_SESSION[$key]);
    }

    /**
     * セッションから値を削除します。
     * 
     * @access public
     * @param string $key 格納キー
     * @return boolean 正常に終了すれば<code>true</code>
     */
    function remove($key)
    {
        if (!$this->_has_started) {
            return false;
        }
        unset($_SESSION[$key]);
        return true;
    }

    /**
     * 画面表示情報を格納します。
     *
     * @access public
     * @param object $response Moony_Responseのインスタンス
     */
    function setResponse($response)
    {
        $this->set(MOONY_PREVIOUS_RESPONSE, $response);
    }

    /**
     * 前回画面表示情報を取得します。
     *
     * @access public
     * @return object Moony_Responseのインスタンス
     */
    function getResponse()
    {
        return $this->get(MOONY_PREVIOUS_RESPONSE);
    }
}
?>
