<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * Smartyのプラグインを実装するためのクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_SmartyPlugin
{
    /**
     * 定義されている各プラグインをSmartyに登録します。
     *
     * @access public
     * @static
     * @param object $smarty Smartyのインスタンス
     */
    function regist(&$smarty)
    {
        $smarty->register_function('moony_token', array('Moony_SmartyPlugin', 'smarty_function_moony_token'));
        $smarty->register_function('moony_error', array('Moony_SmartyPlugin', 'smarty_function_moony_error'));
        $smarty->register_function('moony_text', array('Moony_SmartyPlugin', 'smarty_function_moony_text'));
        $smarty->register_function('moony_password', array('Moony_SmartyPlugin', 'smarty_function_moony_password'));
        $smarty->register_function('moony_hidden', array('Moony_SmartyPlugin', 'smarty_function_moony_hidden'));
        $smarty->register_function('moony_file', array('Moony_SmartyPlugin', 'smarty_function_moony_file'));
        $smarty->register_function('moony_radio', array('Moony_SmartyPlugin', 'smarty_function_moony_radio'));
        $smarty->register_function('moony_checkbox', array('Moony_SmartyPlugin', 'smarty_function_moony_checkbox'));
        $smarty->register_function('moony_multibox', array('Moony_SmartyPlugin', 'smarty_function_moony_multibox'));
        $smarty->register_function('moony_select', array('Moony_SmartyPlugin', 'smarty_function_moony_select'));
        $smarty->register_function('moony_textarea', array('Moony_SmartyPlugin', 'smarty_function_moony_textarea'));
    }

    /**
     * トランザクショントークンをhidden項目として出力します。
     * {moony_token}とテンプレートファイル内に記述することで利用可能です。
     * XHTMLとして出力したい場合、{moony_token xhtml=true}と記述してください。
     * また、GETパラメータに追加したい場合、<a href="xxx?a=1&{moony_token method='get'}>と
     * 記述することでGETパラメータに追加することができます。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_token($params, &$smarty)
    {
        extract($params);

        $is_xhtml = false;
        if (isset($xhtml) && $xhtml === true) {
            $is_xhtml = true;
        }

        $is_post = true;
        if (isset($method) && strtolower($method) == 'get') {
            $is_post = false;
        }

        $name = MOONY_TRANSACTION_TOKEN_NAME;
        $value = Moony_SmartyPlugin::_getVars($name, $smarty);

        if ($is_post) {
            $content = "<input type=\"hidden\" name=\"${name}\" value=\"${value}\"";
            $content .= $is_xhtml ? ' />' : '>';
        } else {
            $content = "${name}=${value}";
        }

        echo $content;
    }

    /**
     * validationエラーがある場合に
     * 設定されたエラーメッセージを表示します。
     * {moony_error name='item_name'}とテンプレートファイル内に記述してください。
     * prefix、postfixを付加したい場合は以下のように記述してください。
     * {moony_error name='item_name' prefix='<span style="color:red;">' postfix='</span>'}
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_error($params, &$smarty)
    {
        extract($params);

        if (!isset($name)) {
            return;
        }
        $prefix = isset($prefix) ? $prefix : '';
        $postfix = isset($postfix) ? $postfix : '';

        $errors = Moony_SmartyPlugin::_getVars(MOONY_VALIDATION_ERROR_KEY, $smarty);

        if ($errors != null && is_array($errors)) {
            if (isset($errors[$name])) {
                echo $prefix, $errors[$name], $postfix;
            }
        }
    }

    /**
     * HTMLの<input type="text" ....>タグを出力します。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_text($params, &$smarty)
    {
        $is_xhtml = Moony_SmartyPlugin::_processInput($params, $smarty);

        // 初期値として空文字を設定
        if (!isset($params['value'])) {
            $params['value'] = '';
        }

        $content = '<input type="text"';
        foreach ($params as $key => $value) {
            $content .= " ${key}=\"${value}\"";
        }
        $content .= $is_xhtml ? ' />' : '>';

        echo $content;
    }

    /**
     * HTMLの<input type="password" ....>タグを出力します。
     * 常に値はクリアされるので注意してください。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_password($params, &$smarty)
    {
        $is_xhtml = Moony_SmartyPlugin::_processInput($params, $smarty);

        // パスワードなのでクリア
        $params['value'] = '';

        $content = '<input type="password"';
        foreach ($params as $key => $value) {
            $content .= " ${key}=\"${value}\"";
        }
        $content .= $is_xhtml ? ' />' : '>';

        echo $content;
    }

    /**
     * HTMLの<input type="hidden" ....>タグを出力します。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_hidden($params, &$smarty)
    {
        $is_xhtml = Moony_SmartyPlugin::_processInput($params, $smarty);

        // 初期値として空文字を設定
        if (!isset($params['value'])) {
            $params['value'] = '';
        }

        $content = '<input type="hidden"';
        foreach ($params as $key => $value) {
            $content .= " ${key}=\"${value}\"";
        }
        $content .= $is_xhtml ? ' />' : '>';

        echo $content;
    }

    /**
     * HTMLの<input type="file" ....>タグを出力します。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_file($params, &$smarty)
    {
        $is_xhtml = Moony_SmartyPlugin::_processInput($params, $smarty);
        $params['value'] = '';

        $content = '<input type="file"';
        foreach ($params as $key => $value) {
            $content .= " ${key}=\"${value}\"";
        }
        $content .= $is_xhtml ? ' />' : '>';

        echo $content;
    }

    /**
     * HTMLの<input type="radio" ....>タグを出力します。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_radio($params, &$smarty)
    {
        // 項目値と設定値を取得
        $value = $params['value'];
        $stored = Moony_SmartyPlugin::_getVars($params['name'], $smarty);

        $is_xhtml = Moony_SmartyPlugin::_processInput($params, $smarty);
        $params['value'] = $value;

        $content = '<input type="radio"';
        foreach ($params as $key => $value) {
            $content .= " ${key}=\"${value}\"";
        }
        if ($value == $stored) {
            $content .= $is_xhtml ? ' checked="checked"' : ' checked';
        }
        $content .= $is_xhtml ? ' />' : '>';

        echo $content;
    }

    /**
     * HTMLの<input type="checkbox" ....>タグを出力します。
     * 単体のcheckboxが必要な場合に使用してください。
     * 値が'true'、'on'、'1'の場合、選択されているものとして扱います。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_checkbox($params, &$smarty)
    {
        $stored = Moony_SmartyPlugin::_getVars($params['name'], $smarty);
        $checked = false;
        if ($stored == 'true' || $stored == 'on' || $stored == '1') {
            $checked = true;
        }

        $is_xhtml = Moony_SmartyPlugin::_processInput($params, $smarty);
        $params['value'] = 'true';

        $content = '<input type="checkbox"';
        foreach ($params as $key => $value) {
            $content .= " ${key}=\"${value}\"";
        }
        if ($checked) {
            $content .= $is_xhtml ? ' checked="checked"' : ' checked';
        }
        $content .= $is_xhtml ? ' />' : '>';

        echo $content;
    }

    /**
     * HTMLの<input type="checkbox" ....>タグを出力します。
     * 複数のcheckboxが必要な場合に使用してください。
     * {moony_multibox name="hoge" values="" ...}と指定された場合、
     * <input type="checkbox" name="hoge[]" value="" ...>のように展開します。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_multibox($params, &$smarty)
    {
        // 項目値と設定値を取得
        $value = $params['value'];
        $stored = Moony_SmartyPlugin::_getVars($params['name'], $smarty);

        $checked = false;
        if (in_array($value, Moony_Utils::toArray($stored))) {
            $checked = true;
        }

        $is_xhtml = Moony_SmartyPlugin::_processInput($params, $smarty);

        $content = '<input type="checkbox"';
        foreach ($params as $key => $value) {
            if ($key == 'name') {
                $value = "${value}[]";
            }
            $content .= " ${key}=\"${value}\"";
        }
        if ($checked) {
            $content .= $is_xhtml ? ' checked="checked"' : ' checked';
        }
        $content .= $is_xhtml ? ' />' : '>';

        echo $content;
    }

    /**
     * HTMLの<select ....>タグを出力します。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_select($params, &$smarty)
    {
        $selected = Moony_Utils::toArray(Moony_SmartyPlugin::_getVars($params['name'], $smarty));
        $is_xhtml = Moony_SmartyPlugin::_processInput($params, $smarty);
        $options = Moony_SmartyPlugin::_getVars($params['list'], $smarty);

        $is_multiple = false;
        if (isset($params['multiple'])) {
            $is_multiple = true;
        }

        if (isset($params['value'])) {
            unset($params['value']);
        }
        if (isset($params['list'])) {
            unset($params['list']);
        }

        $content = '<select';
        foreach ($params as $key => $value) {
            if ($key == 'multiple') {
                $content .= $is_xhtml ? ' multile="multiple"' : ' multiple';
            } else {
                if ($key == 'name' && $is_multiple) {
                    $value = "${value}[]";
                }
                $content .= " ${key}=\"${value}\"";
            }
        }
        $content .= '>';
        foreach ($options as $value => $label) {
            $flg = '';
            if (in_array($value, $selected)) {
                $flg = $is_xhtml ? ' selected="selected"' : ' selected';
            }
            $content .= "<option value=\"${value}\"${flg}>${label}</option>";
        }
        $content .= '</select>';

        echo $content;
    }

    /**
     * HTMLの<textarea ....>タグを出力します。
     *
     * @access public
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     */
    function smarty_function_moony_textarea($params, &$smarty)
    {
        Moony_SmartyPlugin::_processInput($params, $smarty);

        $text = '';
        if (isset($params['value'])) {
            $text = $params['value'];
            unset($params['value']);
        }

        $content = '<textarea ';
        foreach ($params as $key => $value) {
            $content .= " ${key}=\"${value}\"";
        }
        $content .= '>';
        $content .= $text;
        $content .= '</textarea>';

        echo $content;
    }

    /**
     * input関連の関数で使用される共通処理です。
     *
     * @access private
     * @static
     * @param array $params テンプレートから渡されるパラメータ
     * @param object $smarty Smartyのインスタンス
     * @return boolean XHTMLとして出力するかどうか
     */
    function _processInput(&$params, $smarty)
    {
        $value = Moony_SmartyPlugin::_getVars($params['name'], $smarty);
        $errors = Moony_SmartyPlugin::_getVars(MOONY_VALIDATION_ERROR_KEY, $smarty);

        $is_error = isset($errors[$params['name']]);

        if (!is_null($value)) {
            if (is_array($value)) {
            } else {
                $params['value'] = $value;
            }
        }

        if (isset($params['errorClass'])) {
            if ($is_error) {
                $params['class'] = $params['errorClass'];
            }
            unset($params['errorClass']);
        }

        if (isset($params['errorStyle'])) {
            if ($is_error) {
                $params['style'] = $params['errorStyle'];
            }
            unset($params['errorStyle']);
        }

        $is_xhtml = false;
        if (isset($params['xhtml'])) {
            if ($params['xhtml'] === true) {
                $is_xhtml = true;
            }
            unset($params['xhtml']);
        }

        return $is_xhtml;
    }

    /**
     * input関連の関数で使用される共通処理です。
     *
     * @access private
     * @static
     * @param string $name テンプレートに割り当てられた変数名
     * @param object $smarty Smartyのインスタンス
     * @return mixed テンプレートに割り当てられた変数の値
     */
    function _getVars($name, &$smarty) {
        if (isset($smarty->_tpl_vars[$name])) {
            return $smarty->_tpl_vars[$name];
        }
        return null;
    }
}
?>
