<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * ファイルのアップロードを行うためのクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Uploader
{
    /** @var array アップロードされたファイルに関する情報 */
    var $_file;

    /** @var string 処理対象のファイルに付けられた名前 */
    var $_name;

    /** @var string 出力ファイル名に使用するエンコーディング */
    var $_encoding;

    /**
     * コンストラクタです。
     * ファイル名に日本語が含まれる可能性がある場合、
     * 環境に適したエンコーディングを必ず指定してください。
     * 明示的に指定されない場合、
     * デフォルトでPHPの内部エンコーディングが使用されます。
     *
     * @access public
     * @param object $request Moony_Requestのインスタンス
     * @param string $name 処理対象のファイルに付けられた名前
     * @param string $encoding 出力ファイル名に使用するエンコーディング
     */
    function Moony_Uploader(&$request, $name, $encoding = MOONY_INTERNAL_ENCODING)
    {
        $this->_file = $request->getFile($name);
        $this->_name = $name;
        $this->_encoding = $encoding;
    }

    /**
     * クライアント側の元のファイル名を返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return string|array クライアント側の元のファイル名
     */
    function getOriginalName()
    {
        return Moony_Utils::getArrayValue('name', $this->_file);
    }

    /**
     * ファイルのMIMEタイプを返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return string|array ファイルのMIME型
     */
    function getMimeType()
    {
        return Moony_Utils::getArrayValue('type', $this->_file);
    }

    /**
     * ファイルサイズを返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return integer|array ファイルサイズ
     */
    function getSize()
    {
        return Moony_Utils::getArrayValue('size', $this->_file);
    }

    /**
     * サーバ側に保存されているテンポラリファイル名を返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return integer|array サーバ側に保存されているテンポラリファイル名
     */
    function getTmpName()
    {
        return Moony_Utils::getArrayValue('tmp_name', $this->_file);
    }

    /**
     * アップロードのエラーコードを返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return integer|array アップロードのエラーコード
     */
    function getErrorCode()
    {
        return Moony_Utils::getArrayValue('error', $this->_file);
    }

    /**
     * ファイルのアップロード処理を行います。
     * 何かエラーが発生している場合、処理を一切行わずにfalseを返します。
     *
     * @access public
     * @param string $save_dir 保存先のディレクトリ名
     * @param integer $file_mode 保存したファイルのパーミッション（8進数で指定）
     * @param boolean $on_error_remove 処理途中でエラーが発生した場合、アップロード済みのファイルを削除するかどうか
     * @return boolean 全てのアップロードが成功した場合true、そうでない場合false
     */
    function upload($save_dir, $file_mode = 0644, $on_error_remove = false)
    {
        if (!file_exists($save_dir)) {
            Moony_Logger::warn('Directory not found: ' . $save_dir,  __FILE__, __LINE__);
            return false;
        }
        if ($this->_file == null) {
            Moony_Logger::warn('File name not found: ' . $this->_name,  __FILE__, __LINE__);
            return false;
        }
        if ($this->hasError()) {
            Moony_Logger::warn('Upload prepare failed: ' . $this->_name, __FILE__, __LINE__);
            return false;
        }
        if ($this->_encoding != MOONY_INTERNAL_ENCODING) {
            if (is_array($this->_file['name'])) { 
                mb_convert_variables($this->_encoding, MOONY_INTERNAL_ENCODING, &$this->_file['name']);
            } else {
                $this->_file[$name] = mb_convert_encoding($this->_file[$name], $this->_encoding, MOONY_INTERNAL_ENCODING);
            }
        }
        if (is_array($this->_file['name'])) {
            $paths = array();
            foreach ($this->_file['name'] as $key => $name) {
                if ($this->_file['error'][$key] == UPLOAD_ERR_OK) {
                    $org_name = basename($name);
                    $tmp_name = $this->_file['tmp_name'][$key];
                    $save_path = Moony_Utils::buildPath($save_dir, $org_name);
                    if (move_uploaded_file($tmp_name, $save_path)) {
                        chmod($save_path, $file_mode);
                        $paths[] = $save_path;
                    } else {
                        Moony_Logger::warn('File upload failed: ' . $this->_name,  __FILE__, __LINE__);
                        if ($on_error_remove) {
                            foreach ($paths as $path) {
                                @unlink($path);
                            }
                        }
                        return false;
                    }
                }
            }
        } else {
            if ($this->_file['error'] == UPLOAD_ERR_OK) {
                $org_name = basename($this->_file['name']);
                $tmp_name = $this->_file['tmp_name'];
                $save_path = Moony_Utils::buildPath($save_dir, $org_name);
                if (move_uploaded_file($tmp_name, $save_path)) {
                    chmod($save_path, $file_mode);
                } else {
                    Moony_Logger::warn('File upload failed: ' . $this->_name,  __FILE__, __LINE__);
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * ファイルが画像であることが期待される場合の
     * MIMEタイプをチェックします。
     * テンポラリディレクトリに保存されたファイルに対して
     * getimagesize関数を用いて正確なMIMEタイプを取得、
     * $_FILESに格納されているMIMEタイプと合致するかどうか確認します。
     *
     * @access public
     * @return boolean 画像として正常なMIMEタイプであればtrue
     */
    function checkImageMimeType()
    {
        if (is_array($this->_file['name'])) {
            foreach ($this->_file['type'] as $key => $mime_type) {
                if (strlen($this->_file['tmp_name'][$key]) == 0) {
                    continue;
                }
                $tmp_name = $this->_file['tmp_name'][$key];
                $attributes = getimagesize($tmp_name);
                if (!$attributes) {
                    // 画像でない
                    return false;
                }
                if ($attributes['mime'] != $mime_type) {
                    // MIMEタイプが一致しない
                    return false;
                }
            }
        } else {
            if (strlen($this->_file['tmp_name']) == 0) {
                return true;
            }
            $tmp_name = $this->_file['tmp_name'];
            $attributes = getimagesize($tmp_name);
            if (!$attributes) {
                // 画像でない
                return false;
            }
            if ($attributes['mime'] != $this->_file['type']) {
                // MIMEタイプが一致しない
                return false;
            }
        }
        return true;
    }

    /**
     * 対象のファイルが間違いなく
     * アップロード準備されているかどうか調べます。
     *
     * @access public
     * @return boolean 何かエラーがある場合true
     */
    function hasError()
    {
        $error_code = $this->getErrorCode();
        if (is_array($error_code)) {
            foreach ($error_code as $code) {
                if ($code != UPLOAD_ERR_OK && $code != UPLOAD_ERR_NO_FILE) {
                    return true;
                }
            }
        } else {
            if ($error_code != UPLOAD_ERR_OK && $error_code != UPLOAD_ERR_NO_FILE) {
                return true;
            }
        }
        return false;
    }
}
?>
