<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * Validationの基本機能を提供するクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Validator
{
    /**
     * 指定されたコールバック関数を用いて項目のチェックを行います。
     * コールバック関数は入力項目値（単項目）をただ一つの引数として取り、
     * 検証結果をbooleanで返す関数である必要があります。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @param callback $callback 検証用のコールバック関数
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validate(&$request, &$errors, $name, $message, $callback)
    {
        if (($value = $request->get($name, false)) === false) {
            return true;
        }
        $value = Moony_Utils::toArray($value);
        foreach ($value as $each) {
            if (strlen($each) > 0 && !call_user_func($callback, $each)) {
                $errors->set($name, $message);
                return false;
            }
        }
        return true;
    }

    /**
     * 必須項目のチェックを行います。
     * 配列項目の場合、全ての要素が入力されている必要があります。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateRequired(&$request, &$errors, $name, $message)
    {
        if (($value = $request->get($name, false)) === false || count($value) == 0) {
            $errors->set($name, $message);
            return false;
        }
        $value = Moony_Utils::toArray($value);
        foreach ($value as $each) {
            if (strlen($each) == 0) {
                $errors->set($name, $message);
                return false;
            }
        }
        return true;
    }

    /**
     * 正規表現を用いた入力チェックを行います。
     * 配列項目の場合、全ての要素がチェック対象になります。
     * 正規表現としてpreg_matchで使用可能なパターンを指定してください。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @param string $pattern 正規表現パターン（preg_matchで使用可能なパターン）
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateMask(&$request, &$errors, $name, $message, $pattern)
    {
        if (($value = $request->get($name, false)) === false) {
            return true;
        }
        $value = Moony_Utils::toArray($value);
        foreach ($value as $each) {
            if (strlen($each) > 0 && !preg_match($pattern, $each)) {
                $errors->set($name, $message);
                return false;
            }
        }
        return true;
    }

    /**
     * 数値項目のチェックを行います。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateNumeric(&$request, &$errors, $name, $message)
    {
        return Moony_Validator::validate($request, $errors, $name, $message, 'is_numeric');
    }

    /**
     * 英字項目のチェックを行います。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateAlpha(&$request, &$errors, $name, $message)
    {
        return Moony_Validator::validate($request, $errors, $name, $message, 'ctype_alpha');
    }

    /**
     * 英数字項目のチェックを行います。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateAlnum(&$request, &$errors, $name, $message)
    {
        return Moony_Validator::validate($request, $errors, $name, $message, 'ctype_alnum');
    }

    /**
     * カタカナ項目のチェックを行います。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateKatakana(&$request, &$errors, $name, $message)
    {
        return Moony_Validator::validate($request, $errors, $name, $message, array('Moony_Utils', 'isKatakana'));
    }

    /**
     * Eメール項目のチェックを行います。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateEMail(&$request, &$errors, $name, $message)
    {
        return Moony_Validator::validateMask(&$request, &$errors, $name, $message, '/^[a-z0-9\-\._]+@[a-z0-9]([0-9a-z\-]*[a-z0-9]\.){1,}[a-z]{1,4}$/i');
    }

    /**
     * URL項目のチェックを行います。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateUrl(&$request, &$errors, $name, $message)
    {
        return Moony_Validator::validateMask(&$request, &$errors, $name, $message, '/^https?:\/\/[-_.!~*\'()a-z0-9;\/?:\@&=+\$,%#]+$/i');
    }

    /**
     * 日付項目のチェックを行います。
     * 引数$formatには、sscanf関数で日付文字列を「年、月、日」の順番で解析できるようなフォーマットを指定してください。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateDate(&$request, &$errors, $name, $message, $format = '%d-%d-%d')
    {
        if (($value = $request->get($name, false)) === false) {
            return true;
        }
        $value = Moony_Utils::toArray($value);
        foreach ($value as $each) {
            list($year, $month, $day) = sscanf($each, $format);
            if (strlen($each) > 0 && !checkdate($month, $day, $year)) {
                $errors->set($name, $message);
                return false;
            }
        }
        return true;
    }

    /**
     * 文字のバイト長のチェックを行います。
     * 内部エンコーディングによってバイト数は変化するので注意してください。
     * デフォルトもエンコーディングはUTF-8なので、多くの日本語の1文字は3バイトに相当します。
     * 強制的に日本語の文字を2バイトとして扱いたい場合は、
     * 引数の$treat_japanese_as_double_byteにtrueを指定してください。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @param mixed $min 許容されうる最小バイト長
     * @param mixed $max 許容されうる最大バイト長
     * @param boolean $treat_japanese_as_double_byte 日本語の全角文字を2バイトとしてあつかうかどうか
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateLength(&$request, &$errors, $name, $message, $min, $max, $treat_japanese_as_double_byte = false)
    {
        if (($value = $request->get($name, false)) === false) {
            return true;
        }
        $value = Moony_Utils::toArray($value);
        foreach ($value as $each) {
            $len = $treat_japanese_as_double_byte ? Moony_Utils::getStrWidth($each) : strlen($each);
            if (strlen($each) > 0 && ($len < $min || $len > $max)) {
                $errors->set($name, $message);
                return false;
            }
        }
        return true;
    }

    /**
     * 文字数のチェックを行います。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @param mixed $min 許容されうる最小文字数
     * @param mixed $max 許容されうる最大文字数
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateLetterCount(&$request, &$errors, $name, $message, $min, $max)
    {
        if (($value = $request->get($name, false)) === false) {
            return true;
        }
        $value = Moony_Utils::toArray($value);
        foreach ($value as $each) {
            $count = mb_strlen($each);
            if (strlen($each) > 0 && ($count < $min || $count > $max)) {
                $errors->set($name, $message);
                return false;
            }
        }
        return true;
    }

    /**
     * 範囲チェックを行います。
     * 配列項目の場合、全ての要素をチェックします。
     * エラーが存在する場合、指定されたメッセージを追加します。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param object $errors Moony_Messagesのインスタンス（エラーメッセージ設定用）
     * @param string $name 項目名称
     * @param string $message エラーが存在する場合に設定するエラーメッセージ
     * @param mixed $min 許容されうる最小値
     * @param mixed $max 許容されうる最大値
     * @return boolean エラーが存在する場合false、存在しない場合true
     */
    function validateRange(&$request, &$errors, $name, $message, $min, $max)
    {
        if (($value = $request->get($name, false)) === false) {
            return true;
        }
        $value = Moony_Utils::toArray($value);
        foreach ($value as $each) {
            if (strlen($each) > 0 && ($each < $min || $each > $max)) {
                $errors->set($name, $message);
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された項目をcheckboxとして扱い、
     * 値が送信されてこなかった場合に代替値として長さ0の文字列を設定します。
     * validationエラーで入力画面に戻った場合、
     * チェックがはずれた状態になります。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param string $name checkbox項目の名称
     */
    function treatAsCheckbox(&$request, $name)
    {
        if (!$request->exists($name)) {
            $request->params[$name] = '';
        }
    }

    /**
     * 指定された項目を複数項目からなるcheckboxとして扱い、
     * 値が送信されてこなかった場合に代替値として要素のない配列を設定します。
     * validationエラーで入力画面に戻った場合、
     * 全てのチェックがはずれた状態になります。
     *
     * @access public
     * @static
     * @param object $request Moony_Requestのインスタンス
     * @param string $name checkbox項目の名称
     */
    function treatAsMultibox(&$request, $name)
    {
        if (!$request->exists($name)) {
            $request->params[$name] = array();
        }
    }
}
?>
