<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * エラーを処理するクラスです。
 * スタティックなメソッドで構成されます。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Error
{
    /**
     * 発生したエラーのハンドリングを行います。
     * 
     * @access public
     * @static
     * @param integer $code エラーコード
     * @param string $message メッセージ
     * @param string $file 呼び出し元のファイル名
     * @param integer $line 呼び出し元の行番号
     */
    function handle($code, $message, $file, $line)
    {
        switch ($code) {

            case E_STRICT:
                // デバッグログとして記録
                Moony_Logger::debug($message, $file, $line);
                break;

            case E_NOTICE:
            case E_USER_NOTICE:
            case E_USER_WARNING:
            case E_CORE_WARNING:
            case E_COMPILE_WARNING:
                // 警告ログとして記録
                Moony_Logger::warn($message, $file, $line);
                break;

            case E_ERROR:
            case E_PARSE:
            case E_USER_ERROR:
            case E_CORE_ERROR:
            case E_COMPILE_ERROR:
                // エラーログとして記録、Internal Server Errorとして処理終了
                Moony_Logger::error($message, $file, $line);
                Moony_Error::internalServerError();
                break;
        }
    }

    /**
     * エラーを発生させます。
     * エラーコードにはE_USER_ERRORを使用します。
     *
     * @access public
     * @static
     * @param string $message メッセージ
     * @param string $file 呼び出し元のファイル名
     * @param integer $line 呼び出し元の行番号
     */
    function raise($message, $file, $line)
    {
        if (MOONY_USE_ERROR_HANDLER) {
            Moony_Error::handle(E_USER_ERROR, $message, $file, $line);
        } else {
            trigger_error($message, E_USER_ERROR);
        }
    }

    /**
     * テンプレートファイルが見つからない場合のエラー処理です。
     * httpのステータスで404を返し、処理を終了します。
     *
     * @access public
     * @static
     */
    function notFound()
    {
        header('HTTP/1.0 404 Not Found');
        echo 'Not Foud';
        exit;
    }

    /**
     * 動作エラーが発生した場合のエラー処理です。
     * httpのステータスで500を返し、処理を終了します。
     *
     * @access public
     * @static
     */
    function internalServerError()
    {
        header('HTTP/1.0 500 Internal Server Error');
        echo 'Intenal Server Error';
        exit;
    }
}
?>
