<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * Moonyのバージョン
 */
define('MOONY_VERSION', '0.10.2');

/**
 * 内部エンコーディング
 */
define('MOONY_INTERNAL_ENCODING', 'UTF-8');

/**
 * 出力エンコーディング
 */
define('MOONY_OUTPUT_ENCODING', 'SJIS');

/**
 * テンプレートファイルのエンコーディング
 */
define('MOONY_TEMPLATE_ENCODING', 'UTF-8');

/**
 * リクエストパラメータのエンコーディング変換を行うかどうか
 */
define('MOONY_REQUEST_PARAMETER_CONVERT_ENCODING', ini_get('mbstring.encoding_translation') ? false : true);

/**
 * テンプレート表示時にHTMLのエンコーディングに関するヘッダ送出を行うかどうか
 */
define('MOONY_SENDING_HEADER_ABOUT_ENCODING', true);

/**
 * XHTMLとしてのヘッダ（application/xhtml+xml）を使用するかどうか
 */
define('MOONY_USE_XHTML_HEADER', false);

/**
 * PATH_INFOの何階層目までをアクション設定として解釈するか
 */
define('MOONY_EVAL_AS_ACTION_SETTING', 2);

/**
 * セッション名称
 */
define('MOONY_SESSION_NAME', 'MOONY_SESSID');

/**
 * セッションを自動的に開始するかどうか
 */
define('MOONY_SESSION_AUTO_START', true);

/**
 * セッションIDの再生成を行う間隔（秒）
 * （0に設定すると常に再生成を行います）
 */
define('MOONY_SESSION_REGENERATE_TIME', 3600);

/**
 * トランザクショントークンの項目名
 */
define('MOONY_TRANSACTION_TOKEN_NAME', '_moony_token');

/**
 * エラーメッセージの設定キー
 */
define('MOONY_VALIDATION_ERROR_NAME', '_moony_validation_error');

/**
 * 前回画面表示時に使用したMoony_Responseを格納しておくためのキー
 */
define('MOONY_PREVIOUS_RESPONSE_NAME', '_moony_previous_response');

/**
 * エラーハンドラを使用するかどうか
 */
define('MOONY_USE_ERROR_HANDLER', true);

/**
 * Moonyのログファイル名称
 */
define('MOONY_LOG_FILE', 'moony.log');

/**
 * ログファイルに出力する日付の形式（date関数を利用）
 */
define('MOONY_LOG_DATE_FORMAT', 'Y/m/d H:i:s');

/**
 * ログ出力レベル: 致命的エラー
 */
define('MOONY_LOG_LEVEL_FATAL', 5);

/**
 * ログ出力レベル: エラー
 */
define('MOONY_LOG_LEVEL_ERROR', 4);

/**
 * ログ出力レベル: 警告
 */
define('MOONY_LOG_LEVEL_WARN',  3);

/**
 * ログ出力レベル: 通常
 */
define('MOONY_LOG_LEVEL_INFO',  2);

/**
 * ログ出力レベル: デバッグ・トレース
 */
define('MOONY_LOG_LEVEL_DEBUG', 1);

/**
 * ログ出力レベル: 出力しない
 */
define('MOONY_LOG_LEVEL_NONE',  0);

/**
 * ログ出力を行うログレベルの閾値
 */
define('MOONY_LOG_CATEGORY', MOONY_LOG_LEVEL_INFO);

/**
 * 日単位でログのローテートを行うかどうか
 */
define('MOONY_LOG_ROTATE', true);

/**
 * Moonyを構成するファイルのinclude
 */
require_once 'Moony/Checker.php';
require_once 'Moony/Controller.php';
require_once 'Moony/DB.php';
require_once 'Moony/Error.php';
require_once 'Moony/Filter.php';
require_once 'Moony/Flash.php';
require_once 'Moony/Injector.php';
require_once 'Moony/Loader.php';
require_once 'Moony/Logger.php';
require_once 'Moony/Plugins.php';
require_once 'Moony/Request.php';
require_once 'Moony/Response.php';
require_once 'Moony/Router.php';
require_once 'Moony/Session.php';
require_once 'Moony/Uploader.php';
require_once 'Moony/Utils.php';
require_once 'Moony/Validator.php';
require_once 'Moony/View.php';


/**
 * 動作設定を行い、コントローラの起動を行うクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony
{
    /** @var string アクションクラスを格納するディレクトリ */
    var $action_dir = 'action';

    /** @var string テンプレートファイルを格納するディレクトリ */
    var $template_dir = 'template';

    /** @var string テンプレートファイルのコンパイル結果などを格納する作業用ディレクトリ */
    var $work_dir = 'work';

    /** @var string ログファイルの出力先ディレクトリ */
    var $log_dir = 'log';

    /** @var string テンプレートファイルの拡張子 */
    var $template_extension = 'tpl';

    /** @var string テンプレートファイルの左側区切り文字 */
    var $left_delimiter = '{';

    /** @var string テンプレートファイルの右側区切り文字 */
    var $right_delimiter = '}';

    /** @var array 設定ファイル名の配列 */
    var $config_files;

    /**
     * コンストラクタです。
     * 必要な初期設定を行います。
     * 
     * @access public
     */
    function Moony()
    {
        // mbstring設定
        mb_language('ja');
        mb_internal_encoding(MOONY_INTERNAL_ENCODING);
        mb_http_output(MOONY_OUTPUT_ENCODING);

        // エラーハンドラ設定
        if (MOONY_USE_ERROR_HANDLER) {
            set_error_handler(array('Moony_Error', 'handle'));
        }

        // 初期化
        $this->config_files = array();
    }

    /**
     * コントローラを起動します。
     * 
     * @access public
     */
    function dispatch()
    {
        // フレームワーク情報をHTTPヘッダ出力
        header('X-Framework: Moony/' . MOONY_VERSION);

        // 必須オプションチェック
        if (!isset($this->action_dir) ||
            !isset($this->template_dir) ||
            !isset($this->work_dir) ||
            !isset($this->log_dir)) {
            Moony_Error::raise('Configuration failure');
        }

        // workディレクトリ、logディレクトリチェック
        if (!Moony_Utils::isWritableDir($this->work_dir)) {
            Moony_Error::raise('Unable to write: ' . $this->work_dir);
        }
        if (!Moony_Utils::isWritableDir($this->log_dir)) {
            Moony_Error::raise('Unable to write: ' . $this->log_dir);
        }

        // 定義
        define('MOONY_ACTION_DIR', $this->action_dir);
        define('MOONY_TEMPLATE_DIR', $this->template_dir);
        define('MOONY_WORK_DIR', $this->work_dir);
        define('MOONY_LOG_DIR', $this->log_dir);
        define('MOONY_TEMPLATE_EXTENSION', $this->template_extension);
        define('MOONY_TEMPLATE_LEFT_DELIMITER', $this->left_delimiter);
        define('MOONY_TEMPLATE_RIGHT_DELIMITER', $this->right_delimiter);

        // コントローラ生成、処理開始
        $controller =& new Moony_Controller();
        $controller->process($this->config_files);
    }

    /**
     * アクションクラスを格納するディレクトリを
     * 設定します。初期値として「action」が設定されています。
     *
     * @access public
     * @param string $action_dir アクションクラスを格納するディレクトリ
     */
    function setActionDir($action_dir)
    {
        $this->action_dir = $action_dir;
    }

    /**
     * テンプレートファイルを格納するディレクトリを
     * 設定します。初期値として「template」が設定されています。
     *
     * @access public
     * @param string $template_dir テンプレートファイルを格納するディレクトリ
     */
    function setTemplateDir($template_dir)
    {
        $this->template_dir = $template_dir;
    }

    /**
     * テンプレートファイルのコンパイル結果などを格納する作業用ディレクトリを
     * 設定します。初期値として「work」が設定されています。
     *
     * @access public
     * @param string $work_dir テンプレートファイルのコンパイル結果などを格納する作業用ディレクトリ
     */
    function setWorkDir($work_dir)
    {
        $this->work_dir = $work_dir;
    }

    /**
     * ログファイルの出力先ディレクトリを
     * 設定します。初期値として「log」が設定されています。
     *
     * @access public
     * @param string $log_dir ログファイルの出力先ディレクトリ
     */
    function setLogDir($log_dir)
    {
        $this->log_dir = $log_dir;
    }

    /**
     * テンプレートファイルの拡張子を設定します。
     * 初期値として「tpl」が設定されています。
     *
     * @access public
     * @param string $template_extension テンプレートファイルの拡張子
     */
    function setTemplateExtension($template_extension)
    {
        $this->template_extension = $template_extension;
    }

    /**
     * テンプレートファイルの左側の区切り文字を設定します。
     * 初期値として「{」が設定されています。
     *
     * @access public
     * @param string $left_delimiter テンプレートファイルの左側の区切り文字
     */
    function setLeftDelimiter($left_delimiter)
    {
        $this->left_delimiter = $left_delimiter;
    }

    /**
     * テンプレートファイルの右側の区切り文字を設定します。
     * 初期値として「}」が設定されています。
     *
     * @access public
     * @param string $right_delimiter テンプレートファイルの右側の区切り文字
     */
    function setRightDelimiter($right_delimiter)
    {
        $this->right_delimiter = $right_delimiter;
    }

    /**
     * 設定が記述されているiniファイルを読み込みます。
     *
     * @access public
     * @param string $file iniァイルのパス
     * @return boolean ファイルが存在しない場合false
     */
    function loadConfig($file)
    {
        if (file_exists($file)) {
            $this->config_files[] = $file;
            return true;
        }
        return false;
    }

    /**
     * include_pathを追加します。
     *
     * @access public
     * @static
     * @param string $path include_pathに追加するパス
     * @return boolean 追加に成功すればtrue
     */
    function addIncludePath($path)
    {
        if (file_exists($path)) {
            $include_path = get_include_path() . PATH_SEPARATOR . $path;
            if (set_include_path($include_path) !== false) {
                return true;
            }
        }
        return false;
    }
}


/**
 * パスの区切り文字
 */
if (!defined('PATH_SEPARATOR')) {
    if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
        define('PATH_SEPARATOR', ';');
    } else {
        define('PATH_SEPARATOR', ':');
    }
}

/**
 * ディレクトリの区切り文字
 */
if (!defined('DIRECTORY_SEPARATOR')) {
    if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
        define('DIRECTORY_SEPARATOR', '\\');
    } else {
        define('DIRECTORY_SEPARATOR', '/');
    }
}

/**
 * エラーレベル: E_STRICT
 */
if (!defined('E_STRICT')) {
    define('E_STRICT', 2048);
}
?>
