<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * 値のチェックを行うためのユーティリティクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Checker
{
    /**
     * 何らかの値が設定されているかどうかを返します。
     *
     * @access public
     * @static
     * @param string|array $value 検証対象の値
     * @return boolean 設定されていればtrue
     */
    function isPresent($value)
    {
        $values = Moony_Utils::toArray($value);
        foreach ($values as $each) {
            if (strlen($each) == 0) {
                return false;
            }
        }
        return true;
    }

    /**
     * 正規表現パターンにマッチするかどうかを返します。
     * 内部でpreg_match関数を用いてマッチングを行います。
     *
     * @access public
     * @static
     * @param string|array $value 検証対象の値
     * @param string $pattern preg_matchで使用可能な正規表現パターン
     * @return boolean マッチすればtrue
     */
    function isRegex($value, $pattern)
    {
        $values = Moony_Utils::toArray($value);
        foreach ($values as $each) {
            if (strlen($each) > 0 && !preg_match($pattern, $each)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 正規表現パターンにマッチするかどうかを返します（マルチバイト対応）。
     * 内部でmb_ereg関数を用いてマッチングを行います。
     *
     * @access public
     * @static
     * @param string|array $value 検証対象の値
     * @param string $pattern mb_eregで使用可能な正規表現パターン
     * @return boolean マッチすればtrue
     */
    function isRegexMb($value, $pattern)
    {
        mb_regex_encoding(MOONY_INTERNAL_ENCODING);
        $values = Moony_Utils::toArray($value);
        foreach ($values as $each) {
            if (strlen($each) > 0 && !mb_ereg($pattern, $each)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 文字列長が指定された範囲におさまるかどうかを返します。
     *
     * @access public
     * @static
     * @param string|array $value 検証対象の値
     * @param integer $min 最小長
     * @param integer $max 最大長
     * @param boolean $inclusive 最小長/最大長と同値を許容するかどうか
     * @return boolean おさまればtrue
     */
    function isLengthBetween($value, $min, $max, $inclusive = true)
    {
        $values = Moony_Utils::toArray($value);
        foreach ($values as $each) {
            $len = mb_strlen($each);
            if ($len > 0) {
                if ($inclusive) {
                    if ($len < $min || $len > $max) {
                        return false;
                    }
                } else {
                    if ($len <= $min || $len >= $max) {
                        return false;
                    }
                }
            }
        }
        return true;
    }

    /**
     * 指定された範囲におさまるかどうかを返します。
     *
     * @access public
     * @static
     * @param string|array $value 検証対象の値
     * @param mixed $start 最小値
     * @param mixed $end 最大値
     * @param boolean $inclusive 最小値/最大値と同値を許容するかどうか
     * @return boolean おさまればtrue
     */
    function isBetween($value, $start, $end, $inclusive = true)
    {
        $values = Moony_Utils::toArray($value);
        foreach ($values as $each) {
            if (strlen($each) > 0) {
                if ($inclusive) {
                    if ($each < $start || $each > $end) {
                        return false;
                    }
                } else {
                    if ($each <= $start || $each >= $end) {
                        return false;
                    }
                }
            }
        }
        return true;
    }

    /**
     * 日付として正しいかどうかを返します。
     *
     * @access public
     * @static
     * @param string|array $value 検証対象の値
     * @param string $format sscanf関数で日付文字列を「年、月、日」の順番で解析できるようなフォーマット
     * @return boolean 正しければtrue
     */
    function isDate($value, $format)
    {
        $values = Moony_Utils::toArray($value);
        foreach ($values as $each) {
            list($year, $month, $day) = sscanf($each, $format);
            if (strlen($each) > 0 && !checkdate($month, $day, $year)) {
                return false;
            }
        }
        return true;
    }

    /**
     * カタカナかどうかを返します。
     *
     * @access public
     * @static
     * @param string|array $value 検証対象の値
     * @return boolean カタカナで構成されていればtrue
     */
    function isKatakana($value)
    {
        $values = Moony_Utils::toArray($value);
        if (MOONY_INTERNAL_ENCODING != 'EUC-JP') {
            mb_convert_variables('EUC-JP', MOONY_INTERNAL_ENCODING, &$value);
        }
        return Moony_Checker::isRegex($value, '/^(\xA5[\xA1-\xF6]|[\xA1\xBC])+$/');
    }

    /**
     * ひらがなかどうかを返します。
     *
     * @access public
     * @static
     * @param string|array $value 検証対象の値
     * @return boolean ひらがなで構成されていればtrue
     */
    function isHiragana($value)
    {
        $values = Moony_Utils::toArray($value);
        if (MOONY_INTERNAL_ENCODING != 'EUC-JP') {
            mb_convert_variables('EUC-JP', MOONY_INTERNAL_ENCODING, &$value);
        }
        return Moony_Checker::isRegex($value, '/^(\xA4[\xA1-\xF3]|[\xA1\xBC])+$/');
    }

    /**
     * 指定されたコールバック関数を使用して値の検証を行います。
     * コールバック関数はパラメータとして個々の入力値、
     * およびその後ろに$paramsで指定された配列値をとり、戻り値としてbooleanを返す必要があります。
     *
     * @access public
     * @static
     * @param string|array $value 検証対象の値
     * @param callback $callback 検証用のコールバック関数
     * @param array $params コールバック関数で使用するパラメータ配列
     * @return boolean 正しければtrue
     */
    function isValid($value, $callback, $params = null)
    {
        $values = Moony_Utils::toArray($value);
        foreach ($values as $each) {
            $each_params = array($each);
            if (!is_null($params)) {
                $each_params += Moony_Utils::toArray($params);   // combined
            }
            if (strlen($each) > 0 && !call_user_func_array($callback, $each_params)) {
                return false;
            }
        }
        return true;
    }
}
?>
