<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * リクエスト情報を格納するクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Request
{
    /** @var object Moony_Sessionのインスタンス */
    var $session;

    /** @var array 実際のパラメータを格納する配列 */
    var $params;

    /** @var array アップロードされるファイルに関する情報を格納する配列 */
    var $files;

    /** @var array PATH_INFOの値の配列 */
    var $path_info_array;

    /**
     * コンストラクタです。
     * 
     * @access public
     * @param object $session Moony_Sessionのインスタンス
     */
    function Moony_Request(&$session)
    {
        $this->session = &$session;

        // リクエストパラメータ
        $this->params = (strtolower($_SERVER['REQUEST_METHOD']) == 'post') ? $_POST : $_GET;
        $this->files = $_FILES;

        // 不要なエスケープ文字除去（magic_quotes_gpcがONの場合）
        $this->params = $this->_unescape($this->params);

        // NUL文字除去
        $this->params = Moony_Utils::removeNulString($this->params);
        $this->files = Moony_Utils::removeNulString($this->files);

        // エンコーディング変換
        if (MOONY_REQUEST_PARAMETER_CONVERT_ENCODING &&
            MOONY_OUTPUT_ENCODING != MOONY_INTERNAL_ENCODING) {
            mb_convert_variables(MOONY_INTERNAL_ENCODING, MOONY_OUTPUT_ENCODING, &$this->params);
            mb_convert_variables(MOONY_INTERNAL_ENCODING, MOONY_OUTPUT_ENCODING, &$this->files);
        }

        // PATH_INFO取得、拡張子除去
        $path_info = Moony_Utils::getEnvVar('PATH_INFO', '');
        $path_info = Moony_Utils::removeFileExtension($path_info);

        // スラッシュ区切りで分割、不要部分除去
        $this->path_info_array = explode('/', trim($path_info, '/'));
        for ($i = 0; $i < MOONY_EVAL_AS_ACTION_SETTING; $i++) {
            array_shift(&$this->path_info_array);
        }
    }

    /**
     * パラメータ値を取得します。
     * 
     * @access public
     * @param string $name パラメータキー
     * @param mixed $alt パラメータが存在しない場合の代替値（デフォルトnull）
     * @return mixed パラメータ値
     */
    function get($name, $alt = null)
    {
        return Moony_Utils::getArrayValue($name, $this->params, $alt);
    }

    /**
     * 全てのパラメータを連想配列として返します。
     *
     * @access public
     * @return array 全てのパラメータの連想配列
     */
    function getAll()
    {
        return $this->params;
    }

    /**
     * 指定された名称のリクエストパラメータが
     * 存在するかどうかを返します。
     *
     * @access public
     * @param string $name パラメータキー
     * @return boolean 存在する場合true
     */
    function exists($name)
    {
        return array_key_exists($name, $this->params);
    }

    /**
     * パラメータの値を指定された値に置き換えます。
     * 該当する名称のパラメータが存在しなかった場合、
     * 何も処理を行わずにfalseを返します。
     *
     * @access public
     * @param string $name パラメータ名称
     * @param mixed $value パラメータ値を置換する値
     * @return boolean 置換に成功したらtrue、失敗したらfalse
     */
    function replace($name, $value)
    {
        if ($this->exists($name)) {
            $this->params[$name] = $value;
            return true;
        }
        return false;
    }

    /**
     * アップロードされたファイルに関する情報を
     * 連想配列として返します。返される値は$_FILES[$name]と同等です。
     * ファイル名はエンコーディング変換されています。
     * 指定された名称のファイルがアップロード対象でない場合、無条件にnullを返します。
     *
     * @access public
     * @param string $name 処理対象のファイルに付けられた名前
     * @return array アップロードされたファイルに関する情報
     */
    function getFile($name)
    {
        return Moony_Utils::getArrayValue($name, $this->files);
    }

    /**
     * 指定されたインデックスのPATH_INFOの値を返します。
     * 例えば、PATH_INFOが/foo/bar/p1/p2.htmlで
     * アクション名称がFooBarの場合、
     * getPathInfo(0)の戻り値は'p1'、
     * getPathInfo(1)の戻り値は'p2'になります。
     *
     * @access public
     * @param integer $index PATH_INFOの値の配列のインデックス
     * @param string $alt 該当のインデックスが存在しない場合の代替値
     */
    function getPathInfo($index, $alt = null)
    {
        return Moony_Utils::getArrayValue($index, $this->path_info_array, $alt);
    }

    /**
     * トランザクショントークンの妥当性チェックを行います。
     * セッションが開始されていない場合、無条件にfalseを返します。
     * トークンが送信されてこない場合、無条件にfalseを返します。
     * 比較後、セッション内のトークン値は直ちに削除されます。
     *
     * @access public
     * @return boolean トークンがセッションに格納された値と一致する場合true
     */
    function checkToken()
    {
        if (!$this->session->hasStarted() || !$this->exists(MOONY_TRANSACTION_TOKEN_NAME)) {
            return false;
        }

        // セッションから保存しておいたトークン値取得、直ちにセッションから削除
        $saved = $this->session->get(MOONY_TRANSACTION_TOKEN_NAME);
        $this->session->remove(MOONY_TRANSACTION_TOKEN_NAME);

        // リクエストパラメータと比較
        if ($this->get(MOONY_TRANSACTION_TOKEN_NAME) === $saved) {
            return true;
        }

        return false;
    }

    /**
     * magic_quotes_gpcがONになっている場合、
     * パラメータに付加されている不要なエスケープ文字を除去します。
     *
     * @access private
     * @param mixed $var 処理対象の変数
     * @return mixed 不要なエスケープ文字を除去した結果
     */
    function _unescape($var)
    {
        if (is_array($var)) {
            return array_map(array(&$this, '_unescape'), $var);
        }
        if (get_magic_quotes_gpc()) {
            $var = stripslashes($var);
        }
        return $var;
    }
}
?>
