<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * テンプレートファイルに設定する値を
 * 保持するためのクラスです。
 * また、その他の出力情報に関しても取り扱います。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Response
{
    /** @var string 画面描画に使用されるテンプレートファイル名 */
    var $template;

    /** @var array テンプレートに設定する値の連想配列 */
    var $attributes;

    /** @var string 出力バッファ */
    var $buffer;

    /**
     * コンストラクタです。
     *
     * @access public
     */
    function Moony_Response()
    {
        $this->template = '';
        $this->attributes = array();
    }

    /**
     * テンプレートの変数に値を設定します。
     *
     * @access public
     * @param string $name テンプレートの変数名
     * @param mixed $value 設定する値
     * @param boolean $escape 値をエスケープするかどうか
     */
    function set($name, $value, $escape = true)
    {
        if ($escape) {
            $value = $this->escape($value);
        }
        $this->attributes[$name] = $value;
    }

    /**
     * テンプレートに設定する値を設定します。
     *
     * @access public
     * @param array $attributes テンプレートの変数名をキーに、割り当てる値を値に持つ連想配列
     * @param boolean $escape 値をエスケープ処理するかどうか
     */
    function setAll($attributes, $escape = true)
    {
        foreach ($attributes as $name => $value) {
            $this->set($name, $value, $escape);
        }
    }

    /**
     * テンプレートに設定する値を全て取得します。
     *
     * @access public
     * @return array テンプレートに設定する全ての値
     */
    function getAll()
    {
        return $this->attributes;
    }

    /**
     * 画面描画に使用するテンプレートファイル名を設定します。
     * 拡張子を除いた名称を設定してください。
     *
     * @access public
     * @param string $template テンプレートファイル名
     */
    function setTemplate($template)
    {
        $this->template = $template;
    }

    /**
     * 画面描画に使用するテンプレートファイル名を取得します。
     *
     * @access public
     * @return string テンプレートファイル名
     */
    function getTemplate()
    {
        return $this->template;
    }

    /**
     * 指定されたURLにリダイレクトして処理を終了します。
     *
     * @access public
     * @param string $url リダイレクト先のURL
     */
    function redirect($url)
    {
        if (!headers_sent()) {
            if (ob_get_length() !== false) {
                ob_end_clean();
            }
            header("Location: ${url}");
            exit;
        }
    }

    /**
     * 指定されたファイルをHTTPレスポンスに添付します。
     * $nameが指定されない場合、元のファイルの名称がそのまま使用されます。
     *
     * @access public
     * @static
     * @param string $path 添付するファイルのパス
     * @param string $name 添付するファイルにつける名前
     */
    function sendFile($path, $name)
    {
        $has_error = false;
        if (!file_exists($path)) {
            Moony_Logger::warn('File not found: ' . $path, __FILE__, __LINE__);
            $has_error = true;
        }
        if (!($fp = fopen($path, 'r'))) {
            Moony_Logger::warn('Unable to open file: ' . $path, __FILE__, __LINE__);
            $has_error = true;
        } else {
            fclose($fp);
        }
        if (($size = filesize($path)) == 0) {
            Moony_Logger::warn('File size = 0: ' . $path, __FILE__, __LINE__);
            $has_error = true;
        }
        if ($has_error) {
            Moony_Error::notFound();
        }

        if (strlen($name) == 0) {
            $name = basename($path);
        }

        header("Content-Disposition: attachment; filename=\"${name}\"");
        header("Content-Length: ${size}");
        header('Content-Type: application/octet-stream');

        if (!readfile($path)) {
            Moony_Logger::warn('Unable to read file: ' . $path, __FILE__, __LINE__);
            Moony_Error::notFound();
        }
    }

    /**
     * 文字列を出力します。
     * $encodingが指定された場合、エンコーディング変換処理が行われます。
     * このメソッドは複数回実行することが可能です。
     * 出力内容はバッファされ、アクションの処理終了後にまとめて出力されます。
     *
     * @access public
     * @param string $content 出力する内容
     * @param string $encoding 出力する文字エンコーディング
     */
    function output($content, $encoding = MOONY_INTERNAL_ENCODING)
    {
        if ($encoding != MOONY_INTERNAL_ENCODING) {
            $content = mb_convert_encoding($content, $encoding, MOONY_INTERNAL_ENCODING);
        }
        $this->buffer .= $content;
    }

    /**
     * 出力された全ての内容を取得します。
     *
     * @access public
     * @return string 出力された全ての内容
     */
    function getOutput()
    {
        return $this->buffer;
    }

    /**
     * 指定された変数をhtmlに埋め込むための
     * エスケープ処理を行います。
     *
     * @access public
     * @param mixed $var エスケープ対象の変数
     * @return mixed エスケープ済みの変数
     */
    function escape($var)
    {
        if (is_object($var)) {
            return $var;
        }
        if (is_array($var)) {
            return array_map(array(&$this, 'escape'), $var);
        }
        return htmlspecialchars($var, ENT_QUOTES);
    }
}
?>
