<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * ユーティリティを提供するクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Utils
{
    /**
     * パスを組み立てます。
     *
     * @access public
     * @static
     * @param string $dir ディレクトリ名
     * @param string $file ファイル名
     * @param string $ext ファイルの拡張子
     * @return string それぞれを連結したパス文字列
     */
    function buildPath($dir, $file, $ext = null)
    {
        if (strlen($dir) > 0 && substr($dir, -1, 1) == DIRECTORY_SEPARATOR) {
            $dir = substr($dir, 0, -1);
        }
        if (strlen($file) > 0 && substr($file, 0, 1) == DIRECTORY_SEPARATOR) {
            $file = substr($file, 1);
        }
        if (strlen($dir) > 0) {
            $file = $dir . DIRECTORY_SEPARATOR . $file;
        }
        if (strlen($ext) == 0 || strlen($file) == 0 || substr($file, -1, 1) == DIRECTORY_SEPARATOR) {
            return $file;
        }
        return $file . '.' . $ext;
    }

    /**
     * パスからファイル拡張子を取り除いた文字列を返します。
     * パスがDIRECTORY_SEPARATORで終了している場合、何も処理を行いません。
     *
     * @access public
     * @static
     * @param string $path 処理対象のパス
     * @return string 拡張子を取り除いた文字列
     */
    function removeFileExtension($path)
    {
        if (strlen($path) == 0 || substr($path, -1, 1) == DIRECTORY_SEPARATOR) {
            return $path;
        }
        $pos = strrpos($path, '.');
        if ($pos !== false) {
            $path = substr($path, 0, $pos);
        }
        return $path;
    }

    /**
     * 配列でないスカラー値を要素が1つの配列に変換します。
     *
     * @access public
     * @static
     * @param mixed $var 配列として扱う値
     * @return array 配列に変換された値
     */
    function toArray($var) {
        return is_array($var) ? $var : array($var);
    }

    /**
     * 配列の要素の値を取得します。
     * 該当するキーが存在しない場合、指定された代替値を返します。
     *
     * @access public
     * @static
     * @param integer|string $key 配列のキー
     * @param array $array 処理対象の配列
     * @param mixed $alt 代替値
     * @return 該当する要素の値、または代替値
     */
    function getArrayValue($key, $array, $alt = null)
    {
        if (is_array($array) && array_key_exists($key, $array)) {
            return $array[$key];
        }
        return $alt;
    }

    /**
     * CamelCaseに変換します。
     * 例えば、regist_userをRegistUserに変換します。
     *
     * @access public
     * @static
     * @param string $str 変換対象文字列
     * @param string $delim 変換元の単語区切り文字
     * @return string CamelCaseに変換した文字列
     */
    function camelize($str, $delim = '_')
    {
        if (strlen($delim) == 0) {
            return ucwords($str);
        }
        $words = explode($delim, $str);
        for ($i = 0; $i < count($words); $i++) {
            $words[$i] = ucfirst($words[$i]);
        }
        return implode('', $words);
    }

    /**
     * NUL文字を除去します。
     *
     * @access private
     * @param mixed $var 処理対象の変数
     * @return mixed NUL文字を除去した変数
     */
    function removeNulString($var)
    {
        if (is_array($var)) {
            return array_map(array('Moony_Utils', 'removeNulString'), $var);
        }
        return str_replace("\0", '', $var);
    }

    /**
     * Nul文字を取り除いた環境変数（$_SERVER）の値を取得します。
     *
     * @access public
     * @static
     * @param string $name 取得したい環境変数のキー/名称
     * @param mixed $alt 該当の環境変数が存在しない場合の代替値
     * @return mixed 該当する環境変数の値
     */
    function getEnvVar($name, $alt)
    {
        $value = Moony_Utils::getArrayValue($name, $_SERVER, $alt);
        return Moony_Utils::removeNulString($value);
    }

    /**
     * 指定されたファイルが読み込み可能かどうかを調べます。
     * include_pathを考慮して調べます。
     *
     * @access public
     * @static
     * @param string $file_name ファイル名称
     * @return boolean 読み込み可能ならtrue
     */
    function isReadableFile($file_name)
    {
        $fp = @fopen($file_name, 'r', true);
        if (is_resource($fp)) {
            fclose($fp);
            return true;
        }
        return false;
    }

    /**
     * 指定されたディレクトリが書き込み可能かどうか調べます。
     * ディレクトリが存在しない場合、パーミッション0777で作成を試みます。
     * 書き込み権限がない場合、パーミッションを0777に変更を試みます。
     * 最終的に書き込み可能な状態になった場合だけ、trueを返します。
     * それ以外の場合はすべてfalseを返します。
     *
     * @access public
     * @static
     * @param string $dir_name ディレクトリ名
     * @return boolean 書き込み可能ならtrue
     */
    function isWritableDir($dir_name)
    {
        if (!file_exists($dir_name)) {
            if (!mkdir($dir_name, 0777)) {
                return false;
            }
        }
        if (!is_writable($dir_name)) {
            if (!chmod($dir_name, 0777)) {
                return false;
            }
        }
        return true;
    }
}
?>
