<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * アクションの基底クラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Action
{
    /**
     * 出力に使用する値の連想配列
     * @var array
     */
    var $_moony_assignments = array();

    /**
     * Moony_Requestのインスタンス
     * @var object Moony_Reqeust
     */
    var $_moony_request;

    /**
     * Moony_Templateの実装クラスのインスタンス
     * @var object
     */
    var $_moony_template;

    /**
     * Moony_Sessionのインスタンス
     * @var object Moony_Session
     */
    var $_moony_session;

    /**
     * Moony_Flashのインスタンス
     * @var object Moony_Flash
     */
    var $_moony_flash;

    /**
     * 入力エンコーディング
     * @var string
     */
    var $_moony_input_encoding;

    /**
     * 出力エンコーディング
     * @var string
     */
    var $_moony_output_encoding;

    /**
     * 内部エンコーディング
     * @var string
     */
    var $_moony_internal_encoding;

    /**
     * リクエストパラメータとして受け取ったトランザクショントークン文字列
     * @var string
     */
    var $_moony_token_received;

    /**
     * セッションに保存しておいたトランザクショントークン文字列
     * @var string
     */
    var $_moony_token_saved;

    /**
     * PATH_INFO経由で渡された引数の連想配列
     * @var array
     */
    var $_moony_args;

    /**
     * 出力用の値を割り当てます。
     * $varが配列の場合、展開して全ての値を割り当てます。
     * また、同時にテンプレートに値を割り当てます。
     * テンプレートに割り当てる値は、$escapeがtrueの場合エスケープされます。
     * エスケープ処理の挙動を変更したい場合は、
     * Moony_Action::_escape()をオーバーライドしてください。
     *
     * @access public
     * @param string|array $var 値の名称、または値の連想配列
     * @param mixed $value 割り当てられる値
     * @param boolean $escape 値をエスケープ処理するかどうか
     */
    function assign($var, $value = null, $escape = true)
    {
        if (is_array($var)) {
            foreach ($var as $name => $val) {
                $this->_moony_assignments[$name] = $val;
                $this->_moony_template->assign(
                    $name, ($escape ? $this->_escape($val) : $val));
            }
        } else {
            $this->_moony_assignments[$var] = $value;
            $this->_moony_template->assign(
                $var, ($escape ? $this->_escape($value) : $value));
        }
    }

    /**
     * テンプレートの処理結果を取得します。
     * テンプレートファイルが存在しない場合、falseを返します。
     * クラスのプロパティ値もエスケープされた上で
     * テンプレートに割り当てられます。
     * （詳細はMoony_Action::_assignPropertiesToTemplate()を参照）
     *
     * @access public
     * @param string $template テンプレートファイル名
     * @return string|false テンプレートの処理結果
     */
    function fetch($template)
    {
        // プロパティをテンプレートに追加割り当て
        // （値の上書きは行われない）
        $this->_assignPropertiesToTemplate();

        // テンプレートの処理
        return $this->_moony_template->fetch($template);
    }

    /**
     * テンプレートの処理結果を出力します。
     * テンプレートファイルが存在しない場合、
     * HTTPのステータスで404を送出して処理を終了します。
     * クラスのプロパティ値もエスケープされた上で
     * テンプレートに割り当てられます。
     * （詳細はMoony_Action::_assignPropertiesToTemplate()を参照）
     * $convert_encodingがtrueの場合、出力内容のエンコーディングを
     * 内部エンコーディングから出力エンコーディングに変換します。
     *
     * @access public
     * @param string $template テンプレートファイル名
     * @param boolean $convert_encoding エンコーディング変換を行うかどうか
     */
    function display($template, $convert_encoding = true)
    {
        $result = $this->fetch($template);
        if ($result === false) {
            // not found...
            header('HTTP/1.0 404 Not Found');
            exit;
        }
        $this->output($result, $convert_encoding);
    }

    /**
     * 検証エラーが存在する場合、
     * テンプレートの処理結果を出力して処理を終了します。
     * テンプレートファイルが存在しない場合、
     * HTTPのステータスで404を送出して処理を終了します。
     * クラスのプロパティ値もエスケープされた上で
     * テンプレートに割り当てられます。
     * （詳細はMoony_Action::_assignPropertiesToTemplate()を参照）
     * $convert_encodingがtrueの場合、出力内容のエンコーディングを
     * 内部エンコーディングから出力エンコーディングに変換します。
     *
     * @access public
     * @param boolean|object $v 検証結果、またはMoony_Validatorのインスタンス
     * @param string $template テンプレートファイル名
     * @param boolean $convert_encoding エンコーディング変換を行うかどうか
     */
    function displayOnError(&$v, $template, $convert_encoding = true)
    {
        if ($v === false || $v->hasError()) {
            // エラーの場合、テンプレートを表示して処理を終了
            $this->display($template, $convert_encoding);
            exit;
        }
    }

    /**
     * リダイレクト処理を行います。
     *
     * @access public
     * @param string $url リダイレクト先のURL
     */
    function redirect($url)
    {
        if (!headers_sent()) {
            if (ob_get_length() !== false) {
                ob_end_clean();
            }
            header("Location: {$url}");
        }
    }

    /**
     * 検証エラーが存在する場合、
     * 指定されたURLにリダイレクトして処理を終了します。
     *
     * @access public
     * @param boolean|object $v 検証結果、またはMoony_Validatorのインスタンス
     * @param string $url リダイレクト先のURL
     */
    function redirectOnError(&$v, $url)
    {
        if ($v === false || $v->hasError()) {
            $this->redirect($url);
            exit;
        }
    }

    /**
     * テキストを出力します。
     * $convert_encodingがtrueで出力エンコーディングが指定されている場合、
     * 出力内容を内部エンコーディングから出力エンコーディングに変換します。
     *
     * @access public
     * @param string $content 出力するテキスト
     * @param boolean $convert_encoding エンコーディング変換を行うかどうか
     */
    function output($content, $convert_encoding = true)
    {
        if ($convert_encoding && !is_null($this->_moony_output_encoding)) {
            // エンコーディング変換
            $content = mb_convert_encoding($content,
                $this->_moony_output_encoding,
                $this->_moony_internal_encoding);
        }
        echo $content;
    }

    /**
     * $varが配列の場合、展開して全ての値をセッションに設定します。
     *
     * $varが配列でなく、$valueが設定されている場合、
     * セッションに$varという名称で値を設定します。
     *
     * $varが配列でなく、$valueが設定されていない場合、
     * セッションから$varという名称の値を取得します。
     * 該当する値が存在しない場合、nullを返します。
     *
     * @access public
     * @param string|array $var 値の名称、または設定する値の連想配列
     * @param mixed $value 設定する値
     * @return mixed|null 取得した値
     */
    function session($var, $value = null)
    {
        if (is_array($var)) {
            $this->_moony_session->set($var);
        } else {
            if (is_null($value)) {
                return $this->_moony_session->get($var);
            } else {
                $this->_moony_session->set($var, $value);
            }
        }
    }

    /**
     * $varが配列の場合、展開して全ての値をフラッシュ領域に設定します。
     *
     * $varが配列でなく、$valueが設定されている場合、
     * フラッシュ領域に$varという名称で値を設定します。
     *
     * $varが配列でなく、$valueが設定されていない場合、
     * フラッシュ領域から$varという名称の値を取得します。
     * 該当する値が存在しない場合、nullを返します。
     *
     * @access public
     * @param string|array $var 値の名称、または設定する値の連想配列
     * @param mixed $value 設定する値
     * @return mixed|null 取得した値
     */
    function flash($var, $value = null)
    {
        if (is_array($var)) {
            $this->_moony_flash->set($var);
        } else {
            if (is_null($value)) {
                return $this->_moony_flash->get($var);
            } else {
                $this->_moony_flash->set($var, $value);
            }
        }
    }

    /**
     * 出力用に割り当てた値の連想配列を取得します。
     * Moony_Action::assign()で割り当てられた値のみを返し、
     * プロパティの値は戻り値に含まれません。
     *
     * @access public
     * @return array 出力用に割り当てた値の連想配列
     */
    function getAssignments()
    {
        return $this->_moony_assignments;
    }

    /**
     * 出力用に割り当てられた値をJSONフォーマットに変換、取得します。
     * このメソッドを使用するためにはライブラリ「Jsphon」が必要です。
     * （URL: http://www.hawklab.jp/jsonencoder/）
     *
     * @access public
     * @return string JSONフォーマットされた値
     */
    function getJson()
    {
        if (class_exists('Jsphon')) {
            $value = $this->_moony_assignments;
            if ($this->_moony_internal_encoding != 'UTF-8') {
                mb_convert_variables(
                    'UTF-8', $this->_moony_internal_encoding, $value);
            }
            return Jsphon::encode($value);
        } else {
            trigger_error('Not loaded: Jsphon', E_USER_WARNING);
        }
    }

    /**
     * Moony_Requestのインスタンスを取得します。
     *
     * @access public
     * @param object Moony_Request
     */
    function &getRequest()
    {
        return $this->_moony_request;
    }

    /**
     * Moony_Sessionのインスタンスを取得します。
     *
     * @access public
     * @param object Moony_Session
     */
    function &getSession()
    {
        return $this->_moony_session;
    }

    /**
     * Moony_Flashのインスタンスを取得します。
     *
     * @access public
     * @param object Moony_Flash
     */
    function &getFlash()
    {
        return $this->_moony_flash;
    }

    /**
     * Smartyのインスタンスを取得します。
     *
     * @access public
     * @return object Smarty
     */
    function &getSmarty()
    {
        $smarty = null;
        $template_class_name = get_class($this->_moony_template);
        if (strtolower($template_class_name) == 'moony_template_smarty') {
            $smarty =& $this->_moony_template->smarty;
        }
        return $smarty;
    }

    /**
     * PATH_INFO経由で渡されたパラメータを取得します。
     * $indexが設定されていない場合、
     * 全ての引数を「/」で連結した文字列を返します。
     *
     * @access public
     * @param integer $index パラメータのインデックス
     * @return string パラメータの値
     */
    function getArg($index = null)
    {
        if (is_null($index)) {
            return implode('/', $this->_moony_args);
        }
        if ($index < count($this->_moony_args)) {
            return $this->_moony_args[$index];
        }
    }

    /**
     * 出力エンコーディングを設定します。
     * 初期値として、Moony::setOutputEncoding()で
     * 設定された値が設定されています。
     *
     * @access public
     * @param string $output_encoding 出力エンコーディング
     */
    function setOutputEncoding($output_encoding)
    {
        $this->_moony_output_encoding = $output_encoding;
    }

    /**
     * トランザクショントークン文字列の妥当性検査を行います。
     * リクエストパラメータとして受け取った値と、
     * セッションに保存していた値が一致するかどうかを調べます。
     *
     * @access public
     * @return boolean 妥当なトランザクショントークンかどうか
     */
    function checkToken()
    {
        if (is_null($this->_moony_token_received) ||
            is_null($this->_moony_token_saved)) {
            return false;
        }
        return $this->_moony_token_received == $this->_moony_token_saved;
    }

    /**
     * ルータクラスから呼び出され、
     * アクションの各プロセスを実行します。
     *
     * @access public
     * @param object $moony Moony
     * @param array $args PATH_INFO経由で渡された引数の配列
     */
    function process(&$moony, $args)
    {
        ob_start();
        header('X-Framework: Moony/' . MOONY_VERSION);
        $this->_initializeAction($moony, $args);
        $this->_extractParameters();
        $this->_invokeMethods();
        ob_end_flush();
    }

    /**
     * アクションクラスの初期化および設定を行います。
     *
     * @access protected
     * @param object $moony Moony
     * @param array $args the arguments given by the PATH_INFO string
     */
    function _initializeAction(&$moony, $args)
    {
        // コンポーネント生成
        $this->_moony_request =& new Moony_Request();
        $this->_moony_session =& new Moony_Session(true);
        $this->_moony_flash =& new Moony_Flash();

        // テンプレートのレンダラ
        if ($moony->use_smarty) {
            $configs = $moony->smarty_properties;
            $configs['template_dir'] = $moony->template_dir;
            $this->_moony_template =& new Moony_Template_Smarty();
            $this->_moony_template->configure($configs);
        } else {
            $configs = array('template_dir' => $moony->template_dir);
            $this->_moony_template =& new Moony_Template_Php();
            $this->_moony_template->configure($configs);
        }

        // エンコーディング
        if (!is_null($moony->input_encoding)) {
            $this->_moony_input_encoding = $moony->input_encoding;
        }
        if (!is_null($moony->output_encoding)) {
            $this->_moony_output_encoding = $moony->output_encoding;
        }
        if (!is_null($moony->internal_encoding)) {
            $this->_moony_internal_encoding = $moony->internal_encoding;
        } else {
            $this->_moony_internal_encoding = mb_internal_encoding();
        }

        // リクエストパラメータのエンコーディング変換
        if (!is_null($this->_moony_input_encoding)) {
            $this->_moony_request->convertEncoding(
                $this->_moony_input_encoding, $this->_moony_internal_encoding);
        }

        // トランザクショントークンの処理
        $this->_moony_token_received =
            $this->_moony_request->get(MOONY_TOKEN_KEY);
        $this->_moony_token_saved =
            $this->_moony_session->get(MOONY_TOKEN_KEY);
        $this->_moony_session->set(MOONY_TOKEN_KEY, md5(uniqid(rand(), true)));

        // PATH_INFO経由で渡された引数
        $this->_moony_args = $args;
    }

    /**
     * リクエストパラメータを展開、
     * アクションのプロパティとして設定します。
     * 設定されるのは予めプロパティが用意されていた場合だけです。
     *
     * @access protected
     */
    function _extractParameters()
    {
        $parameters = $this->_moony_request->getAll();
        $properties = get_object_vars($this);
        foreach ($parameters as $name => $value) {
            if (array_key_exists($name, $properties)) {
                $this->{$name} = $value;
            }
        }
    }

    /**
     * アクションクラスの各メソッドを実行します。
     * 以下の順番で実行されます（該当メソッドが存在する場合）。
     * (1) prepare():void
     * (2) convert(&$converter:Moony_Converter):void
     * (3) validate(&$validator:Moony_Validator):void
     * (4) execute():void
     *
     * @access protected
     */
    function _invokeMethods()
    {
        // (1) prepare
        if (method_exists($this, 'prepare')) {
            $this->prepare();
        }
 
        // (2) convert
        if (method_exists($this, 'convert')) {
            $converter =& new Moony_Converter($this->_moony_internal_encoding);
            $this->convert($converter);
        }

        // (3) validate
        if (method_exists($this, 'validate')) {
            $validator =& new Moony_Validator($this->_moony_internal_encoding);
            $this->validate($validator);
        }

        // (4) execute
        if (method_exists($this, 'execute')) {
            $this->execute();
        }
    }

    /**
     * クラスのプロパティをテンプレートにだけ割り当てます。
     * 既に同一の名称が出力に使用する値として
     * 割り当てられている場合、値の上書きは行われません。
     * 名称が「_」で始まるプロパティは処理の対象外です。
     * また、割り当てられるプロパティの値は
     * Moony_Action::_escape()を用いてエスケープされます。
     * エスケープしたくない場合はMoony_Action::assign()を用いて
     * 出力に使用する値として明示的に割り当ててください。
     *
     * @access protected
     */
    function _assignPropertiesToTemplate()
    {
        $properties = get_object_vars($this);
        foreach ($properties as $name => $value) {
            if ($name{0} != '_' &&
                !array_key_exists($name, $this->_moony_assignments)) {
                $this->_moony_template->assign($name, $this->_escape($value));
            }
        }
    }

    /**
     * テンプレートに設定される値のエスケープ処理を行います。
     *
     * @access protected
     * @param mixed $var 設定される値
     * @return mixed エスケープされた値
     */
    function _escape($var)
    {
        if (is_object($var)) {
            // オブジェクトの場合、何もしない
            return $var;
        }
        if (is_array($var)) {
            return array_map(array($this, '_escape'), $var);
        }
        return htmlspecialchars($var, ENT_QUOTES);
    }
}
?>
