<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * アクションのルーティングを行うクラスです。
 * 下記例のような順序でアクションクラスを探索、処理を起動します。
 *   - foo => Foo
 *   - foo/bar => Foo_Bar, Foo
 *   - foo/bar/baz => Foo_Bar_Baz, Foo_Bar, Foo
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Router
{
    /**
     * PATH_INFOを元に実行する
     * アクションクラスを探索、処理を起動します。
     *
     * アクションクラスが見つからず、
     * デフォルトアクションがMoony::setDefaultAction()で
     * 設定されている場合はそのアクションを起動します。
     *
     * デフォルトのリダイレクト先がMoony::setDefaultRedirect()で
     * 設定されている場合はそのURLにリダイレクトします。
     *
     * 結局見つからなかった場合、HTTPのステータスで404を送出します。
     *
     * @access public
     * @param object $moony Moony
     */
    function route(&$moony)
    {
        $default_action_name = $moony->default_action_name;

        if (is_null($moony->action_dir)) {
            // アクションクラスを格納するディレクトリが設定されていない
            trigger_error('Not defined: action_dir', E_USER_ERROR);
            header('HTTP/1.0 500 Internal Server Error');
            exit;
        }

        // PATH_INFOを取得、小文字化、分割、CamelCase
        $path_info = $this->_getPathInfo();
        $path_info = strtolower($path_info);
        $paths = explode('/', trim($path_info, '/'));
        $paths = array_map('ucfirst', $paths);

        // PATH_INFO経由で渡されるパラメータを格納
        $args = array();

        while (count($paths) > 0) {

            $action_path = implode(DIRECTORY_SEPARATOR, $paths);
            $action_name = str_replace(DIRECTORY_SEPARATOR, '_', $action_path);

            if ($this->_invokeAction(
                $action_path, $action_name, $moony, $args)) {
                // アクションクラス生成、処理を起動
                return;
            }
            $args[] = strtolower(array_pop($paths));
        }

        // 結局見つからなかった
        if (!is_null($moony->default_action_name)) {
            // デフォルトアクションが設定されている場合
            $action_path = str_replace(
                '_', DIRECTORY_SEPARATOR, $moony->default_action_name);
            if ($this->_invokeAction(
                $action_path, $moony->default_action_name, $moony, $args)) {
                return;
            }
        }
        if (!is_null($moony->default_redirect_url)) {
            // デフォルトリダイレクト先が設定されている場合
            if (!headers_sent()) {
                if (ob_get_length() !== false) {
                    ob_end_clean();
                }
                header('Location: ' . $moony->default_redirect_url);
                return;
            }
        }

        // 404エラー
        header('HTTP/1.0 404 Not Found');
        exit;
    }

    /**
     * PATH_INFO文字列を取得します。
     * 最後の文字がスラッシュの場合（ディレクトリを示していた場合）、
     * 末尾に「index」を付加します。拡張子は無視されます。
     *
     * @access protected
     * @return string PATH_INFO文字列
     */
    function _getPathInfo()
    {
        $path_info = '';
        if (isset($_SERVER['PATH_INFO'])) {
            $path_info = $_SERVER['PATH_INFO'];
            $path_info = str_replace("\0", '', $path_info);
            $pos = strpos($path_info, '.');
            if ($pos !== false) {
                $path_info = substr($path_info, 0, $pos);
            }
        }
        if (strlen($path_info) == 0 ||
            substr($path_info, -1, 1) == '/') {
            // PATH_INFOが存在しない or "/"で終了
            $path_info .= 'index';
        }
        return $path_info;
    }

    /**
     * アクションの処理を起動します。
     *
     * @access protected
     * @param string $action_path アクションパス
     * @param string $action_name アクションクラス名
     * @param object $moony Moony
     * @param array $args PATH_INFOとして渡されたパラメータの配列
     */
    function _invokeAction($action_path, $action_name, $moony, $args)
    {
        $path = $moony->action_dir .
            DIRECTORY_SEPARATOR . $action_path . '.php';
        if (file_exists($path)) {
            include_once $path;
            if (class_exists($action_name)) {
                $action =& new $action_name;
                $action->process($moony, array_reverse($args));
                return true;
            }
        }
        return false;
    }
}
?>
