<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

define('MOONY_SESSION_START_TIME_KEY', '_moony_session_start_time');

/**
 * セッションを取り扱うクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Session
{
    /**
     * コンストラクタです。
     *
     * @access public
     * @param boolean $start セッションを開始するかどうか
     */
    function Moony_Session($start = true)
    {
        if ($start) {
            // セッションの開始
            $this->start();
        }
    }

    /**
     * セッションの値を取得します。
     * 該当する値が存在しない場合、$defaultを返します。
     *
     * @access public
     * @param string $name セッションの値の名称
     * @param mixed $default 値が存在しない場合のデフォルト値
     * @return mixed セッションの値
     */
    function get($name, $default = null)
    {
        if ($this->exists($name)) {
            return $_SESSION[$name];
        }
        return $default;
    }

    /**
     * セッションに値を設定します。
     * $varが配列の場合、展開して全ての値を設定します。
     *
     * @access public
     * @param string|array $var 値の名称、または値の連想配列
     * @param mixed $value 設定する値
     */
    function set($var, $value = null)
    {
        if (is_array($var)) {
            foreach ($var as $name => $val) {
                $_SESSION[$name] = $val;
            }
        } else {
            $_SESSION[$var] = $value;
        }
    }

    /**
     * 指定された名称の値がセッションに
     * 保存されているかどうか調べます。
     *
     * @access public
     * @param string $name セッションの値の名称
     * @return boolean セッションに保存されているかどうか
     */
    function exists($name)
    {
        return array_key_exists($name, $_SESSION);
    }

    /**
     * セッションから値を除去します。
     *
     * @access public
     * @param string $name 除去する値の名称
     * @return mixed 除去された値
     */
    function remove($name)
    {
        if ($this->exists($name)) {
            $value = $_SESSION[$name];
            unset($_SESSION[$name]);
            return $value;
        }
    }

    /**
     * セッションに保存された全ての値を消去します。
     *
     * @access public
     */
    function clean()
    {
        $_SESSION = array();
    }

    /**
     * セッションを開始します。
     *
     * @access public
     */
    function start()
    {
        session_start();
        if (array_key_exists(MOONY_SESSION_START_TIME_KEY, $_SESSION)) {
            // 開始時刻が保存されていない場合、
            // 不正なセッションなので再スタート
            $this->clean();
            $this->regenerateId();
            // 開始時刻を保存
            $this->set(MOONY_SESSION_START_TIME_KEY, time());
        }
    }

    /**
     * セッションを破棄します。
     * セッションに保存された全ての値は破棄されます。
     *
     * @access public
     */
    function destroy()
    {
        $this->clean();
        session_destroy();
    }

    /**
     * セッションIDを再生成します。
     *
     * @access public
    */
    function regenerateId()
    {
        if (version_compare(PHP_VERSION, '5.1.0') >= 0) {
            // PHP 5.1.0以上の場合はsession_regenerate_id(true)
            session_regenerate_id(true);
            return;
        }

        $saved = $_SESSION;

        $this->destory();
        session_id(md5(uniqid(rand(), true)));
        $this->start();
        $this->set($saved);
    }
}
?>
