<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * PHPが記述されたファイルを
 * テンプレートとして扱うためのクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Template_Php extends Moony_Template
{
    /**
     * テンプレートファイルが格納されるディレクトリ
     * @var string
     */
    var $_template_dir;

    /**
     * テンプレートに割り当てる値の連想配列
     * @var array
     */
    var $_template_vars = array();

    /**
     * テンプレートに値を割り当てます。
     * $varが配列の場合、展開して全ての値を割り当てます。
     *
     * @access public
     * @param string|array $var 値の名称、または割り当てる値の連想配列
     * @param mixed $value 割り当てる値
     */
    function assign($var, $value = null)
    {
        if (is_array($var)) {
            foreach ($var as $name => $val) {
                $this->_template_vars[$name] = $val;
            }
        } else {
            $this->_template_vars[$var] = $value;
        }
    }

    /**
     * テンプレートの処理結果を取得します。
     * テンプレートファイルが存在しない場合、falseを返します。
     *
     * @access public
     * @param string $template テンプレートファイル名
     * @return string|false テンプレートの処理結果
     */
    function fetch($template)
    {
        if (is_null($this->_template_dir)) {
            // テンプレートファイルが格納されるディレクトリ未設定
            trigger_error('Not defined: template_dir', E_USER_ERROR);
            header('HTTP/1.0 500 Internal Server Error');
            exit;
        }

        $template_path = $this->_template_dir . DIRECTORY_SEPARATOR . $template;
        if (!file_exists($template_path)) {
            // テンプレートファイルが存在しない
            return false;
        }

        // 割り当てる値を展開
        extract($this->_template_vars, EXTR_SKIP);

        ob_start();
        include $template_path;
        $result = ob_get_clean();

        return $result;
    }

    /**
     * 設定を行います。
     * テンプレートファイルが格納されるディレクトリを設定します。
     *
     * @access public
     * @param array $configs 設定内容の連想配列
     */
    function configure($configs)
    {
        if (isset($configs['template_dir'])) {
            $template_dir = $configs['template_dir'];
            $this->_template_dir = $template_dir;
            Moony_Registry::set(MOONY_TEMPLATE_DIR_KEY, $template_dir);
        }
    }
}

/**
 * 別のテンプレートファイルを読み込んで表示する
 * ヘルパー関数です。テンプレート内部で使用されます。
 * （例: <?php moony_include('foo.php'); ?>）
 * 該当のテンプレートファイルが存在しない場合、何も処理しません。
 *
 * @access public
 * @param string $template 読み込むテンプレートファイル名
 */
function moony_include($template)
{
    $template_dir = Moony_Registry::get(MOONY_TEMPLATE_DIR_KEY);
    $path = $template_dir . DIRECTORY_SEPARATOR . $template;
    if (file_exists($path)) {
        include $path;
    }
}

/**
 * hiddenフィールドとしてトランザクショントークンを描画する
 * ヘルパー関数です。テンプレート内部で使用されます。
 * （例: <?php moony_token(); ?>）
 *
 * @access public
 * @param boolean $xhtml XHTMLのフォーマットで描画するかどうか
 */
function moony_token($xhtml = true)
{
    $close_tag = $xhtml ? ' />' : '>';
    if (isset($_SESSION[MOONY_TOKEN_KEY])) {
        $token = $_SESSION[MOONY_TOKEN_KEY];
        echo '<input type="hidden" value="', $token, '"', $close_tag, "\n";
    }
}

/**
 * 全ての検証エラーメッセージを表示する
 * ヘルパー関数です。テンプレート内部で使用されます。
 * エラーメッセージが設定されていない場合、何も処理しません。
 * （例: <?php moony_errors('<ul class="errors">'); ?>）
 *
 * @access public
 * @param string $header 全体のヘッダ
 * @param string $footer 全体のフッタ
 * @param string $prefix 個々のメッセージの接頭辞
 * @param string $postfix 個々のメッセージの接尾辞
 */
function moony_errors($header = '<ul>',
    $footer = '</ul>', $prefix = '<li>', $postfix = '</li>')
{
    $error_messages = Moony_Registry::get(MOONY_ERROR_MESSAGES_KEY);
    if (!is_null($error_messages)) {
        echo $header;
        foreach ($error_messages as $error_message) {
            echo $prefix, $error_message, $postfix;
        }
        echo $footer;
    }
}

/**
 * 検証エラーの有無を判断する
 * ヘルパー関数です。テンプレート内部で使用されます。
 * （例: <?php if (moony_has_error()): ?>）
 *
 * @access public
 * @return boolean 検証エラーの存在有無
 */
function moony_has_error()
{
    $has_error = Moony_Registry::get(MOONY_HAS_ERROR_KEY);
    return is_null($has_error) ? false : $has_error;
}
?>
