/*
mpaligner is program to align string and string.
Copyright (C) 2010, 2011 Keigo Kubo

mpaligner is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
any later version.

mpaligner is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with mpaligner.  If not, see <http://www.gnu.org/licenses/>.

Imprement mnimum pattern alignment algorithm
date:   2010/9/16
author: Keigo Kubo
belong: Nara Institute Sience and Technology (NAIST)
e-mail: keigo-k{@}is.naist.jp   << Please transform {@} into @
*/

#include "mpAlign.h"

static Parameter addlog10(Parameter x, Parameter y){
	if (x < y) {
		return y + log(1 + pow(10, x-y)) * INV_LOG_TEN;
	} else {
		return x + log(1 + pow(10, y-x)) * INV_LOG_TEN;
	}
}

void readInputFile(TOTAL_INFO *info){
	FILE *fp;
	size_t size=256,byte_num=0;
	int c,i=0,state=0;
	char *s;
	LEN x_size=0,y_size=0;
	PAIR_DATA top,*pair_data;

	// temporary substitution.
	info->pair_data=&top;
	pair_data=&top;

	fprintf(stderr,"Reading the input file: %s\n", info->input_file);

	if ((fp = fopen(info->input_file, "r")) == NULL) {
		fprintf(stderr,"The input file open error: %s\n",info->input_file);
		exit(EXIT_FAILURE);
	}

	if((s=(char *)malloc(size))==NULL){
		fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
		exit(EXIT_FAILURE);
	}

	while(1){
		if(byte_num>size){
			size+=size;
			if((s=(char *)realloc((void *)s,size))==NULL){
				fprintf(stderr,"Don't get memory in realloc.\nYou must need more memory.\n");
				exit(EXIT_FAILURE);
			}
		}
		c=getc(fp);
		byte_num++;
		switch(c){
			case '\t':
				s[byte_num-1]='\0';
				x_size++;
				state++;
				break;
			case '\n':
				i++;
				y_size++;
				if(byte_num>2 && s[byte_num-2]=='\r'){
					s[byte_num-2]='\0';
					byte_num--;
				}else{
					s[byte_num-1]='\0';
				}

				if(state && s[byte_num-2]!='\0'){
					if((pair_data->next=(PAIR_DATA *)malloc(sizeof(PAIR_DATA)
						+(x_size+y_size)*sizeof(char *)+byte_num*sizeof(char)))==NULL){
							fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
							exit(EXIT_FAILURE);
					}

					pair_data=pair_data->next;
					pair_data->next=NULL;

					pair_data->x_part_strs=(char **)(pair_data+1);
					pair_data->y_part_strs=pair_data->x_part_strs+x_size;
					pair_data->x_part_strs[0]=(char *)(pair_data->y_part_strs+y_size);

					memcpy((void *)pair_data->x_part_strs[0],(void *)s,byte_num);
					for(c=1;c<x_size;c++){
						pair_data->x_part_strs[c]=pair_data->x_part_strs[c-1]
						+strlen(pair_data->x_part_strs[c-1])+1;
					}

					pair_data->y_part_strs[0]=pair_data->x_part_strs[c-1]
					+strlen(pair_data->x_part_strs[c-1])+1;
					for(c=1;c<y_size;c++){
						pair_data->y_part_strs[c]=pair_data->y_part_strs[c-1]
						+strlen(pair_data->y_part_strs[c-1])+1;
					}

					pair_data->x_size=x_size;
					pair_data->y_size=y_size;
					if(info->maxX<x_size){
						info->maxX=x_size;
					}
					if(info->maxY<y_size){
						info->maxY=y_size;
					}
				}else{
					if(byte_num>1){
						fprintf(stderr,"WARNING:The format is wrong:%s:%d line",info->input_file,i);
					}
				}

				x_size=0;
				y_size=0;
				state=0;
				byte_num=0;
				break;
			case EOF:
				i++;
				s[byte_num-1]='\0';
				y_size++;
				if(state && s[byte_num-2]!='\0'){
					if((pair_data->next=(PAIR_DATA *)malloc(sizeof(PAIR_DATA)
						+(x_size+y_size)*sizeof(char *)+byte_num*sizeof(char)))==NULL){
							fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
							exit(EXIT_FAILURE);
					}

					pair_data=pair_data->next;
					pair_data->next=NULL;

					pair_data->x_part_strs=(char **)(pair_data+1);
					pair_data->y_part_strs=pair_data->x_part_strs+x_size;
					pair_data->x_part_strs[0]=(char *)(pair_data->y_part_strs+y_size);

					memcpy((void *)pair_data->x_part_strs[0],(void *)s,byte_num);
					for(c=1;c<x_size;c++){
						pair_data->x_part_strs[c]=pair_data->x_part_strs[c-1]
						+strlen(pair_data->x_part_strs[c-1])+1;
					}

					pair_data->y_part_strs[0]=pair_data->x_part_strs[c-1]
					+strlen(pair_data->x_part_strs[c-1])+1;
					for(c=1;c<y_size;c++){
						pair_data->y_part_strs[c]=pair_data->y_part_strs[c-1]
						+strlen(pair_data->y_part_strs[c-1])+1;
					}

					pair_data->x_size=x_size;
					pair_data->y_size=y_size;
					if(info->maxX<x_size){
						info->maxX=x_size;
					}
					if(info->maxY<y_size){
						info->maxY=y_size;
					}
				}else{
					if(byte_num>1){
						fprintf(stderr,"WARNING:The format is wrong:%s:%d line",info->input_file,i);
					}
				}
				goto END_INPUT;
			default:
				if(info->input_sepchar==c){
					s[byte_num-1]='\0';
					if(state){
						y_size++;
					}else{
						x_size++;
					}
				}else{
					s[byte_num-1]=c;
				}
		}
	}

END_INPUT:
	fclose(fp);
	free(s);
	info->pair_data=top.next;
	if(info->restrictX==0){
		fprintf(stderr,"Set restrictX = %d\n",info->maxX);
		info->restrictX=info->maxX;
	}

	if(info->restrictY==0){
		fprintf(stderr,"Set restrictY = %d\n",info->maxY);
		info->restrictY=info->maxY;
	}
}

void readAlignFromFile(TOTAL_INFO *info){
	FILE *fp;
	size_t size=256,byte_num=0;
	int c,i=0,state=0;
	char *s;
	COP *p;

	fprintf(stderr,"Reading the align model file: %s\n",
		info->input_align_file);

	if ((fp = fopen(info->input_align_file, "r")) == NULL) {
		fprintf(stderr,"The align model file open error: %s\n",
			info->input_align_file);
		exit(EXIT_FAILURE);
	}

	if((s=(char *)malloc(size))==NULL){
		fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
		exit(EXIT_FAILURE);
	}

	while(1){
		if(byte_num>size){
			size+=size;
			if((s=(char *)realloc((void *)s,size))==NULL){
				fprintf(stderr,"Don't get memory in realloc.\nYou must need more memory.\n");
				exit(EXIT_FAILURE);
			}
		}
		c=getc(fp);
		byte_num++;
		switch(c){
			case '\t':
				s[byte_num-1]='\0';
				state++;
				break;
			case '\n':
				i++;
				if(byte_num>2 && s[byte_num-2]=='\r'){
					s[byte_num-2]='\0';
					byte_num--;
				}else{
					s[byte_num-1]='\0';
				}

				if(state==2 && s[byte_num-2]!='\0'){
					c=strlen(s)+1;
					p=COP_get(s, 1, &s[c], 1);
					p->para=(Parameter) atof(&s[c]+strlen(&s[c])+1);
				}else{
					if(byte_num>1){
						fprintf(stderr,"WARNING:The format is wrong:%s:%d line",info->input_align_file,i);
					}
				}
				state=0;
				byte_num=0;
				break;
			case EOF:
				i++;
				s[byte_num-1]='\0';
				if(state==2 && s[byte_num-2]!='\0'){
					c=strlen(s)+1;
					p=COP_get(s, 1, &s[c], 1);
					p->para=(Parameter) atof(&s[c]+strlen(&s[c])+1);
				}else{
					if(byte_num>1){
						fprintf(stderr,"WARNING:The format is wrong:%s:%d line",info->input_align_file,i);
					}
				}
				goto END_readAlignFromFile;
			default:
				s[byte_num-1]=c;
		}
	}

END_readAlignFromFile:
	fclose(fp);
	free(s);
}

void readPreviousKnowledge(TOTAL_INFO *info){
	FILE *fp;
	size_t size=256,byte_num=0;
	int c,i=0,state=0;
	char *s;
	COP *p;

	fprintf(stderr,"Reading the previous knowledge: %s\n",info->previous_knowledge_file);

	if ((fp = fopen(info->previous_knowledge_file, "r")) == NULL) {
		fprintf(stderr,"The previous knowledge file open error: %s\n",
			info->previous_knowledge_file);
		exit(EXIT_FAILURE);
	}

	if((s=(char *)malloc(size))==NULL){
		fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
		exit(EXIT_FAILURE);
	}

	while(1){
		if(byte_num>size){
			size+=size;
			if((s=(char *)realloc((void *)s,size))==NULL){
				fprintf(stderr,"Don't get memory in realloc.\nYou must need more memory.\n");
				exit(EXIT_FAILURE);
			}
		}
		c=getc(fp);
		byte_num++;
		switch(c){
			case '\t':
				s[byte_num-1]='\0';
				state++;
				break;
			case '\n':
				i++;
				if(byte_num>2 && s[byte_num-2]=='\r'){
					s[byte_num-2]='\0';
					byte_num--;
				}else{
					s[byte_num-1]='\0';
				}

				if(state && s[byte_num-2]!='\0'){
					if(state==1){
						p=COP_get(s, 1, &s[strlen(s)+1], 1);
						p->para=addlog10(p->para,0);
					}else{
						c=strlen(s)+1;
						p=COP_get(s, 1, &s[c], 1);
						p->para=addlog10(p->para, (Parameter) atof(&s[c]+strlen(&s[c])+1));
					}
				}else{
					if(byte_num>1){
						fprintf(stderr,"WARNING:The format is wrong:%s:%d line",info->previous_knowledge_file,i);
					}
				}
				state=0;
				byte_num=0;
				break;
			case EOF:
				i++;
				s[byte_num-1]='\0';
				if(state && s[byte_num-2]!='\0'){
					if(state==1){
						p=COP_get(s, 1, &s[strlen(s)+1], 1);
						p->para=addlog10(p->para,0);
					}else{
						c=strlen(s)+1;
						p=COP_get(s, 1, &s[c], 1);
						p->para=addlog10(p->para, (Parameter) atof(&s[c]+strlen(&s[c])+1));
					}
				}else{
					if(byte_num>1){
						fprintf(stderr,"WARNING:The format is wrong:%s:%d line",info->previous_knowledge_file,i);
					}
				}
				goto END_PreviousKnowledge;
			default:
				s[byte_num-1]=c;
		}
	}

END_PreviousKnowledge:
	fclose(fp);
	free(s);
}

static void forward(TOTAL_INFO *info, PAIR_DATA *pair_data, double **alpha, LEN x_size, LEN y_size){
	LEN xl,yl,i,j;
	double tmp;

	for(i=0;i<=x_size;i++){
		for(j=0;j<=y_size;j++){
			alpha[i][j]=LOW_LOG_VALUE;
		}
	}

	alpha[0][0]=0.0;
	for(xl = 0; xl <= x_size; xl++){
		for(yl = 0; yl <= y_size; yl++){
			for(i=1; (i <= info->restrictX) && (xl+i <= x_size); i++){
				for(j=1; (j <= info->restrictY) && (yl+j <= y_size); j++){
					tmp=alpha[xl][yl]+(i+j)
						*COP_refer_para(pair_data->x_part_strs[xl], i,
						pair_data->y_part_strs[yl], j);
					alpha[xl+i][yl+j]=addlog10(alpha[xl+i][yl+j], tmp);
				}
			}
		}
	}
}

static void backward(TOTAL_INFO *info, PAIR_DATA *pair_data, double **beta, LEN x_size, LEN y_size){
	LEN xl,yl,i,j;
	double tmp;

	for(i=0;i<=x_size;i++){
		for(j=0;j<=y_size;j++){
			beta[i][j]=LOW_LOG_VALUE;
		}
	}

	beta[x_size][y_size]=0.0;
	for(xl=x_size; xl >= 0; xl--){
		for(yl=y_size; yl >= 0 ; yl--){
			for(i=1; (i <= info->restrictX) && (xl-i >= 0); i++){
				for(j=1; (j <= info->restrictY) && (yl-j >= 0); j++){
					tmp=beta[xl][yl]+(i+j)
						*COP_refer_para(pair_data->x_part_strs[xl-i], i,
						pair_data->y_part_strs[yl-j], j);
					beta[xl-i][yl-j]=addlog10(beta[xl-i][yl-j], tmp);
				}
			}
		}
	}
}

static void expectation(TOTAL_INFO *info,  PAIR_DATA *pair_data, double **alpha, double **beta, LEN x_size, LEN y_size){
	LEN xl,yl,i,j;
	COP *p;
	Parameter update,alpha_xy;

	forward(info, pair_data, alpha, x_size, y_size);
	backward(info, pair_data, beta, x_size, y_size);

	alpha_xy=alpha[x_size][y_size];
	for(xl = 0; xl <= x_size; xl++){
		for(yl = 0; yl <= y_size; yl++){

			if(alpha[xl][yl] > LOW_LOG_VALUE){

				for(i=1; (i <= info->restrictX) && (xl+i <= x_size); i++){
					for(j=1; (j <= info->restrictY) && (yl+j <= y_size); j++){
						if(beta[xl+i][yl+j] > LOW_LOG_VALUE){
							p=COP_get(pair_data->x_part_strs[xl], i,
								pair_data->y_part_strs[yl], j);

							update=alpha[xl][yl]+beta[xl+i][yl+j]
									+(i+j)*p->para - alpha_xy;

							info->total_update = 
								addlog10(info->total_update, update);
							p->update_or_refer.update_para =
								addlog10(p->update_or_refer.update_para, update);
						}
					}
				}
			}
		}
	}
}

static void printSeparatedString(char *s, LEN len){
	while(len--){
		fprintf(stderr,"%s",s);
		s+=strlen(s)+1;
	}
	fprintf(stderr,"\n");
}

/* debug
static void printAlphaBeta(Parameter **a_or_b, LEN x_len, LEN y_len){
int i,j;
for(i=0;i<=x_len;i++){
for(j=0;j<=y_len;j++){
fprintf(stderr,"%4f ", a_or_b[i][j]);
}
fprintf(stderr,"\n");
}
}
*/

static void maximization(TOTAL_INFO *info){
	COP_HASH_VALUE i,j;
	COP *p;

	for(i=0;i<info->sqrt_hash_size;i++){
		for(j=0;j<info->sqrt_hash_size;j++){
			for(p=COP_TABLE[i][j];p!=NULL;p=p->next){
				p->update_or_refer.update_para-=info->total_update;
				if(p->para > p->update_or_refer.update_para){
					info->total_change += pow(10, p->para)-pow(10, p->update_or_refer.update_para);
				}else{
					info->total_change += pow(10, p->update_or_refer.update_para)-pow(10, p->para);
				}
				p->para=p->update_or_refer.update_para;
				p->update_or_refer.update_para=LOW_LOG_VALUE;
			}
		}
	}
}

void training(TOTAL_INFO *info){
	int i,j;
	PAIR_DATA *pair_data;
	double **alpha;
	double **beta;

	// get memory for Alpha and Beta
	if((alpha=(double **)malloc(2*((info->maxX+1)*(sizeof(double *)
		+(info->maxY+1)*sizeof(double)))))==NULL){
			fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
			exit(EXIT_FAILURE);
	}
	for(i=0;i<=info->maxX;i++){
		alpha[i]=((double *)(alpha+info->maxX+1))+i*(info->maxY+1);
	}

	beta=(double **)((double *)(alpha+info->maxX+1)+i*(info->maxY+1));
	for(i=0;i<=info->maxX;i++){
		beta[i]=((double *)(beta+info->maxX+1))+i*(info->maxY+1);
	}

	i=0;
	while(1){
		i++;
		fprintf(stderr,"Iteration:%d\n",i);
		info->total_update=LOW_LOG_VALUE;
		j=1;
		for(pair_data=info->pair_data;pair_data!=NULL;pair_data=pair_data->next){
			if((j%1000)==0){
				fprintf(stderr,"Expectation:%d\r",j);
			}
			j++;
			expectation(info, pair_data, alpha, beta,
				pair_data->x_size, pair_data->y_size);

			/* debug

			fprintf(stderr,"cop size:%d\nx: ",num_of_cop);
			printSeparatedString(pair_data->x_part_strs[0], pair_data->x_size);

			fprintf(stderr,"y: ");
			printSeparatedString(pair_data->y_part_strs[0], pair_data->y_size);

			fprintf(stderr,"alpha: \n");
			printAlphaBeta(alpha, pair_data->x_size, pair_data->y_size);

			fprintf(stderr,"beta: \n");
			printAlphaBeta(beta, pair_data->x_size, pair_data->y_size);
			*/
		}
		fprintf(stderr,"Expectation:%d\n",j-1);
		fprintf(stderr,"Maximization\n");
		info->total_change=0;
		maximization(info);
		fprintf(stderr,"change prob:%f\n\n",info->total_change);
		if( ( 1 <= info->threshold_eot && info->threshold_eot <= i ) 
			|| ( info->threshold_eot < 1 && info->total_change <= info->threshold_eot)){
				break;
		}
	}
	free(alpha);
}

void writeAlignToFile(TOTAL_INFO *info){
	FILE *fp;
	COP_HASH_VALUE i,j;
	COP *p;

	fprintf(stderr,"Writting the align model file: %s\n", info->output_align_file);

	if ((fp = fopen(info->output_align_file, "w")) == NULL) {
		fprintf(stderr,"The align model file open error: %s\n",info->output_align_file);
		exit(EXIT_FAILURE);
	}

	for(i=0;i<info->sqrt_hash_size;i++){
		for(j=0;j<info->sqrt_hash_size;j++){
			for(p=COP_TABLE[i][j];p!=NULL;p=p->next){
				fprintf(fp,"%s\t%s\t%f\n",p->x,p->y,p->para);
			}
		}
	}

	fclose(fp);
}

static void nbest_alignment(TOTAL_INFO *info, PAIR_DATA *pair_data, VITERBI_HYP_LIST **viterbi_table, ALIGN_HYP *align_hyps){
	float inv_total_len;
	LEN xl,yl,i,j,k,rank_index;
	VITERBI_HYP_LIST *vhl,*nvhl,*last_nvhl;
	VITERBI_HYP vh={ -1, //backX
		-1, //backY
		NULL, //prev
		0,  //num_of_trans
		0}; //score
	COP *cop;

	for(xl=0;xl<=pair_data->x_size; xl++){
		for(yl = 0;yl<=pair_data->y_size; yl++){
			viterbi_table[xl][yl].inv_total_len=0;
			viterbi_table[xl][yl].rank_size=0;
			viterbi_table[xl][yl].next=NULL;
		}
	}

	viterbi_table[0][0].inv_total_len=1.0/(pair_data->x_size+pair_data->y_size);
	viterbi_table[0][0].rank_size=1;
	viterbi_table[0][0].viterbi_hyp[0]=vh;
	last_nvhl=&(viterbi_table[pair_data->x_size][pair_data->y_size]);

	for(xl=0;xl<=pair_data->x_size; xl++){
		for(yl = 0;yl<=pair_data->y_size; yl++){
			vhl=&(viterbi_table[xl][yl]);
			vh.backX=xl;
			vh.backY=yl;
			do{
				if(vhl->rank_size==0){
					vhl=vhl->next;
					continue;
				}

				if(info->delX){
					for(i=1; (i <= info->restrictX) && (xl+i <= pair_data->x_size);i++){
						for(rank_index=0;rank_index < vhl->rank_size; rank_index++){
							if((vhl->viterbi_hyp[rank_index].backY == yl) 
								|| ((inv_total_len=(1.0/vhl->inv_total_len)
								-(i+info->del_penalty)) <= 0)){
								continue;
							}

							vh.prev = &(vhl->viterbi_hyp[rank_index]);
							vh.score=(vh.prev->score*(inv_total_len+i+info->del_penalty))/(inv_total_len);
							inv_total_len=1.0/inv_total_len;
							vh.num_of_trans=vh.prev->num_of_trans+1;

							nvhl=&(viterbi_table[xl+i][yl]);
							if(nvhl!=last_nvhl){
								if(nvhl->inv_total_len==0){
									nvhl->inv_total_len=inv_total_len;
								}else if(nvhl->inv_total_len!=inv_total_len){
									for(;(nvhl->next!=NULL 
										&& nvhl->next->inv_total_len!=inv_total_len)
										;nvhl=nvhl->next);

									if(nvhl->next==NULL){
										if((nvhl->next =
											(VITERBI_HYP_LIST *)
											malloc(sizeof(VITERBI_HYP_LIST)
											+info->first_n_best*sizeof(VITERBI_HYP)))
											==NULL){
												fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
												exit(EXIT_FAILURE);
										}
										nvhl=nvhl->next;
										nvhl->viterbi_hyp=(VITERBI_HYP *)(nvhl+1);
										nvhl->inv_total_len=inv_total_len;
										nvhl->next=NULL;
										nvhl->rank_size=0;
									}else{
										nvhl=nvhl->next;
									}
								}
							}

							if((k=nvhl->rank_size)==info->first_n_best){
								if(CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									k--;
									while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
										nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
										k--;
									}
									nvhl->viterbi_hyp[k]=vh;
								}else{
									break;
								}
							}else{
								while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
									k--;
								}
								nvhl->viterbi_hyp[k]=vh;
								nvhl->rank_size++;
							}
						}
					}
				}
				if(info->delY){
					for(j=1; (j <= info->restrictY) && (yl+j <= pair_data->y_size); j++){
						for(rank_index=0;rank_index < vhl->rank_size; rank_index++){
							if((vhl->viterbi_hyp[rank_index].backX == xl) 
								|| ((inv_total_len=(1.0/vhl->inv_total_len)
								-(j+info->del_penalty)) <= 0)){
									continue;
							}

							vh.prev = &(vhl->viterbi_hyp[rank_index]);
							vh.score=(vh.prev->score*(inv_total_len+j+info->del_penalty))/(inv_total_len);
							inv_total_len=1.0/inv_total_len;
							vh.num_of_trans=vh.prev->num_of_trans+1;

							nvhl=&(viterbi_table[xl][yl+j]);
							if(nvhl!=last_nvhl){
								if(nvhl->inv_total_len==0){
									nvhl->inv_total_len=inv_total_len;
								}else if(nvhl->inv_total_len!=inv_total_len){
									for(;(nvhl->next!=NULL 
										&& nvhl->next->inv_total_len!=inv_total_len)
										;nvhl=nvhl->next);

									if(nvhl->next==NULL){
										if((nvhl->next =
											(VITERBI_HYP_LIST *)
											malloc(sizeof(VITERBI_HYP_LIST)
											+info->first_n_best*sizeof(VITERBI_HYP)))
											==NULL){
												fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
												exit(EXIT_FAILURE);
										}
										nvhl=nvhl->next;
										nvhl->viterbi_hyp=(VITERBI_HYP *)(nvhl+1);
										nvhl->inv_total_len=inv_total_len;
										nvhl->next=NULL;
										nvhl->rank_size=0;
									}else{
										nvhl=nvhl->next;
									}
								}
							}

							if((k=nvhl->rank_size)==info->first_n_best){
								if(CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									k--;
									while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
										nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
										k--;
									}
									nvhl->viterbi_hyp[k]=vh;
								}else{
									break;
								}
							}else{
								while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
									k--;
								}
								nvhl->viterbi_hyp[k]=vh;
								nvhl->rank_size++;
							}
						}
					}
				}

				inv_total_len=vhl->inv_total_len;
				for(i=1; (i <= info->restrictX) && (xl+i <= pair_data->x_size); i++){
					for(j=1; (j <= info->restrictY) && (yl+j <= pair_data->y_size); j++){
						//pair_data->x_part_strs[xl];
						//pair_data->y_part_strs[yl];

						cop=COP_refer(pair_data->x_part_strs[xl], i, pair_data->y_part_strs[yl], j);
						if(cop==NULL){
							continue;
						}
						for(rank_index=0;rank_index<vhl->rank_size;rank_index++){
							vh.prev = &(vhl->viterbi_hyp[rank_index]);
							vh.score=(i+j)*inv_total_len*cop->para+vh.prev->score;
							vh.num_of_trans=vh.prev->num_of_trans+1;

							nvhl=&(viterbi_table[xl+i][yl+j]);
							if(nvhl!=last_nvhl){
								if(nvhl->inv_total_len==0){
									nvhl->inv_total_len=inv_total_len;
								}else if(nvhl->inv_total_len!=inv_total_len){
									for(;(nvhl->next!=NULL
										&& nvhl->next->inv_total_len!=inv_total_len)
										;nvhl=nvhl->next);

									if(nvhl->next==NULL){
										if((nvhl->next =
											(VITERBI_HYP_LIST *)
											malloc(sizeof(VITERBI_HYP_LIST)
											+info->first_n_best*sizeof(VITERBI_HYP)))
											==NULL){
												fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
												exit(EXIT_FAILURE);
										}
										nvhl=nvhl->next;
										nvhl->viterbi_hyp=(VITERBI_HYP *)(nvhl+1);
										nvhl->inv_total_len=inv_total_len;
										nvhl->next=NULL;
										nvhl->rank_size=0;
									}else{
										nvhl=nvhl->next;
									}
								}
							}
							if((k=nvhl->rank_size)==info->first_n_best){
								if(CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									k--;
									while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
										nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
										k--;
									}
									nvhl->viterbi_hyp[k]=vh;
								}else{
									break;
								}
							}else{
								while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
									k--;
								}
								nvhl->viterbi_hyp[k]=vh;
								nvhl->rank_size++;
							}
						}
					}
				}
				vhl=vhl->next;
			}while(vhl!=NULL);
		}
	}

	//backTracking
	for(i=0; i < last_nvhl->rank_size ; i++){
		xl=pair_data->x_size;
		yl=pair_data->y_size;
		vh=last_nvhl->viterbi_hyp[i];
		if(!info->double_alignment){
			align_hyps[i].score=last_nvhl->viterbi_hyp[i].score;
		}
		while((xl > 0) || (yl > 0)){
			if(info->double_alignment){
				if(vh.backX != xl && vh.backY != yl){
					cop=COP_refer(pair_data->x_part_strs[vh.backX], xl-vh.backX,
						pair_data->y_part_strs[vh.backY], yl-vh.backY);
					cop->update_or_refer.refer_count++;
					if(info->lowest_para > cop->para){	
						info->lowest_para=cop->para-1;
					}
				}
			}else{
				if(vh.backX == xl){
					cop=COP_get(info->del_char, 1,
						pair_data->y_part_strs[vh.backY], yl-vh.backY);
				}else if(vh.backY == yl){
					cop=COP_get(pair_data->x_part_strs[vh.backX], xl-vh.backX,
						info->del_char, 1);
				}else{
					cop=COP_refer(pair_data->x_part_strs[vh.backX], xl-vh.backX,
						pair_data->y_part_strs[vh.backY], yl-vh.backY);
				}

				align_hyps[i].cop[vh.num_of_trans-1]=cop;
			}

			xl = vh.backX;
			yl = vh.backY;
			vh=*(vh.prev);
		}
	}

	
	for(xl=0;xl<=pair_data->x_size; xl++){
		for(yl = 0;yl<=pair_data->y_size; yl++){
			for(vhl=&(viterbi_table[xl][yl]);vhl->next!=NULL;){
				nvhl=vhl->next;
				vhl->next=nvhl->next;
				free(nvhl);
			}
		}
	}
}

static void nbest_double_alignment(TOTAL_INFO *info, PAIR_DATA *pair_data, VITERBI_HYP_LIST **viterbi_table,
								 ALIGN_HYP *align_hyps,char max_nbest){
	float inv_total_len;
	char first_rank_size;
	LEN xl,yl,i,j,k,rank_index;
	VITERBI_HYP_LIST *vhl,*nvhl,*last_nvhl;
	VITERBI_HYP vh={ -1, //backX
		-1, //backY
		NULL, //prev
		0,  //num_of_trans
		0}; //score
	COP *cop;

	for(xl=0;xl<=pair_data->x_size; xl++){
		for(yl = 0;yl<=pair_data->y_size; yl++){
			viterbi_table[xl][yl].inv_total_len=0;
			viterbi_table[xl][yl].rank_size=0;
			viterbi_table[xl][yl].next=NULL;
		}
	}

	viterbi_table[0][0].inv_total_len=1.0/(pair_data->x_size+pair_data->y_size);
	viterbi_table[0][0].rank_size=1;
	viterbi_table[0][0].viterbi_hyp[0]=vh;
	last_nvhl=&(viterbi_table[pair_data->x_size][pair_data->y_size]);

	for(xl=0;xl<=pair_data->x_size; xl++){
		for(yl = 0;yl<=pair_data->y_size; yl++){
			vhl=&(viterbi_table[xl][yl]);
			vh.backX=xl;
			vh.backY=yl;

			do{
				if(vhl->rank_size==0){
					vhl=vhl->next;
					continue;
				}


				if(info->delX){
					for(i=1; (i <= info->restrictX) && (xl+i <= pair_data->x_size);i++){
						for(rank_index=0;rank_index < vhl->rank_size; rank_index++){
							if((vhl->viterbi_hyp[rank_index].backY == yl) 
								|| ((inv_total_len=(1.0/vhl->inv_total_len)
								-(i+info->del_penalty)) <= 0)){
								continue;
							}

							vh.prev = &(vhl->viterbi_hyp[rank_index]);
							vh.score=(vh.prev->score*(inv_total_len+i+info->del_penalty))/(inv_total_len);
							inv_total_len=1.0/inv_total_len;
							vh.num_of_trans=vh.prev->num_of_trans+1;

							nvhl=&(viterbi_table[xl+i][yl]);
							if(nvhl!=last_nvhl){
								if(nvhl->inv_total_len==0){
									nvhl->inv_total_len=inv_total_len;
								}else if(nvhl->inv_total_len!=inv_total_len){
									for(;(nvhl->next!=NULL 
										&& nvhl->next->inv_total_len!=inv_total_len)
										;nvhl=nvhl->next);

									if(nvhl->next==NULL){
										if((nvhl->next =
											(VITERBI_HYP_LIST *)
											malloc(sizeof(VITERBI_HYP_LIST)
											+max_nbest*sizeof(VITERBI_HYP)))
											==NULL){
												fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
												exit(EXIT_FAILURE);
										}
										nvhl=nvhl->next;
										nvhl->viterbi_hyp=(VITERBI_HYP *)(nvhl+1);
										nvhl->inv_total_len=inv_total_len;
										nvhl->next=NULL;
										nvhl->rank_size=0;
									}else{
										nvhl=nvhl->next;
									}
								}
							}

							if((k=nvhl->rank_size)==info->first_n_best){
								if(CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									k--;
									while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
										nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
										k--;
									}
									nvhl->viterbi_hyp[k]=vh;
								}else{
									break;
								}
							}else{
								while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
									k--;
								}
								nvhl->viterbi_hyp[k]=vh;
								nvhl->rank_size++;
							}
						}
					}
				}

				if(info->delY){
					for(j=1; (j <= info->restrictY) && (yl+j <= pair_data->y_size); j++){
						for(rank_index=0;rank_index < vhl->rank_size; rank_index++){
							if((vhl->viterbi_hyp[rank_index].backX == xl) 
								|| ((inv_total_len=(1.0/vhl->inv_total_len)
								-(j+info->del_penalty)) <= 0)){
									continue;
							}

							vh.prev = &(vhl->viterbi_hyp[rank_index]);
							vh.score=(vh.prev->score*(inv_total_len+j+info->del_penalty))/(inv_total_len);
							inv_total_len=1.0/inv_total_len;
							vh.num_of_trans=vh.prev->num_of_trans+1;

							nvhl=&(viterbi_table[xl][yl+j]);
							if(nvhl!=last_nvhl){
								if(nvhl->inv_total_len==0){
									nvhl->inv_total_len=inv_total_len;
								}else if(nvhl->inv_total_len!=inv_total_len){
									for(;(nvhl->next!=NULL 
										&& nvhl->next->inv_total_len!=inv_total_len)
										;nvhl=nvhl->next);

									if(nvhl->next==NULL){
										if((nvhl->next =
											(VITERBI_HYP_LIST *)
											malloc(sizeof(VITERBI_HYP_LIST)
											+max_nbest*sizeof(VITERBI_HYP)))
											==NULL){
												fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
												exit(EXIT_FAILURE);
										}
										nvhl=nvhl->next;
										nvhl->viterbi_hyp=(VITERBI_HYP *)(nvhl+1);
										nvhl->inv_total_len=inv_total_len;
										nvhl->next=NULL;
										nvhl->rank_size=0;
									}else{
										nvhl=nvhl->next;
									}
								}
							}

							if((k=nvhl->rank_size)==info->first_n_best){
								if(CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									k--;
									while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
										nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
										k--;
									}
									nvhl->viterbi_hyp[k]=vh;
								}else{
									break;
								}
							}else{
								while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
									k--;
								}
								nvhl->viterbi_hyp[k]=vh;
								nvhl->rank_size++;
							}
						}
					}
				}

				inv_total_len=vhl->inv_total_len;
				for(i=1; (i <= info->restrictX) && (xl+i <= pair_data->x_size); i++){
					for(j=1; (j <= info->restrictY) && (yl+j <= pair_data->y_size); j++){
						cop=COP_refer(pair_data->x_part_strs[xl], i, pair_data->y_part_strs[yl], j);
						if(cop==NULL){
							continue;
						}
						for(rank_index=0;rank_index<vhl->rank_size;rank_index++){
							vh.prev = &(vhl->viterbi_hyp[rank_index]);
							vh.score=(i+j)*inv_total_len*cop->para+vh.prev->score;
							vh.num_of_trans=vh.prev->num_of_trans+1;

							nvhl=&(viterbi_table[xl+i][yl+j]);
							if(nvhl!=last_nvhl){
								if(nvhl->inv_total_len==0){
									nvhl->inv_total_len=inv_total_len;
								}else if(nvhl->inv_total_len!=inv_total_len){
									for(;(nvhl->next!=NULL
										&& nvhl->next->inv_total_len!=inv_total_len)
										;nvhl=nvhl->next);

									if(nvhl->next==NULL){
										if((nvhl->next =
											(VITERBI_HYP_LIST *)
											malloc(sizeof(VITERBI_HYP_LIST)
											+max_nbest*sizeof(VITERBI_HYP)))
											==NULL){
												fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
												exit(EXIT_FAILURE);
										}
										nvhl=nvhl->next;
										nvhl->viterbi_hyp=(VITERBI_HYP *)(nvhl+1);
										nvhl->inv_total_len=inv_total_len;
										nvhl->next=NULL;
										nvhl->rank_size=0;
									}else{
										nvhl=nvhl->next;
									}
								}
							}

							if((k=nvhl->rank_size)==info->first_n_best){
								if(CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									k--;
									while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
										nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
										k--;
									}
									nvhl->viterbi_hyp[k]=vh;
								}else{
									break;
								}
							}else{
								while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
									k--;
								}
								nvhl->viterbi_hyp[k]=vh;
								nvhl->rank_size++;
							}
						}
					}
				}

				vhl=vhl->next;
			}while(vhl!=NULL);
		}
	}

	//backTracking
	first_rank_size=last_nvhl->rank_size;
	for(i=0; i < first_rank_size ; i++){
		xl=pair_data->x_size;
		yl=pair_data->y_size;
		vh=last_nvhl->viterbi_hyp[i];
		k=vh.num_of_trans;
		if(vh.num_of_trans < info->maxLEN){
			align_hyps[i].cop[vh.num_of_trans]=NULL;
		}

		while((xl > 0) || (yl > 0)){

			if(vh.backX != xl && vh.backY != yl){
				cop=COP_refer(pair_data->x_part_strs[vh.backX], xl-vh.backX,
					pair_data->y_part_strs[vh.backY], yl-vh.backY);
				align_hyps[i].cop[vh.num_of_trans-1]=cop;
				cop->update_or_refer.refer_count--;
			}else{
				for(j=vh.num_of_trans;j<k;j++){
					align_hyps[i].cop[j-1]=align_hyps[i].cop[j];
				}
				align_hyps[i].cop[j-1]=NULL;
			}

			xl = vh.backX;
			yl = vh.backY;
			vh=*(vh.prev);
		}
	}
	
	for(xl=0;xl<=pair_data->x_size; xl++){
		for(yl = 0;yl<=pair_data->y_size; yl++){
			viterbi_table[xl][yl].rank_size=0;
			for(vhl=viterbi_table[xl][yl].next;vhl!=NULL;vhl=vhl->next){
				vhl->rank_size=0;
			}
		}
	}

	viterbi_table[0][0].rank_size=1;

	for(xl=0;xl<=pair_data->x_size; xl++){
		for(yl = 0;yl<=pair_data->y_size; yl++){
			vhl=&(viterbi_table[xl][yl]);
			vh.backX=xl;
			vh.backY=yl;

			do{
				if(vhl->rank_size==0){
					vhl=vhl->next;
					continue;
				}

				if(info->delX){
					for(i=1; (i <= info->restrictX) && (xl+i <= pair_data->x_size);i++){
						for(rank_index=0;rank_index < vhl->rank_size; rank_index++){
							if((vhl->viterbi_hyp[rank_index].backY == yl) 
								|| ((inv_total_len=(1.0/vhl->inv_total_len)
								-(i+info->del_penalty)) <= 0)){
								continue;
							}

							vh.prev = &(vhl->viterbi_hyp[rank_index]);
							vh.score=(vh.prev->score*(inv_total_len+i+info->del_penalty))/(inv_total_len);
							inv_total_len=1.0/inv_total_len;
							vh.num_of_trans=vh.prev->num_of_trans+1;

							nvhl=&(viterbi_table[xl+i][yl]);
							if(nvhl!=last_nvhl){
								if(nvhl->inv_total_len!=inv_total_len){
									for(;(nvhl->next!=NULL 
										&& nvhl->next->inv_total_len!=inv_total_len)
										;nvhl=nvhl->next);

									if(nvhl->next==NULL){
										if((nvhl->next =
											(VITERBI_HYP_LIST *)
											malloc(sizeof(VITERBI_HYP_LIST)
											+info->second_n_best*sizeof(VITERBI_HYP)))
											==NULL){
												fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
												exit(EXIT_FAILURE);

										}
										nvhl=nvhl->next;
										nvhl->viterbi_hyp=(VITERBI_HYP *)(nvhl+1);
										nvhl->inv_total_len=inv_total_len;
										nvhl->next=NULL;
										nvhl->rank_size=0;
									}else{
										nvhl=nvhl->next;
									}
								}
							}

							if((k=nvhl->rank_size)==info->second_n_best){
								if(CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									k--;
									while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
										nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
										k--;
									}
									nvhl->viterbi_hyp[k]=vh;
								}else{
									break;
								}
							}else{
								while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
									k--;
								}
								nvhl->viterbi_hyp[k]=vh;
								nvhl->rank_size++;
							}
						}
					}
				}

				if(info->delY){
					for(j=1; (j <= info->restrictY) && (yl+j <= pair_data->y_size); j++){
						for(rank_index=0;rank_index < vhl->rank_size; rank_index++){
							if((vhl->viterbi_hyp[rank_index].backX == xl) 
								|| ((inv_total_len=(1.0/vhl->inv_total_len)
								-(j+info->del_penalty)) <= 0)){
									continue;
							}

							vh.prev = &(vhl->viterbi_hyp[rank_index]);
							vh.score=(vh.prev->score*(inv_total_len+j+info->del_penalty))/(inv_total_len);
							inv_total_len=1.0/inv_total_len;
							vh.num_of_trans=vh.prev->num_of_trans+1;

							nvhl=&(viterbi_table[xl][yl+j]);
							if(nvhl!=last_nvhl){
								if(nvhl->inv_total_len!=inv_total_len){
									for(;(nvhl->next!=NULL 
										&& nvhl->next->inv_total_len!=inv_total_len)
										;nvhl=nvhl->next);

									if(nvhl->next==NULL){
										if((nvhl->next =
											(VITERBI_HYP_LIST *)
											malloc(sizeof(VITERBI_HYP_LIST)
											+info->second_n_best*sizeof(VITERBI_HYP)))
											==NULL){
												fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
												exit(EXIT_FAILURE);
										}
										nvhl=nvhl->next;
										nvhl->viterbi_hyp=(VITERBI_HYP *)(nvhl+1);
										nvhl->inv_total_len=inv_total_len;
										nvhl->next=NULL;
										nvhl->rank_size=0;
									}else{
										nvhl=nvhl->next;
									}
								}
							}

							if((k=nvhl->rank_size)==info->second_n_best){
								if(CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									k--;
									while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
										nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
										k--;
									}
									nvhl->viterbi_hyp[k]=vh;
								}else{
									break;
								}
							}else{
								while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
									k--;
								}
								nvhl->viterbi_hyp[k]=vh;
								nvhl->rank_size++;
							}
						}
					}
				}

				inv_total_len=vhl->inv_total_len;
				for(i=1; (i <= info->restrictX) && (xl+i <= pair_data->x_size); i++){
					for(j=1; (j <= info->restrictY) && (yl+j <= pair_data->y_size); j++){
						cop=COP_refer(pair_data->x_part_strs[xl], i, pair_data->y_part_strs[yl], j);
						if(cop==NULL){
							continue;
						}
						for(rank_index=0;rank_index<vhl->rank_size;rank_index++){
							vh.prev = &(vhl->viterbi_hyp[rank_index]);
							if(cop->update_or_refer.refer_count>0){
								vh.score=(i+j)*inv_total_len*cop->para
											+vh.prev->score;
							}else{
								vh.score=(i+j)*inv_total_len*info->lowest_para
												+vh.prev->score;
							}

							vh.num_of_trans=vh.prev->num_of_trans+1;

							nvhl=&(viterbi_table[xl+i][yl+j]);
							if(nvhl!=last_nvhl){
								if(nvhl->inv_total_len!=inv_total_len){
									for(;(nvhl->next!=NULL
										&& nvhl->next->inv_total_len!=inv_total_len)
										;nvhl=nvhl->next);

									if(nvhl->next==NULL){
										if((nvhl->next =
											(VITERBI_HYP_LIST *)
											malloc(sizeof(VITERBI_HYP_LIST)
											+info->second_n_best*sizeof(VITERBI_HYP)))
											==NULL){
												fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
												exit(EXIT_FAILURE);
										}
										nvhl=nvhl->next;
										nvhl->viterbi_hyp=(VITERBI_HYP *)(nvhl+1);
										nvhl->inv_total_len=inv_total_len;
										nvhl->next=NULL;
										nvhl->rank_size=0;
									}else{
										nvhl=nvhl->next;
									}
								}
							}

							if((k=nvhl->rank_size)==info->second_n_best){
								if(CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									k--;
									while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
										nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
										k--;
									}
									nvhl->viterbi_hyp[k]=vh;
								}else{
									break;
								}
							}else{
								while( k && CMP_VITERBI_HYP(vh, nvhl->viterbi_hyp[k-1])){
									nvhl->viterbi_hyp[k]=nvhl->viterbi_hyp[k-1];
									k--;
								}
								nvhl->viterbi_hyp[k]=vh;
								nvhl->rank_size++;
							}
						}
					}
				}

				vhl=vhl->next;
			}while(vhl!=NULL);
		}
	}


	for(i=0; i < first_rank_size ; i++){
		for(j=0;j<info->maxLEN && align_hyps[i].cop[j]!=NULL;j++){
			align_hyps[i].cop[j]->update_or_refer.refer_count++;
		}
	}


	//backTracking
	for(i=0; i < last_nvhl->rank_size ; i++){
		xl=pair_data->x_size;
		yl=pair_data->y_size;
		vh=last_nvhl->viterbi_hyp[i];
		align_hyps[i].score=last_nvhl->viterbi_hyp[i].score;

		while((xl > 0) || (yl > 0)){

			if(vh.backX == xl){
				cop=COP_get(info->del_char, 1,
					pair_data->y_part_strs[vh.backY], yl-vh.backY);
			}else if(vh.backY == yl){
				cop=COP_get(pair_data->x_part_strs[vh.backX], xl-vh.backX,
					info->del_char, 1);
			}else{
				cop=COP_refer(pair_data->x_part_strs[vh.backX], xl-vh.backX,
					pair_data->y_part_strs[vh.backY], yl-vh.backY);
			}

			align_hyps[i].cop[vh.num_of_trans-1]=cop;
			xl = vh.backX;
			yl = vh.backY;
			vh=*(vh.prev);
		}
	}

	for(xl=0;xl<=pair_data->x_size; xl++){
		for(yl = 0;yl<=pair_data->y_size; yl++){
			for(vhl=&(viterbi_table[xl][yl]);vhl->next!=NULL;){
				nvhl=vhl->next;
				vhl->next=nvhl->next;
				free(nvhl);
			}
		}
	}
}

void requireAlignments(TOTAL_INFO *info){
	FILE *fp;
	LEN i,j,k,max_trans;
	char n_best,max_nbest;
	char *align_p,*unalign_p,*start_unalign_p;
	int AlignCount=0,noAlignCount=0; //,lessNbest=0;
	VITERBI_HYP_LIST *vhl, **viterbi_table;
	ALIGN_HYP *align_hyps;
	PAIR_DATA *pair_data;

	fprintf(stderr,"Write alignments : %s\n", info->output_file);

	if((fp=fopen(info->output_file, "w")) == NULL){
		fprintf(stderr,"The output file open error: %s\n",info->output_file);
		exit(EXIT_FAILURE);
	}

	// get memory for Viterbi Algorithm
	info->maxLEN=(info->maxX > info->maxY)? info->maxX : info->maxY;

	max_nbest=(info->first_n_best > info->second_n_best)?
						info->first_n_best :info->second_n_best;

	if((viterbi_table=(VITERBI_HYP_LIST **)malloc((info->maxX+1)
		*(sizeof(VITERBI_HYP_LIST *)
		+(info->maxY+1)*(sizeof(VITERBI_HYP_LIST)
		+max_nbest*sizeof(VITERBI_HYP)))
		+max_nbest*(sizeof(ALIGN_HYP)
		+info->maxLEN*sizeof(COP *))))==NULL){
			fprintf(stderr,"Don't get memory in malloc.\nYou must need more memory.\n");
			exit(EXIT_FAILURE);
	}

	for(i=0;i<=info->maxX;i++){
		viterbi_table[i]=
			(VITERBI_HYP_LIST *)((VITERBI_HYP*)(((VITERBI_HYP_LIST *)(viterbi_table+info->maxX+1))
			+i*(info->maxY+1))+i*(info->maxY+1)*max_nbest);
		for(j=0;j<=info->maxY;j++){
			viterbi_table[i][j].viterbi_hyp=
				(VITERBI_HYP *)(viterbi_table[i]+info->maxY+1)
				+j*max_nbest;
		}
	}

	align_hyps=(ALIGN_HYP *)((VITERBI_HYP*)(((VITERBI_HYP_LIST *)
		(viterbi_table+info->maxX+1))+i*(info->maxY+1))+i*(info->maxY+1)*max_nbest);
	for(i=0;i<max_nbest;i++){
		align_hyps[i].cop=(COP **)(align_hyps+max_nbest)+i*info->maxLEN;
	}

	if(info->double_alignment){
		COP_init_refer_count();
	}

	fprintf(stderr,"Start first alignments.\n");
	for(pair_data=info->pair_data;pair_data!=NULL;pair_data=pair_data->next){

		nbest_alignment(info, pair_data, viterbi_table, align_hyps);
		vhl=&(viterbi_table[pair_data->x_size][pair_data->y_size]);
		n_best=(info->first_n_best < vhl->rank_size)?
			info->first_n_best : vhl->rank_size;
		if(n_best>0){
			AlignCount++;

			if(!info->double_alignment){
				for(i=0;i<n_best;i++){
					max_trans=vhl->viterbi_hyp[i].num_of_trans;
					
					start_unalign_p=unalign_p=pair_data->x_part_strs[0];
					for(j=0;j<max_trans;j++){
						align_p=align_hyps[i].cop[j]->x;
						if(align_p[0]==info->del_char[0] && align_p[1]=='\0'){
							fprintf(fp,"%s%c", info->del_char, info->output_sepchar);
						}else{
							while(*align_p!='\0'){
                                                                if(*align_p!=*unalign_p){
                                                                        *unalign_p=info->output_joinchar;
								}
								align_p++;
                                                                unalign_p++;
							}
							*unalign_p='\0';
							fprintf(fp,"%s%c", start_unalign_p, info->output_sepchar);
							unalign_p++;
							start_unalign_p=unalign_p;
						}						
					}
					fprintf(fp,"\t");
                                        start_unalign_p=unalign_p=pair_data->y_part_strs[0];
                                        for(j=0;j<max_trans;j++){
                                                align_p=align_hyps[i].cop[j]->y;
                                                if(align_p[0]==info->del_char[0] && align_p[1]=='\0'){
                                                        fprintf(fp,"%s%c", info->del_char, info->output_sepchar);
                                                }else{
                                                        while(*align_p!='\0'){
                                                                if(*align_p!=*unalign_p){
                                                                        *unalign_p=info->output_joinchar;
								}
								align_p++;
                                                                unalign_p++;
                                                        }
							*unalign_p='\0';
                                                        fprintf(fp,"%s%c", start_unalign_p, info->output_sepchar);
							unalign_p++;
                                                        start_unalign_p=unalign_p;
                                                }
                                        }
					if (info->printScore){
						fprintf(fp,"\t%d-best\t%f\n", i+1, align_hyps[i].score);
					}else{
						fprintf(fp,"\n");
					}
				}
			}
			if((AlignCount%1000)==0){
				fprintf(stderr,"Aligned:%d\r",AlignCount);
			}
		}else{
			noAlignCount++;
			fprintf(stderr,"\nNo Align Pair:%d\n",noAlignCount);
			fprintf(stderr,"x: ");
			printSeparatedString(pair_data->x_part_strs[0], pair_data->x_size);
			fprintf(stderr,"y: ");
			printSeparatedString(pair_data->y_part_strs[0], pair_data->y_size);
		}
	}

	fprintf(stderr,"End first alignments.\nAlignments:%d, No alignments:%d\n",AlignCount, noAlignCount);

	AlignCount=noAlignCount=0;
	if(info->double_alignment){
		fprintf(stderr,"Start second alignments.\n");
		for(pair_data=info->pair_data;pair_data!=NULL;pair_data=pair_data->next){
			nbest_double_alignment(info, pair_data, viterbi_table,
							align_hyps, max_nbest);

			vhl=&(viterbi_table[pair_data->x_size][pair_data->y_size]);
			n_best=(info->second_n_best < vhl->rank_size)?
						info->second_n_best : vhl->rank_size;
			if(n_best>0){
				AlignCount++;

				for(i=0;i<n_best;i++){
					max_trans=vhl->viterbi_hyp[i].num_of_trans;

                                        start_unalign_p=unalign_p=pair_data->x_part_strs[0];
                                        for(j=0;j<max_trans;j++){
                                                align_p=align_hyps[i].cop[j]->x;
                                                if(align_p[0]==info->del_char[0] && align_p[1]=='\0'){
                                                        fprintf(fp,"%s%c", info->del_char, info->output_sepchar);
                                                }else{
                                                        while(*align_p!='\0'){
                                                                if(*align_p!=*unalign_p){
                                                                        *unalign_p=info->output_joinchar;
								}
								align_p++;
                                                                unalign_p++;
                                                        }
							*unalign_p='\0';
                                                        fprintf(fp,"%s%c", start_unalign_p, info->output_sepchar);
							if (info->printScore){
								if(align_hyps[i].cop[j]->update_or_refer.refer_count>0){
                                                                	fprintf(fp,"%s%c%f%c", align_hyps[i].cop[j]->x, info->output_sepchar, align_hyps[i].cop[j]->para, info->output_sepchar);
								}else{
									fprintf(fp,"%s%c%f%c", align_hyps[i].cop[j]->x, info->output_sepchar, info->lowest_para, info->output_sepchar);
								}
                                        		}
                                                        unalign_p++;
                                                        start_unalign_p=unalign_p;
                                                }
                                        }
                                        fprintf(fp,"\t");

                                        start_unalign_p=unalign_p=pair_data->y_part_strs[0];
                                        for(j=0;j<max_trans;j++){
                                                align_p=align_hyps[i].cop[j]->y;
                                                if(align_p[0]==info->del_char[0] && align_p[1]=='\0'){
                                                        fprintf(fp,"%s%c", info->del_char, info->output_sepchar);
                                                }else{
                                                        while(*align_p!='\0'){
                                                                if(*align_p!=*unalign_p){
                                                                        *unalign_p=info->output_joinchar;
								}
								align_p++;
                                                                unalign_p++;
                                                        }
							*unalign_p='\0';
                                                        fprintf(fp,"%s%c", start_unalign_p, info->output_sepchar);
							if (info->printScore){
								if(align_hyps[i].cop[j]->update_or_refer.refer_count>0){
                                                                	fprintf(fp,"%f%c%f%c%d%c", align_hyps[i].cop[j]->para, info->output_sepchar, info->lowest_para, info->output_sepchar, align_hyps[i].cop[j]->update_or_refer.refer_count, info->output_sepchar);
								}else{
									fprintf(fp,"%f%c%f%c%d%c", align_hyps[i].cop[j]->para, info->output_sepchar, info->lowest_para, info->output_sepchar, align_hyps[i].cop[j]->update_or_refer.refer_count, info->output_sepchar);
								}
                                                        }
                                                        unalign_p++;
                                                        start_unalign_p=unalign_p;

                                                }
                                        }

					if (info->printScore){
						fprintf(fp,"\t%d-best\t%f\ttrans:%d\n", i+1, align_hyps[i].score,max_trans);
					}else{
						fprintf(fp,"\n");
					}
				}

				if((AlignCount%1000)==0){
					fprintf(stderr,"Aligned:%d\r",AlignCount);
				}

			}else{
				noAlignCount++;
				fprintf(stderr,"\nNo Align Pair:%d\n",noAlignCount);
				fprintf(stderr,"x: ");
				printSeparatedString(pair_data->x_part_strs[0], pair_data->x_size);
				fprintf(stderr,"y: ");
				printSeparatedString(pair_data->y_part_strs[0], pair_data->y_size);
			}
		}

		fprintf(stderr,"End second alignments.\nAlignments:%d, No alignments:%d\n",AlignCount, noAlignCount);
	}

	fclose(fp);
	free(viterbi_table);
}
