/*
 * Created on 2004/09/18
 */
package jp.sourceforge.mrsp;

import java.util.List;

import jp.sourceforge.mrsp.model.Artifact;
import jp.sourceforge.mrsp.model.Result;
import jp.sourceforge.mrsp.service.RepoSearchService;
import jp.sourceforge.mrsp.service.RepoSearchServiceException;
import jp.sourceforge.mrsp.service.RepoSearchServiceImpl;
import jp.sourceforge.mrsp.util.MavenDependencyTagMaker;

import org.eclipse.core.runtime.Path;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.part.ViewPart;

/**
 * 
 * @author Ryuzo Yamamoto
 * @version $Id: ResultView.java,v 1.10 2004/10/04 13:59:04 dragon3 Exp $
 */
public class ResultView extends ViewPart {

    private TableViewer viewer;

    private Action searchAction;

    private Action jumpRepositoryAction;

    private Action copyDependencyAction;

    private Text queryText;

    private Combo typeCombo;

    private RepoSearchService searchService;

    private Result result;

    private boolean isSearching;

    public ResultView() {
    }

    public void setFocus() {
        queryText.setFocus();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.ui.IWorkbenchPart#dispose()
     */
    public void dispose() {
        super.dispose();
        this.viewer = null;
        this.searchAction = null;
        this.jumpRepositoryAction = null;
        this.copyDependencyAction = null;
        this.searchService = null;
        this.result = null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.ui.IWorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
     */
    public void createPartControl(Composite parent) {
        this.setPartName("MavenRepoSearch");
        GridLayout gridLayout = new GridLayout(2, false);
        parent.setLayout(gridLayout);

        // Type Composite
        Composite typeComp = new Composite(parent, SWT.NO_FOCUS);
        typeComp.setLayout(new GridLayout(2, false));
        GridData typeCompGridData = new GridData(SWT.NULL);
        typeComp.setLayoutData(typeCompGridData);

        // Type Label
        Label typeLabel = new Label(typeComp, SWT.RIGHT);
        typeLabel.setText("type:");
        GridData typeLabelGridData = new GridData(SWT.NULL);
        typeLabel.setLayoutData(typeLabelGridData);
        typeLabel.setToolTipText(MavenRepoSearchPlugin
                .getResourceString("typeComboToolTip"));

        // Type Combo
        typeCombo = new Combo(typeComp, SWT.READ_ONLY | SWT.DROP_DOWN);
        typeCombo.add("all");
        typeCombo.add("jar");
        typeCombo.add("plugin");
        typeCombo.add("tld");
        GridData typeComboGridData = new GridData(SWT.NULL);
        typeCombo.setLayoutData(typeComboGridData);
        typeCombo.select(0);

        // Keyword Text Field
        queryText = new Text(parent, SWT.SINGLE | SWT.BORDER);
        GridData queryTextGridData = new GridData(GridData.FILL_HORIZONTAL);
        queryText.setLayoutData(queryTextGridData);
        queryText.setToolTipText(MavenRepoSearchPlugin
                .getResourceString("queryTextToolTip"));

        // Result View
        Table table = createTable(parent);
        GridData tableGridData = new GridData(GridData.FILL_BOTH);
        tableGridData.grabExcessHorizontalSpace = true;
        tableGridData.grabExcessVerticalSpace = true;
        tableGridData.horizontalSpan = 2;
        table.setLayoutData(tableGridData);

        viewer = new TableViewer(table);
        viewer.setContentProvider(new ResultContentProvider());
        viewer.setLabelProvider(new ResultLabelProvider());

        searchService = new RepoSearchServiceImpl();
        result = new Result();
        viewer.setInput(result);

        createActions();
        createContextMenu();
        hookActions();

        parent.pack();
    }

    /**
     * TableViewer̐eTable𐶐Ԃ
     * 
     * @param parent
     * @return
     */
    private Table createTable(Composite parent) {
        Table table = new Table(parent, SWT.SINGLE | SWT.H_SCROLL
                | SWT.V_SCROLL | SWT.FULL_SELECTION);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);
        TableColumn column_0 = new TableColumn(table, SWT.LEFT, 0);
        column_0.setText("groupId");
        column_0.setWidth(80);
        column_0.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                viewer.setSorter(new ArtifactViewerSorter(
                        ArtifactViewerSorter.GROUPID));
            }
        });

        TableColumn column_1 = new TableColumn(table, SWT.LEFT, 1);
        column_1.setText("artifactId");
        column_1.setWidth(150);
        column_1.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                viewer.setSorter(new ArtifactViewerSorter(
                        ArtifactViewerSorter.ARTIFACTID));
            }
        });

        TableColumn column_2 = new TableColumn(table, SWT.LEFT, 2);
        column_2.setText("version");
        column_2.setWidth(80);
        column_2.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                viewer.setSorter(new ArtifactViewerSorter(
                        ArtifactViewerSorter.VERSION));
            }
        });

        TableColumn column_3 = new TableColumn(table, SWT.LEFT, 3);
        column_3.setText("type");
        column_3.setWidth(80);
        column_3.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                viewer.setSorter(new ArtifactViewerSorter(
                        ArtifactViewerSorter.TYPE));
            }
        });
        TableColumn column_4 = new TableColumn(table, SWT.LEFT, 4);
        column_4.setText("repository");
        column_4.setWidth(250);
        column_4.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                viewer.setSorter(new ArtifactViewerSorter(
                        ArtifactViewerSorter.REPOSITORY));
            }
        });
        return table;
    }

    /**
     * Action
     *  
     */
    private void createActions() {
        searchAction = new Action() {
            public void run() {
                doSearch();
            }
        };
        jumpRepositoryAction = new Action(MavenRepoSearchPlugin
                .getResourceString("jumpRepositoryActionTitle")) {
            public void run() {
                ISelection selection = viewer.getSelection();
                Object obj = ((IStructuredSelection) selection)
                        .getFirstElement();
                if (obj != null) {
                    Artifact artifact = (Artifact) obj;
                    Program.launch(artifact.getRepoUrl()
                            + artifact.getGroupId() + "/");
                }
            }
        };
        copyDependencyAction = new Action(MavenRepoSearchPlugin
                .getResourceString("copyDependencyActionTitle")) {
            public void run() {
                ISelection selection = viewer.getSelection();
                Object obj = ((IStructuredSelection) selection)
                        .getFirstElement();
                if (obj != null) {
                    Artifact artifact = (Artifact) obj;

                    // ClipbordɃRs[
                    Clipboard clipboard = new Clipboard(viewer.getControl()
                            .getDisplay());
                    clipboard.setContents(
                            new Object[] { MavenDependencyTagMaker
                                    .create(artifact) },
                            new Transfer[] { TextTransfer.getInstance() });
                }
            }
        };
        viewer.addSelectionChangedListener(new ISelectionChangedListener() {
            public void selectionChanged(SelectionChangedEvent event) {
                updateActionEnablement();
            }
        });

        jumpRepositoryAction.setEnabled(false);
        copyDependencyAction.setEnabled(false);
    }

    /**
     * ContextMenu
     *  
     */
    private void createContextMenu() {
        MenuManager menuMgr = new MenuManager();
        menuMgr.setRemoveAllWhenShown(true);
        menuMgr.addMenuListener(new IMenuListener() {
            public void menuAboutToShow(IMenuManager mgr) {
                fillContextMenu(mgr);
            }
        });
        Menu menu = menuMgr.createContextMenu(viewer.getControl());
        viewer.getControl().setMenu(menu);
        getSite().registerContextMenu(menuMgr, viewer);
    }

    /**
     * ContexMenue
     * 
     * @param mgr
     */
    private void fillContextMenu(IMenuManager mgr) {
        mgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
        mgr.add(copyDependencyAction);
        mgr.add(new Separator());
        mgr.add(jumpRepositoryAction);
    }

    private void hookActions() {
        viewer.addDoubleClickListener(new IDoubleClickListener() {
            public void doubleClick(DoubleClickEvent event) {
                jumpRepositoryAction.run();
            }
        });
        queryText.addKeyListener(new KeyAdapter() {
            public void keyReleased(KeyEvent e) {
                if (e.keyCode == SWT.CR) {
                    searchAction.run();
                }
            }
        });
    }

    private void updateActionEnablement() {
        IStructuredSelection sel = (IStructuredSelection) viewer.getSelection();
        jumpRepositoryAction.setEnabled(sel.size() > 0);
        copyDependencyAction.setEnabled(sel.size() > 0);
    }

    private ImageDescriptor getImageDescriptor(String relativePath) {
        String iconPath = "icons/";
        MavenRepoSearchPlugin plugin = MavenRepoSearchPlugin.getDefault();
        return ImageDescriptor.createFromURL(plugin.find(new Path(iconPath
                + relativePath)));
    }

    /**
     * 
     *  
     */
    private void doSearch() {
        if (queryText.getText().length() < 2 || isSearching) {
            return;
        }
        queryText.setEditable(false);
        isSearching = true;
        String keyword = queryText.getText();
        String type = typeCombo.getText();
        try {
            List artifacts = searchService.searchArtifactByKeywordType(keyword,
                    type);
            if (artifacts.size() > 0) {
                result.setArtifacts(artifacts);
            } else {
                showMessage(MavenRepoSearchPlugin.getResourceString("notFound"));
            }
        } catch (RepoSearchServiceException e) {
            MavenRepoSearchPlugin.log(e);
            showMessage(MavenRepoSearchPlugin.getResourceString("searchFail"));
        }
        queryText.setEditable(true);
        viewer.setSelection(null);
        viewer.getTable().setFocus();
        isSearching = false;
    }

    /**
     * MessageDialog\
     * 
     * @param msg
     */
    private void showMessage(String msg) {
        MessageDialog.openWarning(viewer.getControl().getShell(),
                "Maven Repo Search", msg);
    }
}