/*********************************************************/
/*!
 @addtoGroup AB3D
 @file NewAB3DLabeling.cpp
 @brief AB3DLabeling Functions.
 
 @date 2013.07.01
 @author M.Fukuda
 @version 1.0.0

 -------------------------------------------------------
 Copyright(C) 2006-2014 
 Eisai Co., Ltd. All rights reserved.
 -------------------------------------------------------
 @note
 */
/*********************************************************/
#include "StdAfx.h"
#include "NewAB3DLabeling.h"

#include <algorithm>
#include <boost/foreach.hpp>
#include <boost/format.hpp>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
//#include <boost/timer.hpp>
#include <boost/tuple/tuple.hpp>

#include <boost/algorithm/string/split.hpp>

#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/ini_parser.hpp>

#include "Algo_LWA.h"

// memo 2013.08.01
// DB֘ARg[܂B
// PeakPosition(Matrix)ׂ̈̃f[^mۂĂ܂B
#include "DBTools\DbController.h"

// memo 2013.08.01
// Mass++̃vOXbvNXłB
// Ăяo葱ώGȂ̂ōĂ݂܂B
#include "progress\progressAB3D.h"

//  Trading poist̃vWFNgƂĊOɂ̂
// NX ̃vWFNg̒ɓĂ܂B
#include "./AlchemyLib/TragingPost.h"

#include "Thoto/Peaks/Peak3DBase.hpp"
//#include "Thoto/Points/XYPoint.hpp"
//#include "Thoto/Points/XYZPoint.hpp"

const double kPPM(0.000001);

// param(Range)̕֐B
const ELM::LRP splitSettingVal(const std::string& tgtStr){
	ELM::LRP rst;
	if (tgtStr.empty()) return rst;

	std::vector<std::string> tmp;
	boost::algorithm::split(tmp, tgtStr, std::bind2nd(std::equal_to<char>(), ':'));
	if (tmp.empty()) {
		return rst;
	}
	rst.first = (tmp[0].size() > 0) ? boost::lexical_cast<double>(tmp[0]) : -1.0;
	if (tmp.size() > 1) {
		rst.second = (tmp[1].size() > 0) ? boost::lexical_cast<double>(tmp[1]) : -1.0;
	}
	return rst;
}

bool getDataPointsByRT(ELM::vXYZP& vxyzp, double rt, RFN::XYP::vXYP_& vxyp) {
	for (ELM::vXYZP::iterator it = vxyzp.begin(); it != vxyzp.end(); ++it) {
		if (it->z == rt) {
			vxyp = *it->vxyp;
			return true;
		}
	}
	return false;
}

vPeak3D::iterator NewAB3DLabeling::searchMZRange(
	vPeak3D::iterator it, 
	vPeak3D::iterator& endIt, 
	const double& rtTol, 
	const bool& direction, 
	ELM::LCRP& mzRange
) {
	const double currRt = (*it)->rt.c;
	vPeak3D::iterator retIt = it;
	while (it != endIt) {
		if (direction) {
			++it;
		} else {
			--it;
		}

		BS::Peak3D* pk3d = (*it);
		if (abs(pk3d->rt.c - currRt) > rtTol) {
			break;
		}

		// m/z͈̔͂擾B
		mzRange.l = std::min(mzRange.l, pk3d->mz.c);
		mzRange.r = std::max(mzRange.r, pk3d->mz.c);
		retIt = it;
	}
	return retIt;
}


bool NewAB3DLabeling::detectMZRangeByPeakPositions(
		Ptr& peak, vPeak3D& peaks,
		const double& rtTol, const double& intensityThreshold,
		ELM::LCRP& mzRange) {

	// Intensity̏𖞂͈͓̃s[N擾āARTŃ\[gB
	vPeak3D peaksSortedByRT;
	Cnum::getPeaksAboveIntensity(
		peaks, intensityThreshold, peaksSortedByRT);
	Cnum::sortByRTcenter(peaksSortedByRT);

	vPeak3D::iterator currItr, lItr, rItr;
	// RT̑oɌ邽߂ɁAƂȂs[NIterator擾B
	currItr = std::find(peaksSortedByRT.begin(), peaksSortedByRT.end(), peak);
	// RT͈̐̔͌
	rItr = searchMZRange(currItr, peaksSortedByRT.end()-1, rtTol, true, mzRange);
	// RT͈̔͌̕
	lItr = searchMZRange(currItr, peaksSortedByRT.begin(), rtTol, false, mzRange);

	return true;
}

bool NewAB3DLabeling::getXICPeak(ELM::vXYP& xic, const double& rt, ELM::PKPT& xicPeak) {
	const double XIC_LWA_RT_MERGE_DISTANCE = 0;
	const double XIC_LWA_FWHM_RATIO = getSettingMapDouble("MergeDataPointsWeightOfLocalMaximum");
	const ELM::LRP XIC_LWA_FWHM_WIDTH(RFN::LR::NG_LRP);

	double minRTDiff = 100.0;
	double rtDiff;

	// LWAɂs[NsbLOsB
	ELM::vPKPT xicPeaks;
	EMS::LWA::onLWALabeling(xic, XIC_LWA_RT_MERGE_DISTANCE, XIC_LWA_FWHM_RATIO, XIC_LWA_FWHM_WIDTH, xicPeaks);

	// LWAɂ擾s[N̒ŁAΏۂ̃s[NRTƍł߂s[N擾B
	for (ELM::vPKPT::iterator xicPIt = xicPeaks.begin(); xicPIt != xicPeaks.end(); ++xicPIt) {
		rtDiff = abs(xicPIt->Apex.first - rt);
		if (rtDiff < minRTDiff && xicPIt->Lp.first <= rt && rt <= xicPIt->Rp.first) {
			xicPeak = *xicPIt;
			minRTDiff = rtDiff;
		}
		if (rtDiff == 0.0) break;
	}
	return (minRTDiff == 0);
}

inline double NewAB3DLabeling::calcAreaByXIC(
		ELM::vXYP& xic, const double& lowerRT, const double& upperRT) {
	double area = 0.0;
	for (ELM::vXYP::iterator xicItr = xic.begin(); xicItr != xic.end(); ++xicItr) {
		if (lowerRT <= (*xicItr).first && (*xicItr).first <= upperRT) {
			area += (*xicItr).second;
		}
	}
	return area;
}

void NewAB3DLabeling::deleteShoulderPeaks(vPeak3D& detectedPeaks) {

	const double INTENSITY_RATIO = getSettingMapDouble("RemoveShoulderPeaksIntRatio");
	const double RT_DIFF = getSettingMapDouble("RemoveShoulderPeaksRtTol");
	const double MZ_DIFF_PPM = getSettingMapDouble("RemoveShoulderPeaksMzTol");
	//const double MZ_DIFF = MZ_DIFF_PPM * 1/1000000;
	const double MZ_DIFF = MZ_DIFF_PPM * kPPM;

	Cnum::sortByIntensity_DESC(detectedPeaks);

	// s[ÑXgRs[ARTŃ\[gB
	vPeak3D pksSortedByMZ = detectedPeaks;
	//vPeak3D tgtPks;
	vPeak3D rejectPks;

	Cnum::sortByMZcenter(pksSortedByMZ);

	for (vPeak3D::iterator pkItr = detectedPeaks.begin(); pkItr != detectedPeaks.end(); ++pkItr) {
		const double intThr = (*pkItr)->intensity / INTENSITY_RATIO;
		const double mzDiff = (*pkItr)->mz.c * MZ_DIFF;
		const double rtLower = (*pkItr)->rt.l - RT_DIFF;
		const double rtUpper = (*pkItr)->rt.r + RT_DIFF;

		vPeak3D tgtPks;
		if (!Cnum::getPeaksBelowIntensityFromSortedByMz(
				pksSortedByMZ, intThr,
				(*pkItr)->mz.l - mzDiff, (*pkItr)->mz.r + mzDiff,
				rtLower, rtUpper, tgtPks)) {
			continue;
		}

		Cnum::setFlagToPeaks(tgtPks, false);
	}
	Cnum::removePeaksFromList(detectedPeaks);
}

// As[N̏
void NewAB3DLabeling::removeContinuousPeaks(vPeak3D& detectedPeaks, vPeak3D& clusterCandidatePeaks) {
	
	const double REMOVE_CONTINUOUS_PEAKS_MZ_PPM_TOLERANCE = getSettingMapDouble("RemoveContinuousPeaksMzTol");
	const double REMOVE_CONTINUOUS_PEAKS_MZ_RATIO = REMOVE_CONTINUOUS_PEAKS_MZ_PPM_TOLERANCE * kPPM;
	const double REMOVE_CONTINUOUS_PEAKS_RT_TOLERANCE = getSettingMapDouble("RemoveContinuousPeaksRtTol");
	const double REMOVE_CONTINUOUS_PEAKS_INTENSITY_RATIO = getSettingMapDouble("RemoveContinuousPeaksIntRatio");
	
	vPeak3D::iterator tmpItr;
	vPeak3D nearMZPks;
	double currMZ, lowerMZ, upperMZ;
	double upperRT, lowerRT;
	double intThreshold;
	double threshold;

	// |tOɖ߂B
	Cnum::setFlagToPeaks(detectedPeaks, true);

	vPeak3D pksSortedByMZ = detectedPeaks;
	Cnum::sortByMZcenter(pksSortedByMZ);

	Cnum::sortByIntensity_DESC(detectedPeaks);
	for (vPeak3D::iterator pkItr = detectedPeaks.begin(); pkItr != detectedPeaks.end(); ++pkItr) {
		if (!(*pkItr)->sflag) {
			continue;
		}

		intThreshold = (*pkItr)->intensity * REMOVE_CONTINUOUS_PEAKS_INTENSITY_RATIO;
		currMZ = (*pkItr)->mz.c;
		lowerMZ = currMZ - REMOVE_CONTINUOUS_PEAKS_MZ_RATIO * currMZ;
		upperMZ = currMZ + REMOVE_CONTINUOUS_PEAKS_MZ_RATIO * currMZ;

		// MZߖT̃s[NXgNAB
		nearMZPks.clear();
		// MZߖT̃s[N擾B
		// Intensity+1̂́Aꎩg̃s[N擾߁B
		if (!Cnum::getPeaksBelowIntensityFromSortedByMz(
			pksSortedByMZ, (*pkItr)->intensity,
			lowerMZ, upperMZ, -1, -1, nearMZPks)) {
			// m/z̋ߖTɑ݂s[N݂Ȃꍇ
			continue;
		}
		// g̃s[NǉB 
		nearMZPks.push_back(*pkItr);
		// MZߖT̃s[NXgRetention TimeŃ\[gB
		Cnum::sortByRTcenter(nearMZPks);

		// Ώۂ̃s[N̏
		upperRT = (*pkItr)->rt.c;
		lowerRT = (*pkItr)->rt.c;
		for (tmpItr = nearMZPks.begin(); tmpItr != nearMZPks.end(); ++tmpItr) {
			if ((*tmpItr)->rt.c < lowerRT) continue;
			if ((*tmpItr)->rt.l <= upperRT) {
				// I[o[bvĂꍇ
				threshold = (*pkItr)->intensity;
			} else if ((*tmpItr)->rt.l <= upperRT + REMOVE_CONTINUOUS_PEAKS_RT_TOLERANCE) {
				// I[o[bvĂȂAKl͈͓̔̏ꍇ
				threshold =  intThreshold;
			} else {
				// RTcenterŃ\[gĂAs[N͈͂̏̏ԂƂ͈قȂ\邽
				// breakł͂ȂcontinueƂB
				continue;
			}
			// s[N͈͂̍őێB
			upperRT = std::max(upperRT, (*tmpItr)->rt.r);
			if ((*tmpItr)->sflag && (*tmpItr)->intensity < threshold) {
				if ((*tmpItr)->sflag) {
					clusterCandidatePeaks.push_back(*tmpItr);
				}
				(*tmpItr)->sflag = false;
			}
		}

		// Ώۂ̃s[N̏
		std::reverse(nearMZPks.begin(), nearMZPks.end());
		upperRT = (*pkItr)->rt.c;
		lowerRT = (*pkItr)->rt.c;
		for (tmpItr = nearMZPks.begin(); tmpItr != nearMZPks.end(); ++tmpItr) {
			if ((*tmpItr)->rt.c > upperRT) continue;
			if ((*tmpItr)->rt.r >= lowerRT) {
				// I[o[bvĂꍇ
				threshold = (*pkItr)->intensity;
			} else if ((*tmpItr)->rt.r >= lowerRT - REMOVE_CONTINUOUS_PEAKS_RT_TOLERANCE) {
				// I[o[bvĂȂAKl͈͓̔̏ꍇ
				threshold = intThreshold;
			} else {
				// RTcenterŃ\[gĂAs[N͈͂̉̏ԂƂ͈قȂ\邽
				// breakł͂ȂcontinueƂB
				continue;
			}
			// s[N͈͂̍ŏێB
			lowerRT = std::min(lowerRT, (*tmpItr)->rt.l);
			if ((*tmpItr)->sflag && (*tmpItr)->intensity < threshold) {
				if ((*tmpItr)->sflag) {
					clusterCandidatePeaks.push_back(*tmpItr);
				}
				(*tmpItr)->sflag = false;
			}
		}
	}

	Cnum::deleteFalseFlagObject(detectedPeaks);
	Cnum::sortByIntensity_DESC(detectedPeaks);
}


//Cluster̍쐬
void NewAB3DLabeling::createClusters(
	vPeak3D& detectedPeaks, std::map<Ptr, std::pair<int,Ptr>>& clusterMap, vPeak3D& clusterCandidatePeaks, ELM::vXYZP& vxyzp
){
	const double clusterRtTol = getSettingMapDouble("ClusteringRtTol");

	const double clusterMzTolRatio = getSettingMapDouble("ClusteringMzTol") * kPPM;

	// ʑ̃NX^[쐬ہAʑ̂̃s[NIntensityThreahold
	const double minIntensity = getSettingMapDouble("MergeMinIntensity");

	ELM::LRP tmpDbl = splitSettingVal(settingMap_["ClusteringIntRatio"]);
	const double clusterIntensityRateLowerLimit = tmpDbl.first;
	const double clusterIntensityRateUpperLimit = tmpDbl.second;

	tmpDbl = splitSettingVal(settingMap_["ClusteringCharge"]);
	const int clusterMinCharge = (int)tmpDbl.first;
	const int clusterMaxCharge = (int)tmpDbl.second;

	const int minNumOfPeaks = getSettingMapInteger("ClusteringNumberOfPeaks");

	// |tOɖ߂B
	Cnum::setFlagToPeaks(detectedPeaks, true);
	Cnum::setFlagToPeaks(clusterCandidatePeaks, true);

	// s[ÑXgRs[ARTŃ\[gB
	vPeak3D pksSortedByRT = detectedPeaks;
	pksSortedByRT.reserve(pksSortedByRT.size() + clusterCandidatePeaks.size());
	pksSortedByRT.insert(pksSortedByRT.end(), clusterCandidatePeaks.begin(), clusterCandidatePeaks.end());
	Cnum::sortByRTcenter(pksSortedByRT);

	std::map<Ptr, Ptr> pk2MonoIsoPk;

	// s[N̍珈ĂB
	// s[N~ɕłOƂB
	vPeak3D additionalPks;
	for (vPeak3D::iterator pkItr = detectedPeaks.begin(); pkItr != detectedPeaks.end(); ++pkItr) {

		if (clusterMap.find(*pkItr) != clusterMap.end()) {
			continue;
		}

		RFN::XYP::vXYP_ vxyp;
		if (!getDataPointsByRT(vxyzp, (*pkItr)->rt.c, vxyp)) {
			// YScanDataPoint擾łȂƂ
			continue;
		}

		// RTTolerancẽ͈͓s[N擾B
		// m/z͈̔͂ɂẮAΏۂ̃mAC\s[NDamۂĂ΂悢B
		// ۂɂ͂ȍ~̏ōXɍi荞܂B
		vPeak3D neighborPeaksSortedByIntensity;
		if (!Cnum::getPeaksFromSortedByRT(
				pksSortedByRT, 
				(*pkItr)->mz.c - clusterMzTolRatio,	
				(*pkItr)->mz.c + 10.0,				
				(*pkItr)->rt.c - clusterRtTol,
				(*pkItr)->rt.c + clusterRtTol, 
				neighborPeaksSortedByIntensity)) {
			continue;
		}
		Cnum::sortByIntensity_DESC(neighborPeaksSortedByIntensity);

		vPeak3D clusteredPks;
		bool foundExistingCluster = false;
		std::pair<int, Ptr> clusterPair;

		const double mzTol = (*pkItr)->mz.c * clusterMzTolRatio;
		double diffMZ;
		int foundCharge = -1;

		// ̃[v
		// [v̏I
		// ݒ肳Ă鉿͈̔͂𒴂ƂA܂͊̃NX^̃mAC\ƂȂƂ
		for (int charge = clusterMaxCharge;
				!foundExistingCluster && charge >= clusterMinCharge; charge--) {

			diffMZ = 1.0 / (double)charge;
			vPeak3D tmpClusteredPks;

			double latestMZ = (*pkItr)->mz.c;
			double latestIntensity = (*pkItr)->intensity;
			int c = 1;

			// 鉿̓ʑ̃s[Ñ[v
			// [v̏I
			// YʒuɃs[NȂȂƂA
			// ܂͊YʒuɉṽNX^̃mAC\s[NƂ
			while (1) {
				double calcMz = latestMZ + diffMZ;
				const double upperIntensity = latestIntensity * clusterIntensityRateUpperLimit;
				const double lowerIntensity = latestIntensity * clusterIntensityRateLowerLimit;

				// m/zIntensity͈̔͂Ōi荞ށB
				vPeak3D tmpCandidatePks;
				Cnum::getPeaksFromSortedByIntensityDesc(
						neighborPeaksSortedByIntensity,
						lowerIntensity, upperIntensity,
						calcMz - mzTol, calcMz + mzTol,
						tmpCandidatePks);

				// RT̃s[N̂Tolerance𖞂Ă邪A 
				// łIntensitỹs[NRTႢIntensitỹGAOĂs[NOB
				vPeak3D rtOverlappedPeaks;
				if (!Cnum::getRTOverlappedPeaks(
						tmpCandidatePks, (*pkItr)->rt.c, rtOverlappedPeaks)) {
					break;
				}

				// m/z̗_lƂ̍łs[NIB
				Ptr foundPk;
				Cnum::getNearestMzPeak(rtOverlappedPeaks, calcMz, foundPk);
				if (clusterMap.find(foundPk) != clusterMap.end()) {
					// ɃNX^[Ăꍇ
					if (clusterMap[foundPk].first == charge) {
						// vĂꍇ́ÃNX^[Ɋ܂߂B
						clusterPair = clusterMap[foundPk]; 
						foundExistingCluster = true;
					}
					break;
				} else {
					// NX^[ĂȂs[N݂ꍇ
					tmpClusteredPks.push_back(foundPk);
					latestMZ = foundPk->mz.c;
					latestIntensity = foundPk->intensity;
				}
				++c;
			}

			if (tmpClusteredPks.size() >= (minNumOfPeaks - 1)) {
				if (clusteredPks.size() < tmpClusteredPks.size()) {
					// ɌNX^[̌\s[N̐ꍇ
					clusteredPks = tmpClusteredPks;
					foundCharge = charge;
				}
			}
			if (foundExistingCluster) {
				// ɑ݂NX^[Ɋ܂߂ꍇ
				foundCharge = charge;
				break;
			}
		}

		if (foundCharge > 0) {
			if (foundExistingCluster) {
				// ɑ݂NX^[Ɋ܂߂ꍇɂ́A 
				// ̃s[NmAC\ɂȂ邽߂ɁAɑ݂铯ʑ̃s[Ñ}bvuB
				vPeak3D tmpPks;
				std::map<Ptr, Ptr>::const_iterator foundItr;
				for (foundItr = pk2MonoIsoPk.begin(); foundItr != pk2MonoIsoPk.end(); ++foundItr) {
					if (clusterPair.second != foundItr->second) continue;
					tmpPks.push_back(foundItr->first);
				}
				for (vPeak3D::iterator tmpItr = tmpPks.begin(); tmpItr != tmpPks.end(); ++tmpItr) {
					clusterMap[*tmpItr] = std::make_pair(foundCharge, *pkItr);
					pk2MonoIsoPk[*tmpItr] = *pkItr;
				}
			}
			// Cluster ̍쐬
			clusterPair = std::make_pair(foundCharge, *pkItr);
			clusterMap[*pkItr] = clusterPair;
			pk2MonoIsoPk[*pkItr] = *pkItr;
			for (vPeak3D::iterator clsPkItr = clusteredPks.begin(); clsPkItr != clusteredPks.end(); ++clsPkItr) {
				clusterMap[*clsPkItr] = clusterPair;
				vPeak3D::iterator delItr = std::remove(pksSortedByRT.begin(), pksSortedByRT.end(), *clsPkItr);
				pksSortedByRT.erase(delItr, pksSortedByRT.end());
				pk2MonoIsoPk[*clsPkItr] = *pkItr;

				if (std::find(clusterCandidatePeaks.begin(), clusterCandidatePeaks.end(), *clsPkItr) != clusterCandidatePeaks.end()) {
					additionalPks.push_back(*clsPkItr);
				}
			}
		}
	}

	detectedPeaks.reserve(detectedPeaks.size() + additionalPks.size());
	detectedPeaks.insert(detectedPeaks.end(), additionalPks.begin(), additionalPks.end());

	for (vPeak3D::iterator pkItr = additionalPks.begin(); pkItr != additionalPks.end(); ++pkItr) {
		vPeak3D::iterator endItr = std::remove(clusterCandidatePeaks.begin(), clusterCandidatePeaks.end(), *pkItr);
		clusterCandidatePeaks.erase(endItr, clusterCandidatePeaks.end());
	}
}

// settingMap (params擾l)@@Key@-> Val(Integer)󂯎
// Ȃ -1 Ԃ
int NewAB3DLabeling::getSettingMapInteger(const std::string& pname) const 
{
	std::map<std::string, std::string>::const_iterator itr = settingMap_.find(pname);
	if (itr != settingMap_.end()) {
		return boost::lexical_cast<int>(itr->second);
	}
	return -1;
}

double NewAB3DLabeling::getSettingMapDouble(const std::string& pname) const 
{
	std::map<std::string, std::string>::const_iterator itr = settingMap_.find(pname);
	if (itr != settingMap_.end()) {
		return boost::lexical_cast<double>(itr->second);
	}
	return -1.0;
};

// For AB3D Single
// TvJAB3DsꍇɌĂ΂܂B
// [in]  dataSet (mass++̃TṽIuWFNg)
// [out] peaks2d (mass++peaks2D̃IuWFNgi))
void NewAB3DLabeling::doAB3D (
	kome::objects::DataSet& dataSet,
	kome::objects::Peaks2D* peaks2d
) {
	onDoAB3D(dataSet, peaks2d, nullptr);
}

// For AB3D with peakPositions
// DB WizardoRŁ@TvJAB3DsꍇɌĂ΂܂B
// [in]  dataSet (mass++̃TṽIuWFNg)
// [out] dbac    (mass++peaks2Ď)
void NewAB3DLabeling::doAB3D (kome::objects::DataSet& dataSet, tDBAC* dbac) {
	onDoAB3D(dataSet, nullptr, dbac);
}

// vOXo[params({)󂯎āAn܂B
// ...Top.cppŏĂ܂B
// RXgN^ɓꂽقǂ܂B
void NewAB3DLabeling::bindProgress(kome::core::Progress* progress)
{
	myProgress_ = progress;
}

const bool NewAB3DLabeling::isSelected(const char* key) {
	return (std::strcmp(settingMap_[key].c_str(), "true") == 0);
}

// ̊g˂ on ŌĂ΂ɕύXB
bool NewAB3DLabeling::onDoAB3D (
	kome::objects::DataSet& dataSet,
	kome::objects::Peaks2D* peaks2d,
	tDBAC* dbac
) {

	ALC::TradingPost tdPost;

	// ؗpDB擾tO
	const bool needSingleXICDB = (std::strcmp(settingMap_["OutPutForDataBase"].c_str(),"true") == 0);
	// Data PointiXICɎgSpectruml̏W)
	ELM::vXYZP main_vxyzp;
	// vZ͈
	ELM::MZRTP mainMZRT =
		ELM::MZRTP(splitSettingVal(settingMap_["MzRange"]), splitSettingVal(settingMap_["RtRange"]));
	// ̌vZ̔z
	vPeak3D peaksSoterdByIntensity;

	// PPM -> RatioɌvZ
	const double mzPPMTol = getSettingMapDouble("MergeDataPointsMzTol"); // 10.0;
	const double mzTolRatio = mzPPMTol * kPPM;

	ELM::LRP tmpDbl = splitSettingVal(settingMap_["ClusteringCharge"]);
	const int minCharge = (int)tmpDbl.first;
	const int maxCharge = (int)tmpDbl.second;


	const bool removeIsotopicDataPoints =
			isSelected("MergeDataPointsRemoveNonIsotopicDataPoints");

	if (mainMZRT.mz.first == 0 && mainMZRT.mz.second == 0) {
		mainMZRT.mz.first = -1;
		mainMZRT.mz.second = -1;
	}
	if (mainMZRT.rt.first == 0 && mainMZRT.rt.second == 0) {
		mainMZRT.rt.first = -1;
		mainMZRT.rt.second = -1;
	}
	
	tdPost.getSpectrumDataPoints(
		dataSet, mainMZRT, 1, ALC::HAS_CHROM, main_vxyzp, peaksSoterdByIntensity,
		removeIsotopicDataPoints, minCharge, maxCharge, mzTolRatio);

	Cnum::sortByIntensity_DESC(peaksSoterdByIntensity);

	const double availableIntensityRatio = getSettingMapDouble("MergeDataPointsIntRatio");

	const double rtTol = getSettingMapDouble("MergeDataPointsRtTol");//0.5; 
	const double peakendhighprecent = 0.1;
	const double minIntensity = getSettingMapDouble("MergeMinIntensity");

	// ʑ̃NX^[쐬ہAʑ̂̃s[NIntensityThreahold
	const double lowerDatapointIntensityLimit = minIntensity * availableIntensityRatio;

	// m/z͈̔͂肷s[Nm/ẑ܂ܔ͈͂ɓKpĂ܂Ƃ肬XIC擾łȂꍇ̂ŁA
	// L߂ɔ͈͂ݒ肷KvB̗]̕(P:PPM)B
	const double XIC_MZ_MARGIN_PPM = 5.0;
	const double XIC_MZ_MARGIN_RATE_LOWER = (1.0 - XIC_MZ_MARGIN_PPM * kPPM);
	const double XIC_MZ_MARGIN_RATE_UPPER = (1.0 + XIC_MZ_MARGIN_PPM * kPPM);

	// ꎟƗpIuWFNg
	ELM::vPKPT xicLWAPeaks;
	vPeak3D	detectedVPK3D;
	vPeak3D clusterCandidateVPK3D;
	ELM::vXYP rstXIC;

	//Ⴂs[N
	Cnum::eraseRemoveFalseFlagObject(peaksSoterdByIntensity, lowerDatapointIntensityLimit);

	vPeak3D peaksSortedByMz = peaksSoterdByIntensity;
	Cnum::sortByMZcenter(peaksSortedByMz);

	vPeak3D::iterator currItr = peaksSoterdByIntensity.begin();
	detectedVPK3D.reserve(peaksSoterdByIntensity.size());

	// Flagȏ㗭܂@ĎtO
	long falseFlagCount = 0;
	long refleshcycle = static_cast<long>(peaksSoterdByIntensity.size() * 0.04);

	// DataBasepMAP
	std::map<Ptr, int> dbmap_Prof;

	// Progressǉ
	// Progress֘Å֐ĂяoώGȂ̂ŁANXł܂Ƃ߂Ă܂B
	// ؎ɃvOXזȏꍇ́AnullptrɂĂ
	// ȍ~ prog͋Lq܂AS悤ɂȂ܂B
	AB3DProg prog(myProgress_);

	prog.nextStep("Detecting peaks...");

	while ( !myProgress_->isStopped() ) {
		// l𒴂̂ŃZbgB
		if (refleshcycle < falseFlagCount) {

			// tOfalseȃs[N菜B
			// Intensitył̃\[g͈ێĂ̂Ń\[gȂB
			Cnum::removePeaksFromList(peaksSoterdByIntensity);

			peaksSortedByMz = peaksSoterdByIntensity;
			Cnum::sortByMZcenter(peaksSortedByMz);

			currItr = peaksSoterdByIntensity.begin();
			falseFlagCount = 0;
			prog.nextStep();
		}

		// Intensity~̃Xg̖̐擪ɈړB
		Ptr currPeak;
		if (!Cnum::setForeFront(peaksSoterdByIntensity, currPeak, currItr)) {
			break;
		}

		if (currPeak->intensity < minIntensity) {
			// Intensity~ɏs߁A
			// Ώۂ̃s[NIntensity臒lΏIB
			break;
		}

		// Intensity̑؂ɁȂ͈͓SẴs[NIntensity̍~Ŏ擾B
		// RTɊւẮAs[NsbLO̓sART̗]Tolerance2{mۂB
		const double lowerXICRt = currPeak->rt.c - rtTol;
		const double upperXICRt = currPeak->rt.c + rtTol;
		vPeak3D peaksInArea;
		peaksInArea.clear();
		Cnum::getPeaksFromSortedByMz(peaksSortedByMz,
			currPeak->mz.c * (1.0 - mzTolRatio), currPeak->mz.c * (1.0 + mzTolRatio),
			lowerXICRt, upperXICRt, peaksInArea);

		// os[Nm/z̒SƔ͈
		// m/z͈̔͂ɂẮǍ̏Ō肷̂ŁAƂ肠Ώۃs[N̒SZbgB
		ELM::LCRP peakMzRange(currPeak->mz.c, currPeak->mz.c, currPeak->mz.c);

		// ݂s[NAm/z͈͂肷B
		if (!detectMZRangeByPeakPositions(
				currPeak, peaksInArea,
				rtTol, currPeak->intensity * availableIntensityRatio, peakMzRange)) {
			// Ώ۔͈͂Lȃs[N擾łȂꍇ
			currPeak->sflag = false;
			++falseFlagCount;
			continue;
		}

		// 擾s[Nm/z኱L߂ɔ͈͂ݒ肷B
		// Ȃ΁Am/zs[NꂷɂȂĂ܂A҂XIC擾łȂꍇB
		peakMzRange.l *= XIC_MZ_MARGIN_RATE_LOWER;
		peakMzRange.r *= XIC_MZ_MARGIN_RATE_UPPER;

		rstXIC.clear();
		if (!tdPost.createXIC(
				main_vxyzp, peakMzRange.l, peakMzRange.r, lowerXICRt, upperXICRt, rstXIC)) {
			// w͈͂XIC̎擾ɎsꍇA
			// Ώۂ̃XyNgs[N𖳌ɂB
			currPeak->sflag = false;
			++falseFlagCount;
			continue;
		}

		ELM::PKPT xicPeak;
		if (!getXICPeak(rstXIC, currPeak->rt.c, xicPeak)) {
			// XICs[N̎擾ɎsꍇA
			// Ώۂ̃XyNgs[N𖳌ɂB
			currPeak->sflag = false;
			++falseFlagCount;
			continue;
		}

		// os[NRT̒SƔ͈
		ELM::LCRP peakRtRange(xicPeak.Lp.first, currPeak->rt.c, xicPeak.Rp.first);

		// ͈͓ɑ݂f[^|Cg𖳌ɂB
		falseFlagCount += setFalseFlagToUsedPeak3D(
			peaksInArea,
			peakMzRange.l, peakMzRange.r, peakRtRange.l, peakRtRange.r);

		// s[N͈͓ɑ݂f[^|CgIntensity擾āA擾̒l0ɂB
		std::vector<double> intensities;
		tdPost.clearIntensity(
			main_vxyzp,
			peakMzRange.l, peakMzRange.r,
			peakRtRange.l, peakRtRange.r, intensities);

		// S/NȂǂ̊Ńs[N̗LsB
		// XIC̎擾ōsȂ̂́As[Nłꍇł
		// ͈͓̑SẴf[^|Cg̃tO𖳌ɂ߁B
		//if (!verifyPeak(rstXIC, xicPeak, intensities)) {
		const int peakStatus = verifyPeak(rstXIC, xicPeak, intensities);
		if (peakStatus == 0) {
			// ȃs[NƂĔ肳ꂽꍇ
			continue;
		}

		// ͈͓XICvArea擾B
		double peakArea = calcAreaByXIC(rstXIC, peakRtRange.l, peakRtRange.r);


		// s[NǉB
		ELM::PK3D_ptr pk =
			new ELM::Peak3D(currPeak->intensity, peakMzRange, peakRtRange, peakArea,
					0.0, currPeak->logintensity, xicPeak.fwhmVal, intensities.size());
		pk->sflag = true;

		// KvłDBɏ
		if (!writeToDB(dbac, pk, xicPeak, rstXIC, dbmap_Prof)) {
			return false;
		}

		if (peakStatus == 2 && currPeak->intensity >= minIntensity) {
			// Ƃēo^
			detectedVPK3D.push_back(pk);
		} else {
			// NX^O̕⏕Ƃēo^
			clusterCandidateVPK3D.push_back(pk);
		}
	}

	unsigned int numOfPeaks = detectedVPK3D.size();
	LOG_DEBUG(FMT("Detected %d peak(s).", numOfPeaks));

	// vZɎg3Ds[NSď
	// RTvPK3D Ă΂Ȃ悤ɒӁB
	Cnum::deleteAllElm(peaksSoterdByIntensity);

	if( !myProgress_->isStopped() ){

		// RTɘAs[N폜B
		prog.nextStep("Removing continous peaks...");

		removeContinuousPeaks(detectedVPK3D, clusterCandidateVPK3D);
	
		// Shoulders[N폜B
		prog.nextStep("Removing shoulder peaks...");

		deleteShoulderPeaks(detectedVPK3D);

		// Cluster쐬B
		prog.nextStep("Creating isotopic clusters...");

		std::map<Ptr, std::pair<int,Ptr>> clusterChargeMap;
		createClusters(detectedVPK3D, clusterChargeMap, clusterCandidateVPK3D, main_vxyzp);

		if (peaks2d != nullptr) {
			toMassppPeak2D(peaks2d, clusterChargeMap, detectedVPK3D);
		} else {
			// KvłDBɃNX^܂߂ď
			writeClusterInfotoDB(dbac,clusterChargeMap,detectedVPK3D, dbmap_Prof);
		}

		// |C^mېDeleteB
		BOOST_FOREACH(const ELM::XYZP& tmpxyzp, main_vxyzp) {
			delete tmpxyzp.vxyp;
		}

		// S~
		Cnum::deleteAllElm(detectedVPK3D);
		Cnum::deleteAllElm(clusterCandidateVPK3D);
	}
	prog.closeMe();
	return true;
}

// \[gƁAFalseFlagɂȂIuWFNgB
// 󋵂ɂĂ͕svɂȂ邩Ȃ(\[g̕dj
//void NewAB3DLabeling::deleteFalseFlagAndSort(vPeak3D& byIntArr, vPeak3D& byMZArr) {
//	Cnum::sortByMZcenter(byMZArr);
//}

// ͈͓ɑ݂s[ÑtO|B
const unsigned int NewAB3DLabeling::setFalseFlagToUsedPeak3D (
	vPeak3D& peaks,
	const double& lowerMz, const double& upperMz,
	const double& lowerRt, const double& upperRt
){
	vPeak3D peaksInArea;
	peaksInArea.reserve(peaks.size());

	// w肵m/zRT͈͓̔ɑ݂s[N擾邽߂ɁA
	// xʂ̃XgɃRs[āARTŃ\[gĂ
	// Y͈͓̔̃s[N擾B
	vPeak3D& peaksSortedByRt = peaks;
	Cnum::sortByRTcenter(peaksSortedByRt);
	Cnum::getPeaksFromSortedByRT(peaksSortedByRt, lowerMz, upperMz, lowerRt, upperRt, peaksInArea);

	Cnum::setFlagToPeaks(peaksInArea, false);
	return peaksInArea.size();
}

// Y֐͉ׂ̈Ɂ@Lɐ؂肾Be͈ꏏB
// Mass++tH[}bgɕϊ(Peak2D & Cluster) 
void NewAB3DLabeling::toMassppPeak2D (
	kome::objects::Peaks2D* peaks2d, 
	std::map<Ptr, std::pair<int,Ptr>>& clusterChargeMap, 
	vPeak3D& detectedVPK3D
) {
	std::map<Ptr, std::pair<int,Ptr>>::iterator cCmapItr;
	std::map<Ptr, kome::objects::PeaksCluster2D*> clusterMap;
	std::map<Ptr, kome::objects::PeaksCluster2D*>::iterator cmapItr;
	std::pair<int, Ptr> clusterPair;

	// NX^[\Ȃs[NO邩ǂ
	const bool removeNonClustered = isSelected("ClusteringRemoveNonClustered");
	// mAC\݂̂ɍi荞ނǂ
	const bool onlyMonoIsoPeaks = isSelected("ClusteringOutputOnlyMonoisotopic");

	Cnum::sortByMZcenter(detectedVPK3D);

	BOOST_FOREACH(const Ptr pk, detectedVPK3D) {
		cCmapItr = clusterChargeMap.find(pk);

		// NX^[\s[Nǂ
		bool isClustered = (cCmapItr != clusterChargeMap.end());
		// mAC\̃s[Nǂ
		bool isMonoIsotopic = false;
		if (isClustered) {
			clusterPair = cCmapItr->second;
			cmapItr = clusterMap.find(clusterPair.second);
			isMonoIsotopic = (cmapItr == clusterMap.end());
		}

		if (!removeNonClustered || isClustered) {
			// ݒʂŁA"Exclude Non-clustered Peaks"Iꂽꍇ́A
			// NX^[\s[N݂̂s[NƂč̗pB 

			if (isMonoIsotopic || !onlyMonoIsoPeaks || (!removeNonClustered && !isClustered)) {

				// ݒʂŁA"Monoisotopic Peaks Only"Iꂽꍇ́A
				// mAC\̃s[N݂̗̂pB
				kome::objects::Peak2DElement* pe =
					peaks2d->createPeak(
							pk->rt.c, pk->rt.l, pk->rt.r,
							pk->mz.c, pk->mz.l, pk->mz.r, pk->intensity);
				pe->setArea(pk->area);
				if (isClustered) {
					pe->setCharge(clusterPair.first);
				}

				if (isMonoIsotopic) {
					kome::objects::PeaksCluster2D* cluster = peaks2d->createCluster();
					cluster->setCharge(clusterPair.first);
					clusterMap[clusterPair.second] = cluster;
				}
				if (isClustered) {
					pe->setCluster(clusterMap[clusterPair.second]);
				}
			}
		}
	}
}

// DB֏݂̔
// Single& DBWriteĂꍇATDPostoRDBɏށB
// Multȉꍇ́Aꎞe[uProfileɕKvȏĂB
const bool NewAB3DLabeling::writeToDB(
	tDBAC* dbac,
	ELM::PK3D_ptr pk, 
	ELM::PKPT& foundpeak,
	const ELM::vXYP& rstXIC, 
	std::map<Ptr, int>& dbmapProf
){
	if (dbac != nullptr) {
		return onWriteToDB(dbac, pk, foundpeak, rstXIC, dbmapProf);
	} 
	return true;
}

// P
// memo 2013.08.01
//
// dbmap_prof(map) ɂ
// PK3D̃|C^@ DBɕۑ@PeakIdRÂĂmۂĂ܂B

// Q
// memo 2013.08.02
// dbac->setXIC ̃C^tF[XύX܂B
// Thoton̎SĔrĂ܂B
// 1. ΏۂXIĆ@vXYP(XYData)
// @@rstXIC̎󂯑 const std::vector<std::pair<double,double>>& Ŏ󂯂Ă܂B
// @@ELM::vXYP@ typedef̂Ȃ̂œłB

// 2. s[N RT, M/Z, MatrixValue
// @ Peak2D@ƂẮ@s[N̒_ĂB
//    M/Z, RT ́AMatrix@ROW̑\l@ƂĎg܂B
//    Value̒ĺ@Matrix̊iq̒lƂĕ\܂B
//    iɒʐl悤ɂȂƎv܂j

// 3. ΏۂXIĆ@s[N(_(X,Y), Left(X,Y),Right(X,Y))
//    Matrxǐʂ̔g``悳ۂɁAs[NƂĔFE\lłB
//     LWA̒ŉdςĂȂ̂ŁAApexCenter̍W͓Ƒ܂A
//    vZł͂ȂA_̍WnĂB

const bool NewAB3DLabeling::onWriteToDB(
	tDBAC* dbac,
	ELM::PK3D_ptr pk, 
	ELM::PKPT& foundpeak,
	const ELM::vXYP& rstXIC, 
	std::map<Ptr, int>& dbmapProf
){
	// Q
	int dbPeakId = dbac->setXICData(
		rstXIC,
		pk->rt.c,
		pk->mz.c,
		pk->intensity,
		foundpeak.Apex.first,
		foundpeak.Apex.second,
		foundpeak.Lp.first,
		foundpeak.Lp.second,
		foundpeak.Rp.first,
		foundpeak.Rp.second
	);

	// P
	if (dbPeakId >= 0) {
		dbmapProf[pk] = dbPeakId;
		return true;
	}				
	return false;
}


// Cluster & Peak̏DBɏoKvꍇ
// `[WNX^̏񂪓B
//
// memo 2013.08.01
//
// [LPeakł]Ɣ肳ꂽ Peaḱ@Charge 
// dbac->updateCharge(peakId, z); ƂĕύX܂B
// z = -1 or z >= 2 邱Ƃ҂Ă܂B
// ύXȂl͎̂Ă܂B
// LƂ(MatrixɏoPeakj́@KCharge^ĂB

// ύX̂m点
// "Exclude Non-clustered Peaks" @I̎
// Ă܂p^[̂ŁAIF̕ύXĂ܂B

void NewAB3DLabeling::writeClusterInfotoDB(
	tDBAC* dbac, 
	std::map<Ptr, std::pair<int,Ptr>>& clusterChargeMap, 
	vPeak3D& detectedVPK3D,
	std::map<Ptr, int>& dbmap_Prof
){
	// DBsȂ܂Ԃ
	if (dbac == nullptr) {
		return;
	}
	std::map<Ptr, std::pair<int,Ptr>>::iterator mapItr;
	std::set<Ptr> usedCluster;

	// NX^[\Ȃs[NO邩ǂ
	const bool excludeNonClusteredPeaks = isSelected("ClusteringRemoveNonClustered");
	// mAC\݂̂ɍi荞ނǂ
	const bool monoisotopicPeaksOnly = isSelected("ClusteringOutputOnlyMonoisotopic");
	// Mz Ń\[g
	Cnum::sortByMZcenter(detectedVPK3D);

	BOOST_FOREACH(const Ptr pk, detectedVPK3D) {
		int peakId = dbmap_Prof[pk];
		mapItr = clusterChargeMap.find(pk);

		// NX^}bvɑ݂Ȃ̂ĂB
		if(mapItr == clusterChargeMap.end()) {
			if (!excludeNonClusteredPeaks) {
				// "Exclude Non-clustered Peaks" @I̎ -1 Ƃĕ⊮
				// Peak2dƂē邪ANX^̖̂ɂȂB
				dbac->updateCharge(peakId, -1);
			}
			continue;
		}

		Ptr monoIsoPtr = mapItr->second.second;
		int charge = mapItr->second.first;

		if(monoisotopicPeaksOnly) {
			// ݒʂŁA"Monoisotopic Peaks Only"Iꂽꍇ́A
			// mAC\̃s[N݂̗̂pB
			if(usedCluster.count(monoIsoPtr) == 0) {
				usedCluster.insert(monoIsoPtr);
				dbac->updateCharge(peakId, charge);
			}
		} else {
			dbac->updateCharge(peakId, charge);
		}
	}
}

int NewAB3DLabeling::verifyPeak(
	ELM::vXYP& xic, ELM::PKPT& foundPeak, std::vector<double>& intensities) {

	// XICRetention Timẽs[N
	const double peakMinRtSpan = getSettingMapDouble("VerificationPeakSpan");
	if (foundPeak.Rp.first - foundPeak.Lp.first < peakMinRtSpan) {
		// s[NRetention Timeݒlȉ̏ꍇ
		return 0;
	}

	// XICs[N̔l̍ŏl
	const double peakMinFWHM = getSettingMapDouble("VerificationPeakFWHM");
	if (foundPeak.fwhmVal < peakMinFWHM) {
		return 0;
	}

	// XICs[Nɑ݂N̍ő吔
	const int maxUndulations = getSettingMapInteger("VerificationPeakUndulations");
	int lInd = 0, rInd = 0;
	int numOfUndulations = 0;
	double xicInt = -1;
	bool isAscent = false;
	for (ELM::vXYP::iterator xicItr = xic.begin(); xicItr != xic.end(); ++xicItr) {
		if ((*xicItr).first == foundPeak.Lp.first) {
			lInd = xicItr - xic.begin();
		}
		if (xicInt == -1) {
			xicInt = (*xicItr).second;
		} else {
			if (lInd > 0 && rInd == 0 && xicInt > (*xicItr).second && isAscent) {
				// N݂ꍇ
				++numOfUndulations;
				if (numOfUndulations > maxUndulations) {
					return 0;
				}
			}
			isAscent = (xicInt < (*xicItr).second);
			xicInt = (*xicItr).second;
		}
		if ((*xicItr).first == foundPeak.Rp.first) {
			rInd = xicItr - xic.begin();
			break;
		}
	}

	const double snRatio = getSettingMapDouble("VerificationPeakSNRatio");
	const int dataPoints = getSettingMapInteger("VerificationNumOfDataPoints");
	const double baselineRatio = getSettingMapDouble("VerificationBaselineIntRatio");
	const bool baselineMethodMedian = (settingMap_["VerificationBaselineMethod"] != "baseline_mean");

	std::vector<double> tmpIntensities = std::vector<double>();
	double sum = 0.0;
	for (std::vector<double>::iterator itr = intensities.begin(); itr != intensities.end(); ++itr) {
		double intensity = *itr;
		if (intensity > 0) {
			tmpIntensities.push_back(intensity);
			if (!baselineMethodMedian) {
				sum += log10(intensity);
			}
		}
	}
	std::sort(tmpIntensities.begin(), tmpIntensities.end(), std::greater<double>());
	int lSize = tmpIntensities.size();
	double baseline;
	if (baselineMethodMedian) {
		if (lSize % 2 == 0) {
			baseline = (tmpIntensities[lSize/2-1] + tmpIntensities[lSize/2]) / 2;
		} else {
			baseline = tmpIntensities[lSize/2];
		}
	} else {
		baseline = pow(10, (sum/lSize));
	}

	double s = 0.0;
	for (std::vector<double>::iterator itr = tmpIntensities.begin(); itr != tmpIntensities.end(); ++itr) {
		s += (*itr - baseline) * (*itr - baseline);
	}

	double maxIntensity = *(tmpIntensities.begin());
	double stdev = std::sqrt(s/lSize);
	double sn;
	if (stdev > 0) {
		sn = (maxIntensity - baseline) / stdev;
	} else {
		sn = 0.0;
	}

	// ep[^̏𖞂Ă邩ǂ̔
	bool rightSn = (sn >= snRatio);
	bool rightDps = (lSize >= dataPoints);
	bool rightBaseline = (baseline / maxIntensity) <= baselineRatio;

	// SĂ̏𖞂ꍇ́A2ԂB
	// AS/NȊȌ𖞂āAS/N̏1/2𖞂Ăꍇ́A1ԂB
	if (rightDps && rightBaseline) {
		if (rightSn) return 2;
		if (sn >= snRatio / 2) return 1;
	}
	return 0;
}
