/*
 * Copyright (C) 2010-2011 Mtzky.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.log;

/**
 * <p>
 * Thread utilities.
 * </p>
 * 
 * @author mtzky
 */
public abstract class ThreadUtils {

	/**
	 * @return The calling {@link StackTraceElement}
	 */
	public static StackTraceElement getCallingStackTraceElement() {
		return getStackTrace(4);
	}

	/**
	 * @return The calling method name
	 */
	public static String getCallingMethodName() {
		return getStackTrace(4).getMethodName();
	}

	/**
	 * @return The calling class name
	 */
	public static String getCallingClassName() {
		return getStackTrace(4).getClassName();
	}

	/**
	 * @return Own {@link StackTraceElement}
	 */
	public static StackTraceElement getOwnStackTraceElement() {
		return getStackTrace(3);
	}

	/**
	 * @return Own method name
	 */
	public static String getOwnMethodName() {
		return getStackTrace(3).getMethodName();
	}

	/**
	 * @return Own class name
	 */
	public static String getOwnClassName() {
		return getStackTrace(3).getClassName();
	}

	/**
	 * @return Own simple class name
	 */
	public static String getOwnSimpleClassName() {
		final String className = getStackTrace(3).getClassName();
		return className.substring(className.lastIndexOf('.') + 1);
	}

	/**
	 * <p lang="ja">
	 * {@code level} に応じた {@link StackTraceElement} を返します。
	 * </p>
	 * <table border="1">
	 * <thead>
	 * <tr>
	 * <th>Level</th>
	 * <th>{@link StackTraceElement}</th>
	 * </tr>
	 * </thead><tbody>
	 * <tr>
	 * <th>0</th>
	 * <td>{@link Thread#getStackTrace()}</td>
	 * </tr>
	 * <tr>
	 * <th>1</th>
	 * <td>this method</td>
	 * </tr>
	 * <tr>
	 * <th>2</th>
	 * <td>calling method</td>
	 * </tr>
	 * </tbody>
	 * </table>
	 * 
	 * @param level
	 * @return {@link StackTraceElement}
	 */
	protected static StackTraceElement getStackTrace(final int level) {
		if (level < 0) {
			throw new IllegalArgumentException("level");
		}
		final StackTraceElement[] st = Thread.currentThread().getStackTrace();
		if (st.length < level) {
			throw new UnsupportedOperationException("StackTraceElement");
		}
		return st[level];
	}

}
