/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.reflect;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/**
 * <p>
 * Descriptors for {@link PropDesc Properties} ({@link Method Getter,
 * Setter} , and <strong>public</strong> {@link Field}) with the specified
 * {@link Annotation}.
 * </p>
 * 
 * @author mtzky
 * @since 0.1.1
 */
public class PropsDesc {

	private final PropDesc[] descs;

	/**
	 * @param clazz
	 * @param annotationClass
	 */
	public PropsDesc(final Class<?> clazz,
			final Class<? extends Annotation> annotationClass) {
		final Map<String, PropDesc> descs = new HashMap<String, PropDesc>();
		for (Class<?> c = clazz; (c != Object.class && c != null); c = c
				.getSuperclass()) {
			/* getter/setter methods */
			for (final Method m : c.getMethods()) {
				if (m.getAnnotation(annotationClass) == null) {
					continue;
				}

				final String mName = m.getName();
				final String name;
				if (mName.startsWith("get") || mName.startsWith("set")) {
					if (mName.length() < 4) {
						final String msg = "Invalid method name: " + m;
						throw new IllegalArgumentException(msg);
					}
					name = mName.substring(3, 4).toLowerCase(Locale.ENGLISH)
							+ mName.substring(4);
				} else if (mName.startsWith("is")) {
					if (mName.length() < 3) {
						final String msg = "Invalid method name: " + m;
						throw new IllegalArgumentException(msg);
					}
					name = mName.substring(2, 3).toLowerCase(Locale.ENGLISH)
							+ mName.substring(3);
				} else {
					final String msg = "NOT getter/setter method: " + m;
					throw new IllegalArgumentException(msg);
				}
				if (descs.containsKey(name)) {
					continue;
				}

				final PropertyDescriptor propDesc;
				try {
					propDesc = new PropertyDescriptor(name, clazz);
				} catch (final IntrospectionException e) {
					final String msg = "Requires to define the pair of getter and setter: ";
					throw new IllegalArgumentException(msg + name, e);
				}
				final PropDesc desc = new PropDesc(name);
				desc.setGetter(propDesc.getReadMethod());
				desc.setSetter(propDesc.getWriteMethod());
				desc.addAnnotations(m.getAnnotations());
				descs.put(name, desc);
			}

			/* public fields */
			for (final Field f : c.getFields()) {
				if (f.getAnnotation(annotationClass) == null) {
					continue;
				}

				final String name = f.getName();
				if (descs.containsKey(name)) {
					continue;
				}

				final PropDesc desc = new PropDesc(name);
				desc.setField(f);
				desc.addAnnotations(f.getAnnotations());
				descs.put(name, desc);
			}
		}
		this.descs = descs.values().toArray(new PropDesc[descs.size()]);
		descs.clear();
	}

	public PropDesc[] getDescriptors() {
		return descs;
	}

	/**
	 * @param <T>
	 * @param callback
	 *            {@link WrapCallback}
	 * @param type
	 *            for getting type; Don't pass {@code null}.
	 * @return An array of instance wrapping {@link PropDesc}
	 * @since 0.1.2
	 */
	@SuppressWarnings("unchecked")
	public <T> T[] wrap(final WrapCallback<T> callback, final T... type) {
		if (type == null) {
			throw new NullPointerException("type");
		}
		final Class<T> t = (Class<T>) type.getClass().getComponentType();
		final int len = descs.length;
		final T[] a = (T[]) Array.newInstance(t, len);
		for (int i = 0; i < len; i++) {
			a[i] = callback.create(descs[i]);
		}
		return a;
	}

	/**
	 * @author mtzky
	 * @since 0.1.2
	 */
	public interface WrapCallback<T> {
		T create(PropDesc desc);
	}

	/**
	 * <p>
	 * Find the {@link FindCallback#match(PropDesc) matched} {@link PropDesc
	 * property descriptor}.
	 * </p>
	 * 
	 * @param callback
	 *            {@link FindCallback}
	 * @return {@link PropDesc}
	 * @see FindCallback
	 */
	public PropDesc find(final FindCallback callback) {
		for (final PropDesc desc : descs) {
			if (callback.match(desc)) {
				return desc;
			}
		}
		return null;
	}

	/**
	 * @author mtzky
	 */
	public interface FindCallback {
		boolean match(PropDesc desc);
	}

}
