/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.reflect;

import static java.lang.String.*;
import static java.util.Locale.*;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>
 * Descriptors for {@link PropDesc Property} ({@link Method Getter, Setter} ,
 * and <strong>public</strong> {@link Field}).
 * </p>
 * <p>
 * Note for {@link #getAnnotation(Class)} and {@link #getAnnotations()} method:
 * Prevails the annotation on a getter method if same
 * {@link Annotation#annotationType() type} is defined.
 * </p>
 * 
 * @author mtzky
 * @since 0.1.1
 */
public class PropDesc {

	private final Method getter;
	private final Method setter;
	private final Field field;
	private final Map<Class<? extends Annotation>, Annotation> annotations = new HashMap<Class<? extends Annotation>, Annotation>();
	private final String name;

	/**
	 * @param name
	 * @param beanClass
	 */
	public PropDesc(final String name, final Class<?> beanClass) {
		if (name == null) {
			throw new NullPointerException("name");
		}
		if (beanClass == null) {
			throw new NullPointerException("beanClass");
		}
		this.name = name;
		final PropertyDescriptor desc;
		try {
			desc = new PropertyDescriptor(name, beanClass);
		} catch (final IntrospectionException e) {
			this.getter = null;
			this.setter = null;
			try {
				this.field = beanClass.getField(name);
			} catch (final Throwable t) {
				final String msg = "Requires the pair of getter and setter or the public field: "
						+ name;
				throw new IllegalArgumentException(msg, t);
			}
			addAnnotations(field.getAnnotations());
			return;
		}
		this.getter = desc.getReadMethod();
		this.setter = desc.getWriteMethod();
		this.field = null;
		addAnnotations(setter.getAnnotations());
		addAnnotations(getter.getAnnotations());
	}

	/**
	 * @param accessor
	 */
	public PropDesc(final Method accessor) {
		this(extractName(accessor), accessor.getDeclaringClass());
	}

	/**
	 * @param getter
	 * @param setter
	 */
	public PropDesc(final Method getter, final Method setter) {
		this(extractName(getter), getter, setter);
	}

	/**
	 * @param name
	 * @param getter
	 * @param setter
	 */
	public PropDesc(final String name, final Method getter, final Method setter) {
		if (name == null) {
			throw new NullPointerException("name");
		}
		if (getter == null) {
			throw new NullPointerException("getter");
		}
		if (setter == null) {
			throw new NullPointerException("setter");
		}
		this.name = name;
		this.getter = getter;
		this.setter = setter;
		this.field = null;
		addAnnotations(setter.getAnnotations());
		addAnnotations(getter.getAnnotations());
	}

	/**
	 * @param field
	 */
	public PropDesc(final Field field) {
		this(field.getName(), field);
	}

	/**
	 * @param name
	 * @param field
	 */
	public PropDesc(final String name, final Field field) {
		if (name == null) {
			throw new NullPointerException("name");
		}
		if (field == null) {
			throw new NullPointerException("field");
		}
		this.name = name;
		this.getter = null;
		this.setter = null;
		this.field = field;
		addAnnotations(field.getAnnotations());
	}

	private static String extractName(final Method accessor) {
		final String n = accessor.getName();
		final int len = n.length();
		if (len < 3) {
			throw new IllegalArgumentException("Invalid method name: "
					+ accessor);
		}
		final Class<?> returnType = accessor.getReturnType();
		if (returnType == boolean.class || returnType == Boolean.class) {
			/* Getter (is-method) */
			if (n.startsWith("is")) {
				return n.substring(2, 3).toLowerCase(ENGLISH) + n.substring(3);
			}
		}
		if (len < 4 || !(n.startsWith("get") || n.startsWith("set"))) {
			throw new IllegalArgumentException("Invalid method name: "
					+ accessor);
		}
		return n.substring(3, 4).toLowerCase(ENGLISH) + n.substring(4);
	}

	private void addAnnotations(final Annotation[] annotations) {
		for (final Annotation a : annotations) {
			this.annotations.put(a.annotationType(), a);
		}
	}

	/**
	 * <p>
	 * Gets a property value from the object. Returns {@code null} if an
	 * {@link Exception} occurred.
	 * </p>
	 * 
	 * @param <T>
	 * @param obj
	 *            the instance to get property value
	 * @return property value
	 * @throws InvocationTargetRuntimeException
	 *             if failed to get
	 */
	@SuppressWarnings("unchecked")
	public <T> T get(final Object obj) {
		try {
			return (T) (getter != null ? getter.invoke(obj) : field.get(obj));
		} catch (final Exception e) {
			final String fmt = "FAILED to get a value from the property '%s'";
			throw new InvocationTargetRuntimeException(format(fmt, name), e);
		}
	}

	/**
	 * <p>
	 * Sets a property value to the object. Returns {@code false} if an
	 * {@link Exception} occurred.
	 * </p>
	 * 
	 * @param obj
	 *            the instance to set property value
	 * @param value
	 * @throws InvocationTargetRuntimeException
	 *             if failed to set
	 */
	public void set(final Object obj, final Object value) {
		try {
			if (setter != null) {
				setter.invoke(obj, value);
			} else {
				field.set(obj, value);
			}
		} catch (final Exception e) {
			final String fmt = "FAILED to set [%s] to the property '%s'";
			final Object[] args = { value, name };
			throw new InvocationTargetRuntimeException(format(fmt, args), e);
		}
	}

	/**
	 * @return property type
	 */
	public Class<?> getType() {
		if (getter != null) {
			return getter.getReturnType();
		}
		return field.getType();
	}

	public String getName() {
		return name;
	}

	public boolean hasGetter() {
		return getter != null;
	}

	public boolean hasSetter() {
		return getter != null;
	}

	public boolean hasField() {
		return field != null;
	}

	public Annotation[] getAnnotations() {
		return annotations.values().toArray(new Annotation[annotations.size()]);
	}

	@SuppressWarnings("unchecked")
	public <T extends Annotation> T getAnnotation(final Class<T> annotationClass) {
		return (T) annotations.get(annotationClass);
	}

}
