/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene;

import static org.apache.lucene.util.Version.*;
import static org.mtzky.log.GenericMarker.*;
import static org.slf4j.LoggerFactory.*;

import java.io.IOException;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.Term;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.LockObtainFailedException;
import org.mtzky.io.Closable;
import org.mtzky.io.ClosingGuardian;
import org.mtzky.lucene.descriptor.LuceneDocumentDesc;
import org.slf4j.Logger;

/**
 * @param <E>
 *            {@code entity}
 * @see IndexWriter
 * @author mtzky
 */
public class LuceneIndexWriter<E> extends IndexWriter implements Closable {

	private static final Logger LOG = getLogger(LuceneIndexWriter.class);

	@SuppressWarnings("unused")
	private final Object guardian = new ClosingGuardian(this);
	private boolean closed = false;

	private final LuceneDocumentDesc<E> desc;

	/**
	 * @param d
	 *            {@link Document}
	 * @param desc
	 *            {@link LuceneDocumentDesc}
	 * @throws CorruptIndexException
	 * @throws LockObtainFailedException
	 * @throws IOException
	 */
	public LuceneIndexWriter(final Directory d, final LuceneDocumentDesc<E> desc)
			throws CorruptIndexException, LockObtainFailedException,
			IOException {
		super(d, new IndexWriterConfig(LUCENE_31, desc.getAnalyzer()));
		this.desc = desc;
	}

	/**
	 * @param entity
	 * @throws CorruptIndexException
	 * @throws IOException
	 * @see IndexWriter#addDocument(Document)
	 */
	public void addDocument(final E entity) throws CorruptIndexException,
			IOException {
		final Document doc = desc.createDocument(entity);
		addDocument(doc);
		if (LOG.isDebugEnabled(ADD)) {
			LOG.debug(ADD, "add a document [{}]", doc);
		}
	}

	/**
	 * <p>
	 * Updates {@link Document} by ID.
	 * </p>
	 * 
	 * @param entity
	 * @throws CorruptIndexException
	 * @throws IOException
	 * @see IndexWriter#updateDocument(Term, Document)
	 */
	public void updateDocument(final E entity) throws CorruptIndexException,
			IOException {
		final Document doc = desc.createDocument(entity);
		updateDocument(desc.getIdTermByEntity(entity), doc);
		if (LOG.isDebugEnabled(CHANGE)) {
			LOG.debug(CHANGE, "update a document [{}]", doc);
		}
	}

	/**
	 * <p>
	 * Deletes {@link Document} by ID.
	 * </p>
	 * 
	 * @param entity
	 * @throws CorruptIndexException
	 * @throws IOException
	 * @see IndexWriter#deleteDocuments(Term)
	 */
	public void deleteDocument(final E entity) throws CorruptIndexException,
			IOException {
		final Term term = desc.getIdTermByEntity(entity);
		deleteDocuments(term);
		if (LOG.isDebugEnabled(DELETE)) {
			LOG.debug(DELETE, "delete a document by term [{}]", term);
		}
	}

	/**
	 * <p>
	 * Deletes {@link Document} by ID value.
	 * </p>
	 * 
	 * @param value
	 * @throws CorruptIndexException
	 * @throws IOException
	 * @see IndexWriter#deleteDocuments(Term)
	 */
	public void deleteDocumentById(final Object value)
			throws CorruptIndexException, IOException {
		deleteDocuments(desc.getIdTermByValue(value));
		if (LOG.isDebugEnabled(DELETE)) {
			LOG.debug(DELETE, "delete a document by ID [{}]", value);
		}
	}

	@Override
	/* XXX 2011/02/12 mtzky @SuppressWarnings("all") */
	@SuppressWarnings("all")
	public boolean isClosed() {
		return this.closed;
	}

	@Override
	public void close() throws IOException {
		super.close();
		this.closed = true;
	}

}
