/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.normalizer;

import static org.mtzky.io.IOUtils.*;

import java.io.IOException;
import java.io.PushbackReader;
import java.io.Reader;
import java.util.LinkedList;
import java.util.Queue;

import org.mtzky.io.Closable;
import org.mtzky.io.ClosingGuardian;

/**
 * <p>
 * Normalizes the voiced and semi-voiced sound in Hiragana and Katakana.
 * </p>
 * <table border="1">
 * <caption>Voiced Sound Kana (U+3099|U+309B|U+FF9E)</caption><thead>
 * <tr>
 * <th rowspan="2">glyph</th>
 * <th colspan="2">code</th>
 * </tr>
 * <tr>
 * <th>before</th>
 * <th>after</th>
 * </tr>
 * </thead><tbody>
 * <tr>
 * <td>が</td>
 * <td>U+304B</td>
 * <td>U+304C</td>
 * </tr>
 * <tr>
 * <td>ガ</td>
 * <td>(U+30AB|U+FF76)</td>
 * <td>U+30AC</td>
 * </tr>
 * <tr>
 * <td>ぎ</td>
 * <td>U+304D</td>
 * <td>U+304E</td>
 * </tr>
 * <tr>
 * <td>ギ</td>
 * <td>(U+30AD|U+FF77)</td>
 * <td>U+30AE</td>
 * </tr>
 * <tr>
 * <td>ぐ</td>
 * <td>U+304F</td>
 * <td>U+3050</td>
 * </tr>
 * <tr>
 * <td>グ</td>
 * <td>(U+30AF|U+FF78)</td>
 * <td>U+30B0</td>
 * </tr>
 * <tr>
 * <td>げ</td>
 * <td>U+3051</td>
 * <td>U+3052</td>
 * </tr>
 * <tr>
 * <td>ゲ</td>
 * <td>(U+30B1|U+FF79)</td>
 * <td>U+30B2</td>
 * </tr>
 * <tr>
 * <td>ご</td>
 * <td>U+3053</td>
 * <td>U+3054</td>
 * </tr>
 * <tr>
 * <td>ゴ</td>
 * <td>(U+30B3|U+FF7A)</td>
 * <td>U+30B4</td>
 * </tr>
 * <tr>
 * <td>ざ</td>
 * <td>U+3055</td>
 * <td>U+3056</td>
 * </tr>
 * <tr>
 * <td>ザ</td>
 * <td>(U+30B5|U+FF7B)</td>
 * <td>U+30B6</td>
 * </tr>
 * <tr>
 * <td>じ</td>
 * <td>U+3057</td>
 * <td>U+3058</td>
 * </tr>
 * <tr>
 * <td>ジ</td>
 * <td>(U+30B7|U+FF7C)</td>
 * <td>U+30B8</td>
 * </tr>
 * <tr>
 * <td>ず</td>
 * <td>U+3059</td>
 * <td>U+305A</td>
 * </tr>
 * <tr>
 * <td>ズ</td>
 * <td>(U+30B9|U+FF7D)</td>
 * <td>U+30BA</td>
 * </tr>
 * <tr>
 * <td>ぜ</td>
 * <td>U+305B</td>
 * <td>U+305C</td>
 * </tr>
 * <tr>
 * <td>ゼ</td>
 * <td>(U+30BB|U+FF7E)</td>
 * <td>U+30BC</td>
 * </tr>
 * <tr>
 * <td>ぞ</td>
 * <td>U+305D</td>
 * <td>U+305E</td>
 * </tr>
 * <tr>
 * <td>ゾ</td>
 * <td>(U+30BD|U+FF7F)</td>
 * <td>U+30BE</td>
 * </tr>
 * <tr>
 * <td>だ</td>
 * <td>U+305F</td>
 * <td>U+3060</td>
 * </tr>
 * <tr>
 * <td>ダ</td>
 * <td>(U+30BF|U+FF80)</td>
 * <td>U+30C0</td>
 * </tr>
 * <tr>
 * <td>ぢ</td>
 * <td>U+3061</td>
 * <td>U+3062</td>
 * </tr>
 * <tr>
 * <td>ヂ</td>
 * <td>(U+30C1|U+FF81)</td>
 * <td>U+30C2</td>
 * </tr>
 * <tr>
 * <td>づ</td>
 * <td>U+3064</td>
 * <td>U+3065</td>
 * </tr>
 * <tr>
 * <td>ヅ</td>
 * <td>(U+30C4|U+FF82)</td>
 * <td>U+30C5</td>
 * </tr>
 * <tr>
 * <td>で</td>
 * <td>U+3066</td>
 * <td>U+3067</td>
 * </tr>
 * <tr>
 * <td>デ</td>
 * <td>(U+30C6|U+FF83)</td>
 * <td>U+30C7</td>
 * </tr>
 * <tr>
 * <td>ど</td>
 * <td>U+3068</td>
 * <td>U+3069</td>
 * </tr>
 * <tr>
 * <td>ド</td>
 * <td>(U+30C8|U+FF84)</td>
 * <td>U+30C9</td>
 * </tr>
 * <tr>
 * <td>ば</td>
 * <td>U+306F</td>
 * <td>U+3070</td>
 * </tr>
 * <tr>
 * <td>バ</td>
 * <td>(U+30CF|U+FF8A)</td>
 * <td>U+30D0</td>
 * </tr>
 * <tr>
 * <td>び</td>
 * <td>U+3072</td>
 * <td>U+3073</td>
 * </tr>
 * <tr>
 * <td>ビ</td>
 * <td>(U+30D2|U+FF8B)</td>
 * <td>U+30D3</td>
 * </tr>
 * <tr>
 * <td>ぶ</td>
 * <td>U+3075</td>
 * <td>U+3076</td>
 * </tr>
 * <tr>
 * <td>ブ</td>
 * <td>(U+30D5|U+FF8C)</td>
 * <td>U+30D6</td>
 * </tr>
 * <tr>
 * <td>べ</td>
 * <td>U+3078</td>
 * <td>U+3079</td>
 * </tr>
 * <tr>
 * <td>ベ</td>
 * <td>(U+30D8|U+FF8D)</td>
 * <td>U+30D9</td>
 * </tr>
 * <tr>
 * <td>ぼ</td>
 * <td>U+307B</td>
 * <td>U+307C</td>
 * </tr>
 * <tr>
 * <td>ボ</td>
 * <td>(U+30DB|U+FF8E)</td>
 * <td>U+30DC</td>
 * </tr>
 * </tbody>
 * </table>
 * <table border="1">
 * <caption>Semi-Voiced Sound Kana (U+309A|U+309C|U+FF9F)</caption><thead>
 * <tr>
 * <th rowspan="2">glyph</th>
 * <th colspan="2">code</th>
 * </tr>
 * <tr>
 * <th>before</th>
 * <th>after</th>
 * </tr>
 * </thead><tbody>
 * <tr>
 * <td>ぱ</td>
 * <td>U+306F</td>
 * <td>U+3071</td>
 * </tr>
 * <tr>
 * <td>パ</td>
 * <td>(U+30CF|U+FF8A)</td>
 * <td>U+30D1</td>
 * </tr>
 * <tr>
 * <td>ぴ</td>
 * <td>U+3072</td>
 * <td>U+3074</td>
 * </tr>
 * <tr>
 * <td>ピ</td>
 * <td>(U+30D2|U+FF8B)</td>
 * <td>U+30D4</td>
 * </tr>
 * <tr>
 * <td>ぷ</td>
 * <td>U+3075</td>
 * <td>U+3077</td>
 * </tr>
 * <tr>
 * <td>プ</td>
 * <td>(U+30D5|U+FF8C)</td>
 * <td>U+30D7</td>
 * </tr>
 * <tr>
 * <td>ぺ</td>
 * <td>U+3078</td>
 * <td>U+307A</td>
 * </tr>
 * <tr>
 * <td>ペ</td>
 * <td>(U+30D8|U+FF8D)</td>
 * <td>U+30DA</td>
 * </tr>
 * <tr>
 * <td>ぽ</td>
 * <td>U+307B</td>
 * <td>U+307D</td>
 * </tr>
 * <tr>
 * <td>ポ</td>
 * <td>(U+30DB|U+FF8E)</td>
 * <td>U+30DD</td>
 * </tr>
 * </tbody>
 * </table>
 * <table border="1">
 * <caption>Phonetic Transcription Kana (U+309A|U+309C|U+FF9F)</caption><thead>
 * <tr>
 * <th rowspan="2">glyph</th>
 * <th colspan="2">code</th>
 * <th rowspan="2">note</th>
 * </tr>
 * <tr>
 * <th>before</th>
 * <th>after</th>
 * </tr>
 * </thead><tbody>
 * <tr>
 * <td>う</td>
 * <td>U+3046</td>
 * <td>U+3046</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>ウ</td>
 * <td>(U+30A6|U+FF73)</td>
 * <td>U+30A6</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>か</td>
 * <td>U+304B</td>
 * <td>U+304B</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>カ</td>
 * <td>(U+30AB|U+FF76)</td>
 * <td>U+30AB</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>き</td>
 * <td>U+304D</td>
 * <td>U+304D</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>キ</td>
 * <td>(U+30AD|U+FF77)</td>
 * <td>U+30AD</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>く</td>
 * <td>U+304F</td>
 * <td>U+304F</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>ク</td>
 * <td>(U+30AF|U+FF78)</td>
 * <td>U+30AF</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>け</td>
 * <td>U+3051</td>
 * <td>U+3051</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>ケ</td>
 * <td>(U+30B1|U+FF79)</td>
 * <td>U+30B1</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>こ</td>
 * <td>U+3053</td>
 * <td>U+3053</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>コ</td>
 * <td>(U+30B3|U+FF7A)</td>
 * <td>U+30B3</td>
 * <td>nasal sound</td>
 * </tr>
 * <tr>
 * <td>つぁ</td>
 * <td>U+3055</td>
 * <td>U+3064 U+3041</td>
 * <td>vernacular sound of Edo</td>
 * </tr>
 * <tr>
 * <td>ツァ</td>
 * <td>(U+30B5|U+FF7B)</td>
 * <td>U+30C4 U+30A1</td>
 * <td>vernacular sound of Edo</td>
 * </tr>
 * <tr>
 * <td>つぃ</td>
 * <td>U+3057</td>
 * <td>U+3064 U+3043</td>
 * <td>vernacular sound of Edo</td>
 * </tr>
 * <tr>
 * <td>ツィ</td>
 * <td>(U+30B7|U+FF7C)</td>
 * <td>U+30C4 U+30A3</td>
 * <td>vernacular sound of Edo</td>
 * </tr>
 * <tr>
 * <td>つぅ</td>
 * <td>U+3059</td>
 * <td>U+3064 U+3045</td>
 * <td>vernacular sound of Edo</td>
 * </tr>
 * <tr>
 * <td>ツゥ</td>
 * <td>(U+30B9|U+FF7D)</td>
 * <td>U+30C4 U+30A5</td>
 * <td>vernacular sound of Edo</td>
 * </tr>
 * <tr>
 * <td>つぇ</td>
 * <td>U+305B</td>
 * <td>U+3064 U+3047</td>
 * <td>vernacular sound of Edo (<strong>NOT</strong> Ainu languages 'チェ')</td>
 * </tr>
 * <tr>
 * <td>ツェ</td>
 * <td>(U+30BB|U+FF7E)</td>
 * <td>U+30C4 U+30A7</td>
 * <td>vernacular sound of Edo (<strong>NOT</strong> Ainu languages 'チェ')</td>
 * </tr>
 * <tr>
 * <td>つぉ</td>
 * <td>U+305D</td>
 * <td>U+3064 U+3049</td>
 * <td>vernacular sound of Edo</td>
 * </tr>
 * <tr>
 * <td>ツォ</td>
 * <td>(U+30BD|U+FF7F)</td>
 * <td>U+30C4 U+30A9</td>
 * <td>vernacular sound of Edo</td>
 * </tr>
 * <tr>
 * <td>とぅ</td>
 * <td>U+3064</td>
 * <td>U+3068 U+3045</td>
 * <td>vernacular sound of Ainu languages</td>
 * </tr>
 * <tr>
 * <td>トゥ</td>
 * <td>(U+30C4|U+FF82)</td>
 * <td>U+30C8 U+30A5</td>
 * <td>vernacular sound of Ainu languages</td>
 * </tr>
 * <tr>
 * <td>とぅ</td>
 * <td>U+3068</td>
 * <td>U+3068 U+3045</td>
 * <td>vernacular sound of Ainu languages</td>
 * </tr>
 * <tr>
 * <td>トゥ</td>
 * <td>(U+30C8|U+FF84)</td>
 * <td>U+30C8 U+30A5</td>
 * <td>vernacular sound of Ainu languages</td>
 * </tr>
 * <tr>
 * <td>ら</td>
 * <td>U+3089</td>
 * <td>U+3089</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * <tr>
 * <td>ラ</td>
 * <td>(U+30E9|U+FF97)</td>
 * <td>U+30E9</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * <tr>
 * <td>り</td>
 * <td>U+308A</td>
 * <td>U+308A</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * <tr>
 * <td>リ</td>
 * <td>(U+30EA|U+FF98)</td>
 * <td>U+30EA</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * <tr>
 * <td>る</td>
 * <td>U+308B</td>
 * <td>U+308B</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * <tr>
 * <td>ル</td>
 * <td>(U+30EB|U+FF99)</td>
 * <td>U+30EB</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * <tr>
 * <td>れ</td>
 * <td>U+308C</td>
 * <td>U+308C</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * <tr>
 * <td>レ</td>
 * <td>(U+30EC|U+FF9A)</td>
 * <td>U+30EC</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * <tr>
 * <td>ろ</td>
 * <td>U+308D</td>
 * <td>U+308D</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * <tr>
 * <td>ロ</td>
 * <td>(U+30ED|U+FF9B)</td>
 * <td>U+30ED</td>
 * <td>alveolar lateral approximant</td>
 * </tr>
 * </tbody>
 * </table>
 * 
 * @author mtzky
 */
public class VoicedSoundKanaNormalizer extends Reader implements Closable {

	@SuppressWarnings("unused")
	private final Object guardian = new ClosingGuardian(this);
	private boolean close = false;

	private final Queue<Character> buf = new LinkedList<Character>();
	private final PushbackReader in;

	public VoicedSoundKanaNormalizer(final Reader in) {
		this(toPushbackReader(in));
	}

	public VoicedSoundKanaNormalizer(final PushbackReader in) {
		super(in);
		this.in = in;
	}

	@Override
	public int read(final char[] cbuf, final int off, final int len)
			throws IOException {
		final int limit = off + len;
		int read = 0;
		for (int i = off; i < limit; i++) {
			if (!buf.isEmpty()) {
				cbuf[i] = buf.poll();
				read++;
				continue;
			}
			final int c1 = in.read();
			if (c1 < 0) {
				break;
			}
			read++;
			final int c2 = in.read();
			if (c2 < 0) {
				cbuf[i] = (char) c1;
				break;
			}
			if (c2 == 0xFF9E || c2 == 0x309B || c2 == 0x3099) {
				/* VOICED SOUND MARK */
				switch (c1) {
				case 'か':
					cbuf[i] = 'が';
					continue;
				case 'カ':
				case 'ｶ':
					cbuf[i] = 'ガ';
					continue;
				case 'き':
					cbuf[i] = 'ぎ';
					continue;
				case 'キ':
				case 'ｷ':
					cbuf[i] = 'ギ';
					continue;
				case 'く':
					cbuf[i] = 'ぐ';
					continue;
				case 'ク':
				case 'ｸ':
					cbuf[i] = 'グ';
					continue;
				case 'け':
					cbuf[i] = 'げ';
					continue;
				case 'ケ':
				case 'ｹ':
					cbuf[i] = 'ゲ';
					continue;
				case 'こ':
					cbuf[i] = 'ご';
					continue;
				case 'コ':
				case 'ｺ':
					cbuf[i] = 'ゴ';
					continue;
				case 'さ':
					cbuf[i] = 'ざ';
					continue;
				case 'サ':
				case 'ｻ':
					cbuf[i] = 'ザ';
					continue;
				case 'し':
					cbuf[i] = 'じ';
					continue;
				case 'シ':
				case 'ｼ':
					cbuf[i] = 'ジ';
					continue;
				case 'す':
					cbuf[i] = 'ず';
					continue;
				case 'ス':
				case 'ｽ':
					cbuf[i] = 'ズ';
					continue;
				case 'せ':
					cbuf[i] = 'ぜ';
					continue;
				case 'セ':
				case 'ｾ':
					cbuf[i] = 'ゼ';
					continue;
				case 'そ':
					cbuf[i] = 'ぞ';
					continue;
				case 'ソ':
				case 'ｿ':
					cbuf[i] = 'ゾ';
					continue;
				case 'た':
					cbuf[i] = 'だ';
					continue;
				case 'タ':
				case 'ﾀ':
					cbuf[i] = 'ダ';
					continue;
				case 'ち':
					cbuf[i] = 'ぢ';
					continue;
				case 'チ':
				case 'ﾁ':
					cbuf[i] = 'ヂ';
					continue;
				case 'つ':
					cbuf[i] = 'づ';
					continue;
				case 'ツ':
				case 'ﾂ':
					cbuf[i] = 'ヅ';
					continue;
				case 'て':
					cbuf[i] = 'で';
					continue;
				case 'テ':
				case 'ﾃ':
					cbuf[i] = 'デ';
					continue;
				case 'と':
					cbuf[i] = 'ど';
					continue;
				case 'ト':
				case 'ﾄ':
					cbuf[i] = 'ド';
					continue;
				case 'は':
					cbuf[i] = 'ば';
					continue;
				case 'ハ':
				case 'ﾊ':
					cbuf[i] = 'バ';
					continue;
				case 'ひ':
					cbuf[i] = 'び';
					continue;
				case 'ヒ':
				case 'ﾋ':
					cbuf[i] = 'ビ';
					continue;
				case 'ふ':
					cbuf[i] = 'ぶ';
					continue;
				case 'フ':
				case 'ﾌ':
					cbuf[i] = 'ブ';
					continue;
				case 'へ':
					cbuf[i] = 'べ';
					continue;
				case 'ヘ':
				case 'ﾍ':
					cbuf[i] = 'ベ';
					continue;
				case 'ほ':
					cbuf[i] = 'ぼ';
					continue;
				case 'ホ':
				case 'ﾎ':
					cbuf[i] = 'ボ';
					continue;
				}
			}
			if (c2 == 0xFF9F || c2 == 0x309C || c2 == 0x309A) {
				/* SEMI-VOICED SOUND MARK */
				switch (c1) {
				case 'は':
					cbuf[i] = 'ぱ';
					continue;
				case 'ハ':
				case 'ﾊ':
					cbuf[i] = 'パ';
					continue;
				case 'ひ':
					cbuf[i] = 'ぴ';
					continue;
				case 'ヒ':
				case 'ﾋ':
					cbuf[i] = 'ピ';
					continue;
				case 'ふ':
					cbuf[i] = 'ぷ';
					continue;
				case 'フ':
				case 'ﾌ':
					cbuf[i] = 'プ';
					continue;
				case 'へ':
					cbuf[i] = 'ぺ';
					continue;
				case 'ヘ':
				case 'ﾍ':
					cbuf[i] = 'ペ';
					continue;
				case 'ほ':
					cbuf[i] = 'ぽ';
					continue;
				case 'ホ':
				case 'ﾎ':
					cbuf[i] = 'ポ';
					continue;

					/* Phonetic transcription */
				case 'う':
					cbuf[i] = 'う';
					continue;
				case 'ウ':
				case 'ｳ':
					cbuf[i] = 'ウ';
					continue;
				case 'か':
					cbuf[i] = 'か';
					continue;
				case 'カ':
				case 'ｶ':
					cbuf[i] = 'カ';
					continue;
				case 'き':
					cbuf[i] = 'き';
					continue;
				case 'キ':
				case 'ｷ':
					cbuf[i] = 'キ';
					continue;
				case 'く':
					cbuf[i] = 'く';
					continue;
				case 'ク':
				case 'ｸ':
					cbuf[i] = 'ク';
					continue;
				case 'け':
					cbuf[i] = 'け';
					continue;
				case 'ケ':
				case 'ｹ':
					cbuf[i] = 'ケ';
					continue;
				case 'こ':
					cbuf[i] = 'こ';
					continue;
				case 'コ':
				case 'ｺ':
					cbuf[i] = 'コ';
					continue;
				case 'さ':
					cbuf[i] = 'つ';
					buf.offer('ぁ');
					continue;
				case 'サ':
				case 'ｻ':
					cbuf[i] = 'ツ';
					buf.offer('ァ');
					continue;
				case 'し':
					cbuf[i] = 'つ';
					buf.offer('ぃ');
					continue;
				case 'シ':
				case 'ｼ':
					cbuf[i] = 'ツ';
					buf.offer('ィ');
					continue;
				case 'す':
					cbuf[i] = 'つ';
					buf.offer('ぅ');
					continue;
				case 'ス':
				case 'ｽ':
					cbuf[i] = 'ツ';
					buf.offer('ゥ');
					continue;
				case 'せ':
					cbuf[i] = 'つ';
					buf.offer('ぇ');
					continue;
				case 'セ':
				case 'ｾ':
					cbuf[i] = 'ツ';
					buf.offer('ェ');
					continue;
				case 'そ':
					cbuf[i] = 'つ';
					buf.offer('ぉ');
					continue;
				case 'ソ':
				case 'ｿ':
					cbuf[i] = 'ツ';
					buf.offer('ォ');
					continue;
				case 'つ':
					cbuf[i] = 'と';
					buf.offer('ぅ');
					continue;
				case 'ツ':
				case 'ﾂ':
					cbuf[i] = 'ト';
					buf.offer('ゥ');
					continue;
				case 'と':
					cbuf[i] = 'と';
					buf.offer('ぅ');
					continue;
				case 'ト':
				case 'ﾄ':
					cbuf[i] = 'ト';
					buf.offer('ゥ');
					continue;
				case 'ら':
					cbuf[i] = 'ら';
					continue;
				case 'ラ':
				case 'ﾗ':
					cbuf[i] = 'ラ';
					continue;
				case 'り':
					cbuf[i] = 'り';
					continue;
				case 'リ':
				case 'ﾘ':
					cbuf[i] = 'リ';
					continue;
				case 'る':
					cbuf[i] = 'る';
					continue;
				case 'ル':
				case 'ﾙ':
					cbuf[i] = 'ル';
					continue;
				case 'れ':
					cbuf[i] = 'れ';
					continue;
				case 'レ':
				case 'ﾚ':
					cbuf[i] = 'レ';
					continue;
				case 'ろ':
					cbuf[i] = 'ろ';
					continue;
				case 'ロ':
				case 'ﾛ':
					cbuf[i] = 'ロ';
					continue;
				}
			}
			cbuf[i] = (char) c1;
			in.unread(c2);
		}
		return read == 0 ? -1 : read;
	}

	@Override
	public boolean isClosed() {
		return close;
	}

	@Override
	public void close() throws IOException {
		in.close();
		close = true;
	}

}
