/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene;

import static org.apache.lucene.search.BooleanClause.Occur.*;
import static org.apache.lucene.search.SortField.*;

import java.util.ArrayList;
import java.util.List;

import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Filter;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.BooleanClause.Occur;

/**
 * @author mtzky
 * @see LuceneIndexSearcher#find(LuceneIndexSearcherRequest)
 */
public abstract class AbstractIndexSearcherRequest implements
		LuceneIndexSearcherRequest {

	private BooleanQuery query = null;
	private int limit = DEFAULT_LIMIT;
	private int offset = 0;
	private Filter filter = null;
	private final List<SortField> sortFields = new ArrayList<SortField>();

	@Override
	public Query getQuery() {
		if (query == null) {
			return DEFAULT_QUERY;
		}
		return query;
	}

	@Override
	public Filter getFilter() {
		return filter;
	}

	@Override
	public int getLimit() {
		return limit;
	}

	@Override
	public int getOffset() {
		return offset;
	}

	@Override
	public Sort getSort() {
		final int len = sortFields.size();
		if (0 < len) {
			return new Sort(sortFields.toArray(new SortField[len]));
		}
		return null;
	}

	/**
	 * <p>
	 * Adds {@link Query} as the {@link Occur#MUST AND} condition.
	 * </p>
	 * 
	 * @param query
	 *            {@link Query}
	 * @see #addQuery(Query, org.apache.lucene.search.BooleanClause.Occur)
	 */
	protected void addQuery(final Query query) {
		addQuery(query, MUST);
	}

	/**
	 * <p>
	 * Adds {@link Query}.
	 * </p>
	 * 
	 * @param query
	 *            {@link Query}
	 * @param occur
	 *            {@link Occur}
	 */
	protected void addQuery(final Query query, final Occur occur) {
		if (this.query == null) {
			this.query = new BooleanQuery();
		}
		this.query.add(query, occur);
	}

	/**
	 * <p>
	 * Sets {@link Filter}.
	 * </p>
	 * 
	 * @param filter
	 *            {@link Filter}
	 */
	protected void setFilter(final Filter filter) {
		this.filter = filter;
	}

	/**
	 * <p>
	 * Sets {@code limit}.
	 * </p>
	 * 
	 * @param limit
	 */
	protected void setLimit(final int limit) {
		this.limit = limit;
	}

	/**
	 * <p>
	 * Sets {@code offset}.
	 * </p>
	 * 
	 * @param offset
	 */
	protected void setOffset(final int offset) {
		this.offset = offset;
	}

	/**
	 * <p>
	 * Adds {@link SortField sort field}s as natural order.
	 * </p>
	 * 
	 * @param fields
	 *            Name of field to sort by.
	 */
	protected void addSort(final String... fields) {
		for (final String field : fields) {
			sortFields.add(new SortField(field, STRING));
		}
	}

	/**
	 * <p>
	 * Adds {@link SortField sort field}s.
	 * </p>
	 * 
	 * @param field
	 *            Name of field to sort by.
	 * @param reverse
	 *            {@code true} if natural order should be reversed.
	 */
	protected void addSort(final String field, final boolean reverse) {
		sortFields.add(new SortField(field, STRING, reverse));
	}

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		sb.append("query=[").append(getQuery()).append("]");
		if (filter != null) {
			sb.append(", filter=[").append(filter).append("]");
		}
		sb.append(", limit=").append(limit);
		sb.append(", offset=").append(offset);
		if (!sortFields.isEmpty()) {
			sb.append(", sort=").append(sortFields);
		}
		return sb.toString();
	}

}
