/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene;

import static org.apache.lucene.index.IndexReader.*;
import static org.mtzky.log.GenericMarker.*;
import static org.mtzky.log.ThreadUtils.*;
import static org.slf4j.LoggerFactory.*;

import java.io.IOException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.store.Directory;
import org.mtzky.io.Closable;
import org.mtzky.io.ClosingGuardian;
import org.mtzky.lucene.descriptor.LuceneDocumentDesc;
import org.mtzky.lucene.query.LuceneQueryParser;
import org.slf4j.Logger;

/**
 * @param <E>
 *            {@code entity}
 * @author mtzky
 */
public class LuceneIndexSearcher<E> extends IndexSearcher implements Closable {

	private static final Logger LOG = getLogger(LuceneIndexSearcher.class);

	@SuppressWarnings("unused")
	private final Object guardian = new ClosingGuardian(this);
	private boolean closed = false;
	private final Lock delayCloseLock = new ReentrantLock();
	private boolean closeWhenDone = false;
	private int usageCount = 0;
	private final long version;

	private final Analyzer analyzer;
	private final LuceneDocumentDesc<E> desc;
	private final String defaultField;

	/**
	 * @param d
	 *            {@link Document}
	 * @param desc
	 *            {@link LuceneDocumentDesc}
	 * @throws CorruptIndexException
	 * @throws IOException
	 */
	public LuceneIndexSearcher(final Directory d,
			final LuceneDocumentDesc<E> desc) throws CorruptIndexException,
			IOException {
		super(d, true);
		this.analyzer = desc.getAnalyzer();
		this.desc = desc;
		this.defaultField = desc.getIdName();
		this.version = getCurrentVersion(d);
	}

	Query parse(final String query) throws ParseException {
		return new LuceneQueryParser(defaultField, analyzer).parse(query);
	}

	/**
	 * <p>
	 * Returns {@link FluentIndexSearcherRequest} as the fluent interface. For
	 * example:
	 * </p>
	 * 
	 * <pre>
	 * {@link LuceneIndexSearcherResponse} res = {@link LuceneIndexSearcher searcher}.findBy()
	 * 	.query("text: lucene")
	 * 	.query("text: mtzky", {@link Occur#MUST_NOT})
	 * 	.filter(...)
	 * 	.limit(100)
	 * 	.sort("id", true)
	 * 	.execute();
	 * </pre>
	 * 
	 * @return {@link FluentIndexSearcherRequest}
	 */
	public FluentIndexSearcherRequest<E> findBy() {
		return new FluentIndexSearcherRequest<E>(this);
	}

	/**
	 * <p>
	 * Finds the top {@link LuceneIndexSearcherRequest#DEFAULT_LIMIT} hits for
	 * {@code
	 * query}.
	 * </p>
	 * 
	 * @param query
	 *            to search
	 * @return {@link LuceneIndexSearcherResponse Entities} hit by {@code query}
	 * @throws CorruptIndexException
	 * @throws IOException
	 * @throws ParseException
	 */
	public LuceneIndexSearcherResponse<E> find(final String query)
			throws CorruptIndexException, IOException, ParseException {
		return findBy().query(query).execute();
	}

	/**
	 * <p>
	 * Finds the top {@link LuceneIndexSearcherRequest#DEFAULT_LIMIT} hits for
	 * {@code
	 * query}.
	 * </p>
	 * 
	 * @param query
	 *            to search
	 * @return {@link LuceneIndexSearcherResponse Entities} hit by {@code query}
	 * @throws CorruptIndexException
	 * @throws IOException
	 */
	public LuceneIndexSearcherResponse<E> find(final Query query)
			throws CorruptIndexException, IOException {
		return findBy().query(query).execute();
	}

	/**
	 * <p>
	 * Finds by {@link LuceneIndexSearcherRequest}.
	 * </p>
	 * 
	 * @param req
	 * @return {@link LuceneIndexSearcherResponse Entities} hit by
	 *         {@link LuceneIndexSearcherRequest}
	 * @throws CorruptIndexException
	 * @throws IOException
	 * @see #findBy()
	 */
	public LuceneIndexSearcherResponse<E> find(
			final LuceneIndexSearcherRequest req) throws CorruptIndexException,
			IOException {
		final Sort sort = req.getSort();
		final int limit = req.getLimit();
		final int offset = req.getOffset();
		final int maxIndex = limit + offset;
		final TopDocs topDocs;
		if (sort == null) {
			topDocs = search(req.getQuery(), req.getFilter(), maxIndex);
		} else {
			topDocs = search(req.getQuery(), req.getFilter(), maxIndex, sort);
		}

		final ScoreDoc[] scoreDocs = topDocs.scoreDocs;
		final int len = scoreDocs.length;
		final int size = len - offset;
		final LuceneIndexSearcherResponse<E> res = new LuceneIndexSearcherResponse<E>(
				size < 0 ? 0 : size);
		for (int i = offset; i < len; i++) {
			res.add(desc.createEntity(doc(scoreDocs[i].doc)));
		}
		res.setLimit(limit);
		res.setOffset(offset);
		res.setTotalHits(topDocs.totalHits);

		if (LOG.isDebugEnabled(LIST)) {
			final String fmg = "find request=[{}], response=[{}]";
			LOG.debug(LIST, fmg, req, res);
		}
		return res;
	}

	/**
	 * <p>
	 * Finds the specified (top one) hit by ID field with {@code value}. If not
	 * hit, returns {@code null}.
	 * </p>
	 * 
	 * @param value
	 *            to search
	 * @return entity hit by ID field with {@code value}
	 * @throws CorruptIndexException
	 * @throws IOException
	 */
	public E findById(final Object value) throws CorruptIndexException,
			IOException {
		final Query query = new TermQuery(desc.getIdTermByValue(value));
		final ScoreDoc[] scoreDocs = search(query, 1).scoreDocs;
		if (LOG.isDebugEnabled(SHOW)) {
			final String fmg = "find by id [{}], hits={}";
			LOG.debug(SHOW, fmg, value, scoreDocs.length);
		}
		if (scoreDocs.length < 1) {
			return null;
		}
		return desc.createEntity(doc(scoreDocs[0].doc));
	}

	/**
	 * <p>
	 * This should be called whenever this instances is passed as a new
	 * IndexSearcher. Only when each call to {@link #open()} is balanced with a
	 * call to {@link #close()}, and {@link #closeWhenDone()} has been called,
	 * will super.close() be called.
	 * </p>
	 */
	protected void open() {
		try {
			delayCloseLock.lock();
			if (closeWhenDone) {
				throw new IllegalStateException(
						"closeWhenDone() already called");
			}
			usageCount++;
		} finally {
			delayCloseLock.unlock();
		}
		if (LOG.isDebugEnabled(OPEN)) {
			final String fmg = "[version: {}] open; usageCount: {}";
			LOG.debug(OPEN, fmg, version, usageCount);
		}
	}

	@Override
	public boolean isClosed() {
		return this.closed;
	}

	@Override
	public void close() throws IOException {
		if (ClosingGuardian.class.getName().equals(getCallingClassName())) {
			/* closes forcibly if finalizer */
			super.close();
			this.closed = true;
			if (LOG.isWarnEnabled(CLOSE)) {
				final String fmt = "[version: {}] close forcibly; usageCount: {}";
				LOG.warn(CLOSE, fmt, version, usageCount);
			}
			return;
		}
		boolean doClose;
		try {
			delayCloseLock.lock();
			doClose = (--usageCount <= 0 && closeWhenDone);
		} finally {
			delayCloseLock.unlock();
		}
		if (doClose) {
			super.close();
			this.closed = true;
			if (LOG.isDebugEnabled(CLOSE)) {
				LOG.debug(CLOSE, "[version: {}] close", version);
			}
		} else {
			if (LOG.isDebugEnabled(CANCEL)) {
				final String fmg = "[version: {}] cancel to close; usageCount: {}, closeWhenDone: {}";
				final Object[] args = { version, usageCount, closeWhenDone };
				LOG.debug(CLOSE, fmg, args);
			}
		}
	}

	/**
	 * <p>
	 * Signals that this instance may really close when all {@link #open()}
	 * calls have been balanced with a call to {@link #close()}.
	 * </p>
	 * 
	 * @throws IOException
	 */
	public void closeWhenDone() throws IOException {
		final boolean doClose;
		try {
			delayCloseLock.lock();
			if (closeWhenDone) {
				if (LOG.isDebugEnabled(CANCEL)) {
					final String fmg = "[version: {}] closeWhenDone() already called; usageCount: {}";
					LOG.debug(CLOSE, fmg, version, usageCount);
				}
			} else {
				closeWhenDone = true;
			}
			doClose = (usageCount <= 0);
		} finally {
			delayCloseLock.unlock();
		}
		if (LOG.isDebugEnabled(DESTROY)) {
			final String fmt = "[version: {}] close when done; usageCount: {}";
			LOG.debug(DESTROY, fmt, version, usageCount);
		}
		if (doClose) {
			super.close();
			this.closed = true;
			if (LOG.isDebugEnabled(CLOSE)) {
				LOG.debug(CLOSE, "[version: {}] close", version);
			}
		}
	}

	@Override
	public String toString() {
		return "LuceneIndexSearcher[version: " + version + "]";
	}

}
