/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene;

import static org.mtzky.lucene.LuceneIndexSearcherRequest.*;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

/**
 * <p>
 * The response for {@link LuceneIndexSearcher#find(LuceneIndexSearcherRequest)}
 * .
 * </p>
 * 
 * @param <E>
 *            entity
 * @author mtzky
 */
public class LuceneIndexSearcherResponse<E> implements List<E>, Serializable {

	private static final long serialVersionUID = -1925791784907186977L;

	private List<E> entities;

	private int limit = DEFAULT_LIMIT;
	private int offset = 0;
	private int totalHits = 0;

	/**
	 * <p>
	 * Constructs an empty list with an initial capacity of
	 * {@link LuceneIndexSearcherRequest#DEFAULT_LIMIT}.
	 * </p>
	 */
	public LuceneIndexSearcherResponse() {
		this(DEFAULT_LIMIT);
	}

	/**
	 * <p>
	 * Constructs an empty list with the specified initial capacity.
	 * </p>
	 * 
	 * @param initialCapacity
	 *            the initial capacity of the list
	 * @exception IllegalArgumentException
	 *                if the specified initial capacity
	 *                is negative
	 */
	public LuceneIndexSearcherResponse(final int initialCapacity) {
		this.entities = new ArrayList<E>(initialCapacity);
	}

	/**
	 * <p>
	 * Returns a current page index (zero-origin). The result is incremented if
	 * the {@code offset} can't be divided evenly by the {@code limit}.
	 * </p>
	 * 
	 * @return a current page index (zero-origin)
	 * @see #getCurrentPageNumber()
	 */
	public int getCurrentPageIndex() {
		final int i = offset / limit;
		if (offset % limit == 0) {
			return i;
		}
		return i + 1;
	}

	/**
	 * <p>
	 * Returns a current page number (one-origin). The result is incremented if
	 * the {@code offset} can't be divided evenly by the {@code limit}.
	 * </p>
	 * 
	 * @return a current page number (one-origin)
	 * @see #getCurrentPageIndex()
	 */
	public int getCurrentPageNumber() {
		return getCurrentPageIndex() + 1;
	}

	/**
	 * <p>
	 * Returns {@code true} if has previous page.
	 * </p>
	 * 
	 * @return {@code true} if has previous page
	 */
	public boolean hasPrevPage() {
		return 0 < offset;
	}

	/**
	 * <p>
	 * Returns {@code true} if has next page.
	 * </p>
	 * 
	 * @return {@code true} if has next page
	 */
	public boolean hasNextPage() {
		return offset + limit < totalHits;
	}

	/**
	 * <p>
	 * Accessor to {@link #hasPrevPage()} for the EL Expression.
	 * </p>
	 * 
	 * @return {@link #hasPrevPage()}
	 */
	public boolean getHasPrevPage() {
		return hasPrevPage();
	}

	/**
	 * <p>
	 * Accessor to {@link #hasNextPage()} for the EL Expression.
	 * </p>
	 * 
	 * @return {@link #hasNextPage()}
	 */
	public boolean getHasNextPage() {
		return hasNextPage();
	}

	/* ================ @Override Object ================ */

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + limit;
		result = prime * result + offset;
		result = prime * result + totalHits;
		result = prime * result
				+ ((entities == null) ? 0 : entities.hashCode());
		return result;
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		@SuppressWarnings("unchecked")
		final LuceneIndexSearcherResponse<E> other = (LuceneIndexSearcherResponse<E>) obj;
		if (limit != other.limit) {
			return false;
		}
		if (offset != other.offset) {
			return false;
		}
		if (totalHits != other.totalHits) {
			return false;
		}
		if (entities == null) {
			if (other.entities != null) {
				return false;
			}
		} else if (!entities.equals(other.entities)) {
			return false;
		}
		return true;
	}

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		sb.append("totalHits: ").append(totalHits);
		sb.append(", limit: ").append(limit);
		sb.append(", offset: ").append(offset);
		sb.append(", size: ").append(size());
		if (!entities.isEmpty()) {
			sb.append(", first=[").append(entities.get(0)).append("]");
		}
		return sb.toString();
	}

	/* ================ getter/setter ================ */

	/**
	 * @return limit
	 */
	public int getLimit() {
		return limit;
	}

	/**
	 * @param limit
	 */
	public void setLimit(final int limit) {
		this.limit = limit;
	}

	/**
	 * @return offset
	 */
	public int getOffset() {
		return offset;
	}

	/**
	 * @param offset
	 */
	public void setOffset(final int offset) {
		this.offset = offset;
	}

	/**
	 * @return totalHits
	 */
	public int getTotalHits() {
		return totalHits;
	}

	/**
	 * @param totalHits
	 */
	public void setTotalHits(final int totalHits) {
		this.totalHits = totalHits;
	}

	/* ================ implements List ================ */

	@Override
	public boolean add(final E e) {
		return entities.add(e);
	}

	@Override
	public void add(final int index, final E element) {
		entities.add(index, element);
	}

	@Override
	public boolean addAll(final Collection<? extends E> c) {
		return entities.addAll(c);
	}

	@Override
	public boolean addAll(final int index, final Collection<? extends E> c) {
		return entities.addAll(index, c);
	}

	@Override
	public void clear() {
		entities.clear();
	}

	@Override
	public boolean contains(final Object o) {
		return entities.contains(o);
	}

	@Override
	public boolean containsAll(final Collection<?> c) {
		return entities.containsAll(c);
	}

	@Override
	public E get(final int index) {
		return entities.get(index);
	}

	@Override
	public int indexOf(final Object o) {
		return entities.indexOf(o);
	}

	@Override
	public boolean isEmpty() {
		return entities.isEmpty();
	}

	@Override
	public Iterator<E> iterator() {
		return entities.iterator();
	}

	@Override
	public int lastIndexOf(final Object o) {
		return entities.lastIndexOf(o);
	}

	@Override
	public ListIterator<E> listIterator() {
		return entities.listIterator();
	}

	@Override
	public ListIterator<E> listIterator(final int index) {
		return entities.listIterator(index);
	}

	@Override
	public boolean remove(final Object o) {
		return entities.remove(o);
	}

	@Override
	public E remove(final int index) {
		return entities.remove(index);
	}

	@Override
	public boolean removeAll(final Collection<?> c) {
		return entities.removeAll(c);
	}

	@Override
	public boolean retainAll(final Collection<?> c) {
		return entities.removeAll(c);
	}

	@Override
	public E set(final int index, final E element) {
		return entities.set(index, element);
	}

	@Override
	public int size() {
		return entities.size();
	}

	@Override
	public List<E> subList(final int fromIndex, final int toIndex) {
		return entities.subList(fromIndex, toIndex);
	}

	@Override
	public Object[] toArray() {
		return entities.toArray();
	}

	@Override
	public <T> T[] toArray(final T[] a) {
		return entities.toArray(a);
	}

}
