/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.normalizer;

import java.io.Reader;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>
 * Normalizes Roman numbers. Converts characters ['Ⅰ' ({@code U+2160}) - 'Ⅻ' (
 * {@code U+216B})] and ['ⅰ' ({@code U+2170}) - 'ⅻ' ({@code U+217B})] to [1 -
 * 12].
 * </p>
 * <p lang="ja">
 * NFKC や NFKD する前にこの {@link Reader} を通すことで、例えば 'Ⅻ' ({@code
 * U+216B}) が {@code "XII"} と分解される事を回避することができる。
 * </p>
 * 
 * @author mtzky
 */
public class RomanNumberNormalizer extends CharToStringMappingNormalizer {

	private static final Map<Integer, String> MAP = new HashMap<Integer, String>();
	static {
		/* Ⅰ - Ⅻ */
		MAP.put(0x2160, "1");
		MAP.put(0x2161, "2");
		MAP.put(0x2162, "3");
		MAP.put(0x2163, "4");
		MAP.put(0x2164, "5");
		MAP.put(0x2165, "6");
		MAP.put(0x2166, "7");
		MAP.put(0x2167, "8");
		MAP.put(0x2168, "9");
		MAP.put(0x2169, "10");
		MAP.put(0x216A, "11");
		MAP.put(0x216B, "12");

		/* ⅰ - ⅻ */
		MAP.put(0x2170, "1");
		MAP.put(0x2171, "2");
		MAP.put(0x2172, "3");
		MAP.put(0x2173, "4");
		MAP.put(0x2174, "5");
		MAP.put(0x2175, "6");
		MAP.put(0x2176, "7");
		MAP.put(0x2177, "8");
		MAP.put(0x2178, "9");
		MAP.put(0x2179, "10");
		MAP.put(0x217A, "11");
		MAP.put(0x217B, "12");
	}

	/**
	 * @param in
	 */
	public RomanNumberNormalizer(final Reader in) {
		super(in);
	}

	@Override
	protected String convert(final int c) {
		return MAP.get(c);
	}

}
