/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.normalizer;

import static org.mtzky.io.IOUtils.*;

import java.io.IOException;
import java.io.PushbackReader;
import java.io.Reader;

import org.mtzky.io.Closable;
import org.mtzky.io.ClosingGuardian;

/**
 * <p>
 * Normalizes the voiced labiodental fricative in Hiragana and Katakana.
 * </p>
 * <table border="1">
 * <thead>
 * <tr>
 * <th colspan="2">before</th>
 * <th colspan="2">after</th>
 * </tr>
 * <tr>
 * <th>code</th>
 * <th>glyph</th>
 * <th>code</th>
 * <th>glyph</th>
 * </tr>
 * </thead><tbody>
 * <tr>
 * <td>U+30F4 U+30A1</td>
 * <td>ヴァ</td>
 * <td>U+30D0</td>
 * <td>バ</td>
 * </tr>
 * <tr>
 * <td>U+30F4 U+30A3</td>
 * <td>ヴィ</td>
 * <td>U+30D3</td>
 * <td>ビ</td>
 * </tr>
 * <tr>
 * <td>U+30F4 U+30A5</td>
 * <td><strong>ヴゥ</strong></td>
 * <td>U+30D6</td>
 * <td>ブ</td>
 * </tr>
 * <tr>
 * <td>U+30F4 U+30A7</td>
 * <td>ヴェ</td>
 * <td>U+30D9</td>
 * <td>ベ</td>
 * </tr>
 * <tr>
 * <td>U+30F4 U+30A9</td>
 * <td>ヴォ</td>
 * <td>U+30DC</td>
 * <td>ボ</td>
 * </tr>
 * <tr>
 * <td>U+30F4</td>
 * <td>ヴ</td>
 * <td>U+30D6</td>
 * <td>ブ</td>
 * </tr>
 * <tr>
 * <td>U+3094 U+3041</td>
 * <td>&#x3094;ぁ</td>
 * <td>U+30D0</td>
 * <td>ば</td>
 * </tr>
 * <tr>
 * <td>U+3094 U+3043</td>
 * <td>&#x3094;ぃ</td>
 * <td>U+30D3</td>
 * <td>び</td>
 * </tr>
 * <tr>
 * <td>U+3094 U+3045</td>
 * <td><strong>&#x3094;ぅ</strong></td>
 * <td>U+30D6</td>
 * <td>ぶ</td>
 * </tr>
 * <tr>
 * <td>U+3094 U+3047</td>
 * <td>&#x3094;ぇ</td>
 * <td>U+30D9</td>
 * <td>べ</td>
 * </tr>
 * <tr>
 * <td>U+3094 U+3049</td>
 * <td>&#x3094;ぉ</td>
 * <td>U+30DC</td>
 * <td>ぼ</td>
 * </tr>
 * <tr>
 * <td>U+3094</td>
 * <td>&#x3094;</td>
 * <td>U+30D6</td>
 * <td>ぶ</td>
 * </tr>
 * <tr>
 * <td>U+FF73 U+FF9E U+FF67</td>
 * <td>&#xFF73;&#xFF9E;&#xFF67;</td>
 * <td>U+FF8A U+FF9E</td>
 * <td>&#xFF8A;&#xFF9E;</td>
 * </tr>
 * <tr>
 * <td>U+FF73 U+FF9E U+FF68</td>
 * <td>&#xFF73;&#xFF9E;&#xFF68;</td>
 * <td>U+FF8B U+FF9E</td>
 * <td>&#xFF8B;&#xFF9E;</td>
 * </tr>
 * <tr>
 * <td>U+FF73 U+FF9E U+FF69</td>
 * <td><strong>&#xFF73;&#xFF9E;&#xFF69;</strong></td>
 * <td>U+FF8C U+FF9E</td>
 * <td>&#xFF8C;&#xFF9E;</td>
 * </tr>
 * <tr>
 * <td>U+FF73 U+FF9E U+FF6A</td>
 * <td>&#xFF73;&#xFF9E;&#xFF6A;</td>
 * <td>U+FF8D U+FF9E</td>
 * <td>&#xFF8D;&#xFF9E;</td>
 * </tr>
 * <tr>
 * <td>U+FF73 U+FF9E U+FF6B</td>
 * <td>&#xFF73;&#xFF9E;&#xFF6B;</td>
 * <td>U+FF8E U+FF9E</td>
 * <td>&#xFF8E;&#xFF9E;</td>
 * </tr>
 * <tr>
 * <td>U+FF73 U+FF9E</td>
 * <td>&#xFF73;&#xFF9E;</td>
 * <td>U+FF8C U+FF9E</td>
 * <td>&#xFF8C;&#xFF9E;</td>
 * </tr>
 * <tr>
 * <td>U+30F7</td>
 * <td>&#x30F7;</td>
 * <td>U+30D0</td>
 * <td>バ</td>
 * </tr>
 * <tr>
 * <td>U+30F8</td>
 * <td>&#x30F8;</td>
 * <td>U+30D3</td>
 * <td>ビ</td>
 * </tr>
 * <tr>
 * <td>U+30F9</td>
 * <td>&#x30F9;</td>
 * <td>U+30D9</td>
 * <td>ベ</td>
 * </tr>
 * <tr>
 * <td>U+30FA</td>
 * <td>&#x30FA;</td>
 * <td>U+30DC</td>
 * <td>ボ</td>
 * </tr>
 * </tbody>
 * </table>
 * <p>
 * <strong>NOTE</strong>: NOT support the combining character.
 * </p>
 * 
 * @author mtzky
 */
public class VLFKanaNormalizer extends Reader implements Closable {

	@SuppressWarnings("unused")
	private final Object guardian = new ClosingGuardian(this);
	private boolean close = false;

	private final PushbackReader in;

	public VLFKanaNormalizer(final Reader in) {
		this(toPushbackReader(in));
	}

	public VLFKanaNormalizer(final PushbackReader in) {
		super(in);
		this.in = in;
	}

	@Override
	public int read(final char[] cbuf, final int off, final int len)
			throws IOException {
		final int limit = off + len;
		int read = 0;
		for (int i = off; i < limit; i++) {
			final int r = in.read();
			if (r < 0) {
				break;
			}
			switch ((char) r) {
			case 'ヴ': {
				final int next = in.read();
				switch ((char) next) {
				case 'ァ':
					cbuf[i] = 'バ';
					break;
				case 'ィ':
					cbuf[i] = 'ビ';
					break;
				case 'ゥ':
					cbuf[i] = 'ブ';
					break;
				case 'ェ':
					cbuf[i] = 'ベ';
					break;
				case 'ォ':
					cbuf[i] = 'ボ';
					break;
				default:
					cbuf[i] = 'ブ';
					if (0 <= next) {
						in.unread(next);
					}
					break;
				}
				break;
			}
			case /* ゔ */'\u3094': {
				final int next = in.read();
				switch ((char) next) {
				case 'ぁ':
					cbuf[i] = 'ば';
					break;
				case 'ぃ':
					cbuf[i] = 'び';
					break;
				case 'ぅ':
					cbuf[i] = 'ぶ';
					break;
				case 'ぇ':
					cbuf[i] = 'べ';
					break;
				case 'ぉ':
					cbuf[i] = 'ぼ';
					break;
				default:
					cbuf[i] = 'ぶ';
					if (0 <= next) {
						in.unread(next);
					}
					break;
				}
				break;
			}
			case 'ｳ': {
				{
					final int next1 = in.read();
					if (next1 != /* ﾞ */0xFF9E) {
						cbuf[i] = 'ｳ';
						if (0 <= next1) {
							in.unread(next1);
						}
						break;
					}
				}
				final int next2 = in.read();
				switch ((char) next2) {
				case 'ｧ':
					cbuf[i++] = 'ﾊ';
					read++;
					cbuf[i] = 'ﾞ';
					break;
				case 'ｨ':
					cbuf[i++] = 'ﾋ';
					read++;
					cbuf[i] = 'ﾞ';
					break;
				case 'ｩ':
					cbuf[i++] = 'ﾌ';
					read++;
					cbuf[i] = 'ﾞ';
					break;
				case 'ｪ':
					cbuf[i++] = 'ﾍ';
					read++;
					cbuf[i] = 'ﾞ';
					break;
				case 'ｫ':
					cbuf[i++] = 'ﾎ';
					read++;
					cbuf[i] = 'ﾞ';
					break;
				default:
					cbuf[i++] = 'ﾌ';
					read++;
					cbuf[i] = 'ﾞ';
					if (0 <= next2) {
						in.unread(next2);
					}
					break;
				}
				break;
			}
			case 'ヷ': {
				cbuf[i] = 'バ';
				break;
			}
			case 'ヸ': {
				cbuf[i] = 'ビ';
				break;
			}
			case 'ヹ': {
				cbuf[i] = 'ベ';
				break;
			}
			case 'ヺ': {
				cbuf[i] = 'ボ';
				break;
			}
			default:
				cbuf[i] = (char) r;
				break;
			}
			read++;
		}
		return read == 0 ? -1 : read;
	}

	@Override
	public boolean isClosed() {
		return close;
	}

	@Override
	public void close() throws IOException {
		in.close();
		close = true;
	}

}
