/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.type;

import static java.math.BigInteger.*;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DecimalFormat;
import java.text.ParsePosition;

import org.apache.lucene.document.Fieldable;
import org.mtzky.lucene.descriptor.LucenePropertyConfig;

/**
 * @author mtzky
 */
@LuceneFieldStrategyType(BigInteger.class)
public class LuceneBigIntegerStrategy extends LuceneStringStrategy {

	private final boolean original;
	private final String format;

	/**
	 * @param config
	 */
	public LuceneBigIntegerStrategy(final LucenePropertyConfig config) {
		super(config);
		this.format = config.getFormat();
		this.original = format.isEmpty();
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T getValue(final Fieldable field) {
		final String v = field.stringValue();
		if (original) {
			return (T) new BigDecimal(v).toBigInteger();
		}
		final DecimalFormat format = new DecimalFormat(this.format);
		format.setParseBigDecimal(true);
		final BigDecimal d = (BigDecimal) format.parse(v, new ParsePosition(0));
		return (T) d.toBigInteger();
	}

	@Override
	protected String toString(final Object value) {
		final BigInteger v = (value != null) ? (BigInteger) value : ZERO;
		if (original) {
			return v.toString();
		}
		return new DecimalFormat(this.format).format(v);
	}

}
