/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.date;

import static java.text.DateFormat.*;
import static java.util.Calendar.*;

import java.sql.Time;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

/**
 * <p>
 * DateHelper is utilities for the {@link Calendar} and {@link Date} object.
 * </p>
 * <p>
 * Setter methods are NOT destructive, returns another instance, and can chain
 * methods.
 * </p>
 * 
 * <pre>
 * final DateHelper dateHelper = new DateHelper();
 * {@link Date} date = dateHelper
 * 	.{@link #add(int, int) add}({@link Calendar#YEAR}, -2)
 * 	.{@link #add(int, int) add}({@link Calendar#MONTH}, 6)
 * 	.{@link #set(int, int) set}({@link Calendar#DATE}, 25)
 * 	.{@link #set(int, int) set}({@link Calendar#HOUR_OF_DAY}, 10)
 * 	.{@link #truncate(int) truncate}({@link Calendar#HOUR_OF_DAY})
 * 	.{@link #getDate()};
 * </pre>
 * 
 * @author mtzky
 * @since 0.1.9
 * @see #setDefaultTimeZone(TimeZone)
 * @see #setDefaultLocale(Locale)
 * @see #setDefaultFirstDayOfWeek(int)
 */
public class DateHelper implements Comparable<DateHelper>, Cloneable {

	private static final int[] FIELDS = { YEAR, MONTH, DATE, HOUR_OF_DAY,
			MINUTE, SECOND, MILLISECOND };

	private static TimeZone DEFAULT_TIME_ZONE = TimeZone.getDefault();
	private static Locale DEFAULT_LOCALE = Locale.getDefault();
	private static int DEFAULT_FIRST_DAY_OF_WEEK = SUNDAY;

	private TimeZone timeZone = DEFAULT_TIME_ZONE;
	private Locale locale = DEFAULT_LOCALE;
	private int firstDayOfWeek = DEFAULT_FIRST_DAY_OF_WEEK;

	/**
	 * <p>
	 * NOTE: Do NOT access this directly. Instead, use the method
	 * {@link #getTimeMillis()}.
	 * </p>
	 */
	private final long timeMillis;

	/**
	 * <p>
	 * Initializes time as {@link System#currentTimeMillis()}.
	 * </p>
	 * 
	 * @see #DateHelper(long)
	 * @see #setTimeZone(TimeZone)
	 * @see #setLocale(Locale)
	 * @see #setFirstDayOfWeek(int)
	 */
	public DateHelper() {
		this(System.currentTimeMillis());
	}

	/**
	 * <p>
	 * Initializes time as {@code timeMillis}.
	 * </p>
	 * 
	 * @param timeMillis
	 * @see #setTimeZone(TimeZone)
	 * @see #setLocale(Locale)
	 * @see #setFirstDayOfWeek(int)
	 */
	public DateHelper(final long timeMillis) {
		this.timeMillis = timeMillis;
	}

	/**
	 * <p>
	 * Initializes time with the given {@code args} for the corresponding
	 * calendar fields. For more information, please see
	 * {@link #DateHelper(TimeZone, Locale, int[])}.
	 * </p>
	 * 
	 * @param args
	 *            the values for the {@link Calendar} fields
	 * @see #DateHelper(TimeZone, Locale, int[])
	 * @see #setTimeZone(TimeZone)
	 * @see #setLocale(Locale)
	 * @see #setFirstDayOfWeek(int)
	 */
	public DateHelper(final int... args) {
		this(DEFAULT_TIME_ZONE, DEFAULT_LOCALE, args);
	}

	/**
	 * <p>
	 * Initializes time with the given {@code args} for the corresponding
	 * calendar fields. For more information, please see
	 * {@link #DateHelper(TimeZone, Locale, int[])}.
	 * </p>
	 * 
	 * @param timeZone
	 *            the time zone to use
	 * @param args
	 *            the values for the {@link Calendar} fields
	 * @see #DateHelper(TimeZone, Locale, int[])
	 * @see #setTimeZone(TimeZone)
	 * @see #setLocale(Locale)
	 * @see #setFirstDayOfWeek(int)
	 */
	public DateHelper(final TimeZone timeZone, final int... args) {
		this(timeZone, DEFAULT_LOCALE, args);
	}

	/**
	 * <p>
	 * Initializes time with the given {@code args} for the corresponding
	 * calendar fields. For more information, please see
	 * {@link #DateHelper(TimeZone, Locale, int[])}.
	 * </p>
	 * 
	 * @param locale
	 *            the locale for the week data
	 * @param args
	 *            the values for the {@link Calendar} fields
	 * @see #DateHelper(TimeZone, Locale, int[])
	 * @see #setTimeZone(TimeZone)
	 * @see #setLocale(Locale)
	 * @see #setFirstDayOfWeek(int)
	 */
	public DateHelper(final Locale locale, final int... args) {
		this(DEFAULT_TIME_ZONE, locale, args);
	}

	/**
	 * <p>
	 * Initializes time with the given {@code args} for the corresponding
	 * calendar fields.
	 * </p>
	 * <ol>
	 * <li>{@link Calendar#YEAR}</li>
	 * <li>{@link Calendar#MONTH} (zero-origin; use {@link Calendar#JANUARY} ...
	 * {@link Calendar#DECEMBER})</li>
	 * <li>{@link Calendar#DATE}</li>
	 * <li>{@link Calendar#HOUR_OF_DAY}</li>
	 * <li>{@link Calendar#MINUTE}</li>
	 * <li>{@link Calendar#SECOND}</li>
	 * <li>{@link Calendar#MILLISECOND}</li>
	 * </ol>
	 * <p>
	 * NOTE: This method execute {@link Calendar#setTimeInMillis(long)
	 * setTimeInMillis(0L)} before {@link Calendar#set(int, int) set} to the
	 * corresponding calendar field. If you pass locale {@link Locale ja_JP_JP},
	 * {@code JapaneseImperialCalendar} is used as {@link Calendar}, and
	 * initialized that {@link Calendar#ERA ERA} is Showa (3). <strong>Showa 45
	 * is A.D. 1970</strong>.
	 * </p>
	 * 
	 * @param timeZone
	 *            the time zone to use
	 * @param locale
	 *            the locale for the week data
	 * @param args
	 *            the values for the {@link Calendar} fields
	 * @throws NullPointerException
	 *             if {@code args} is {@code null}
	 * @throws IllegalArgumentException
	 *             if {@code args.length} is eight or more
	 * @see #setFirstDayOfWeek(int)
	 */
	public DateHelper(final TimeZone timeZone, final Locale locale,
			final int... args) {
		this(buildCalendar(timeZone, locale, args), locale);
	}

	private static Calendar buildCalendar(final TimeZone timeZone,
			final Locale locale, final int... args) {
		if (args == null) {
			throw new NullPointerException("args");
		}
		if (FIELDS.length < args.length) {
			throw new IllegalArgumentException("too many args: "
					+ Arrays.toString(args));
		}
		final Calendar cal = Calendar.getInstance(timeZone, locale);
		cal.setTimeInMillis(0L);
		final int len = args.length;
		for (int i = 0; i < len; i++) {
			cal.set(FIELDS[i], args[i]);
		}
		return cal;
	}

	/**
	 * <p>
	 * Initializes time as {@link Date#getTime()}.
	 * </p>
	 * 
	 * @param date
	 *            {@link Date}
	 * @see #setTimeZone(TimeZone)
	 * @see #setLocale(Locale)
	 * @see #setFirstDayOfWeek(int)
	 */
	public DateHelper(final Date date) {
		this(date.getTime());
	}

	/**
	 * <p>
	 * Initializes time as {@link Calendar#getTimeInMillis()}.
	 * </p>
	 * 
	 * @param calendar
	 *            {@link Calendar}
	 * @see #setLocale(Locale)
	 */
	public DateHelper(final Calendar calendar) {
		this(calendar, DEFAULT_LOCALE);
	}

	/**
	 * <p>
	 * Initializes time as {@link Calendar#getTimeInMillis()}.
	 * </p>
	 * 
	 * @param calendar
	 *            {@link Calendar}
	 * @param locale
	 *            the locale for the week data
	 * @see #setLocale(Locale)
	 */
	public DateHelper(final Calendar calendar, final Locale locale) {
		this(calendar.getTimeInMillis());
		this.timeZone = calendar.getTimeZone();
		this.locale = locale;
		this.firstDayOfWeek = calendar.getFirstDayOfWeek();
	}

	/* ---------------------------------------------------------------- */

	/**
	 * <p>
	 * Adds or subtracts the specified amount of time to the given calendar
	 * field, based on the calendar's rules.
	 * </p>
	 * 
	 * @param field
	 *            the calendar field.
	 * @param amount
	 *            the amount of date or time to be added to the field.
	 * @return {@link DateHelper} to chain method
	 * @see Calendar#add(int, int)
	 */
	public final DateHelper add(final int field, final int amount) {
		final Calendar cal = getCalendar();
		cal.add(field, amount);
		return new DateHelper(cal).setLocale(this.locale);
	}

	/**
	 * <p>
	 * Adds the specified (signed) amount to the specified calendar field
	 * without changing larger fields. A negative amount means to roll down.
	 * </p>
	 * 
	 * @param field
	 *            the given calendar field.
	 * @param amount
	 *            the signed amount to add to the calendar field.
	 * @return {@link DateHelper} to chain method
	 * @see Calendar#roll(int, int)
	 */
	public final DateHelper roll(final int field, final int amount) {
		final Calendar cal = getCalendar();
		cal.roll(field, amount);
		return new DateHelper(cal).setLocale(this.locale);
	}

	/**
	 * <p>
	 * Sets the given calendar field to the given value.
	 * </p>
	 * 
	 * @param field
	 *            the given calendar field.
	 * @param value
	 *            the value to be set for the given calendar field.
	 * @return {@link DateHelper} to chain method
	 */
	public final DateHelper set(final int field, final int value) {
		final Calendar cal = getCalendar();
		cal.set(field, value);
		return new DateHelper(cal).setLocale(this.locale);
	}

	/**
	 * <p>
	 * Truncates a time leaving the specified field.
	 * </p>
	 * <table border="1">
	 * <thead>
	 * <tr>
	 * <th>Specified Field</th>
	 * <th>Truncated Date</th>
	 * </tr>
	 * <tr>
	 * <td>Original Date</td>
	 * <td>2011-12-13 14:15:16.017</td>
	 * </tr>
	 * </thead><tbody>
	 * <tr>
	 * <td>{@link Calendar#ERA}</td>
	 * <td>0001-01-01 00:00:00.000</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#YEAR}</td>
	 * <td>2011-01-01 00:00:00.000</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#MONTH}</td>
	 * <td>2011-12-01 00:00:00.000</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#WEEK_OF_YEAR}</td>
	 * <td rowspan="2">2011-12-11 00:00:00.000</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#WEEK_OF_MONTH}</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#DATE}</td>
	 * <td rowspan="5">2011-12-13 00:00:00.000</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#DAY_OF_MONTH}</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#DAY_OF_YEAR}</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#DAY_OF_WEEK}</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#DAY_OF_WEEK_IN_MONTH}</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#AM_PM}</td>
	 * <td rowspan="2">2011-12-13 12:00:00.000</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#HOUR}</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#HOUR_OF_DAY}</td>
	 * <td>2011-12-13 14:00:00.000</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#MINUTE}</td>
	 * <td>2011-12-13 14:15:00.000</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#SECOND}</td>
	 * <td>2011-12-13 14:15:16.000</td>
	 * </tr>
	 * <tr>
	 * <td>{@link Calendar#MILLISECOND}</td>
	 * <td>2011-12-13 14:15:16.017</td>
	 * </tr>
	 * </tbody>
	 * </table>
	 * <p>
	 * NOTE: Takes {@link Calendar#getFirstDayOfWeek()} as a truncated time
	 * leaving {@link Calendar#WEEK_OF_YEAR}/{@link Calendar#WEEK_OF_MONTH}.
	 * </p>
	 * 
	 * @param field
	 *            the calendar field.
	 * @return {@link DateHelper} to chain method
	 * @see Calendar#getFirstDayOfWeek()
	 */
	public final DateHelper truncate(final int field) {
		final Calendar cal = getCalendar();
		switch (field) {
		case ERA:
			cal.set(YEAR, 0);
		case YEAR:
			cal.set(MONTH, JANUARY);
		case MONTH:
		case WEEK_OF_YEAR:
		case WEEK_OF_MONTH:
			switch (field) {
			case WEEK_OF_YEAR:
			case WEEK_OF_MONTH:
				final int first = cal.getFirstDayOfWeek();
				final int current = cal.get(DAY_OF_WEEK);
				if (current < first) {
					cal.add(DATE, (first - current - 7));
				} else {
					cal.add(DATE, (first - current));
				}
				break;
			default:
				cal.set(DATE, 1);
				break;
			}
		case DAY_OF_MONTH /* DATE */:
		case DAY_OF_YEAR:
		case DAY_OF_WEEK:
		case DAY_OF_WEEK_IN_MONTH:
		case AM_PM:
		case HOUR:
			switch (field) {
			case AM_PM:
			case HOUR:
				if (cal.get(AM_PM) == PM) {
					cal.set(HOUR_OF_DAY, 12);
					break;
				}
			default:
				cal.set(HOUR_OF_DAY, 0);
				break;
			}
		case HOUR_OF_DAY:
			cal.set(MINUTE, 0);
		case MINUTE:
			cal.set(SECOND, 0);
		case SECOND:
			cal.set(MILLISECOND, 0);
		case MILLISECOND:
			break;
		default:
			throw new UnsupportedOperationException("Unsupported field: "
					+ field);
		}
		return new DateHelper(cal).setLocale(this.locale);
	}

	/* ---------------------------------------------------------------- */

	/**
	 * @return time in milliseconds.
	 */
	public long getTimeMillis() {
		return timeMillis;
	}

	/**
	 * <p>
	 * Gets a calendar initialized to represent the {@link #getTimeMillis()
	 * specified number of milliseconds}, using the specified
	 * {@link #setTimeZone(TimeZone) time zone}, {@link #setLocale(Locale)
	 * locale}, and {@link #setFirstDayOfWeek(int) first day of week}.
	 * </p>
	 * 
	 * @return {@link Calendar} initialized to represent the specified number of
	 *         milliseconds.
	 * @see #getTimeMillis()
	 * @see #setTimeZone(TimeZone)
	 * @see #setLocale(Locale)
	 * @see #setFirstDayOfWeek(int)
	 */
	public final Calendar getCalendar() {
		final Calendar cal = Calendar.getInstance(timeZone, locale);
		cal.setFirstDayOfWeek(firstDayOfWeek);
		cal.setTimeInMillis(getTimeMillis());
		return cal;
	}

	/**
	 * @return {@link java.util.Date} initialized to represent the specified
	 *         number of
	 *         milliseconds.
	 */
	public final Date getDate() {
		return new Date(getTimeMillis());
	}

	/**
	 * @return {@link java.sql.Date} initialized to represent the specified
	 *         number of milliseconds.
	 */
	public final java.sql.Date getSqlDate() {
		return new java.sql.Date(getTimeMillis());
	}

	/**
	 * @return {@link Timestamp} initialized to represent the specified
	 *         number of milliseconds.
	 */
	public final Timestamp getSqlTimestamp() {
		return new Timestamp(getTimeMillis());
	}

	/**
	 * @return {@link Time} initialized to represent the specified
	 *         number of milliseconds.
	 */
	public final Time getSqlTime() {
		return new Time(getTimeMillis());
	}

	/* ---------------------------------------------------------------- */

	/**
	 * @return formatted {@link #getDate() date} string with {@link DateFormat}
	 * @see #format(int, int)
	 */
	public String format() {
		return format(DEFAULT, DEFAULT);
	}

	/**
	 * @param dateStyle
	 *            the given date formatting style
	 * @param timeStyle
	 *            the given time formatting style
	 * @return formatted {@link #getDate() date} string with {@link DateFormat}
	 * @see DateFormat#FULL
	 * @see DateFormat#LONG
	 * @see DateFormat#MEDIUM
	 * @see DateFormat#SHORT
	 * @see DateFormat#DEFAULT
	 */
	public String format(final int dateStyle, final int timeStyle) {
		return DateFormat.getDateTimeInstance(dateStyle, timeStyle, locale)
				.format(getDate());
	}

	/**
	 * @param pattern
	 *            the pattern describing the date and time format
	 * @return formatted {@link #getDate() date} string with
	 *         {@link SimpleDateFormat}
	 */
	public String format(final String pattern) {
		return new SimpleDateFormat(pattern, locale).format(getDate());
	}

	/* ---------------------------------------------------------------- */

	/**
	 * @param defaultLocale
	 *            {@link Locale}
	 */
	public static void setDefaultLocale(final Locale defaultLocale) {
		if (defaultLocale == null) {
			throw new NullPointerException("defaultLocale");
		}
		DateHelper.DEFAULT_LOCALE = defaultLocale;
	}

	/**
	 * @param defaultTimeZone
	 *            {@link TimeZone}
	 */
	public static void setDefaultTimeZone(final TimeZone defaultTimeZone) {
		if (defaultTimeZone == null) {
			throw new NullPointerException("defaultTimeZone");
		}
		DateHelper.DEFAULT_TIME_ZONE = defaultTimeZone;
	}

	/**
	 * @param defaultFirstDayOfWeek
	 *            {@link Calendar#setFirstDayOfWeek(int)}
	 */
	public static void setDefaultFirstDayOfWeek(final int defaultFirstDayOfWeek) {
		if (defaultFirstDayOfWeek < SUNDAY || SATURDAY < defaultFirstDayOfWeek) {
			throw new IllegalArgumentException("out of bounds:"
					+ defaultFirstDayOfWeek);
		}
		DateHelper.DEFAULT_FIRST_DAY_OF_WEEK = defaultFirstDayOfWeek;
	}

	/* ---------------------------------------------------------------- */

	/**
	 * @param locale
	 *            {@link Locale}
	 */
	public DateHelper setLocale(final Locale locale) {
		if (locale == null) {
			throw new NullPointerException("locale");
		}
		this.locale = locale;
		return this;
	}

	/**
	 * @param timeZone
	 *            {@link TimeZone}
	 */
	public DateHelper setTimeZone(final TimeZone timeZone) {
		if (timeZone == null) {
			throw new NullPointerException("timeZone");
		}
		this.timeZone = timeZone;
		return this;
	}

	/**
	 * @param firstDayOfWeek
	 *            {@link Calendar#setFirstDayOfWeek(int)}
	 */
	public DateHelper setFirstDayOfWeek(final int firstDayOfWeek) {
		if (firstDayOfWeek < SUNDAY || SATURDAY < firstDayOfWeek) {
			throw new IllegalArgumentException("out of bounds:"
					+ firstDayOfWeek);
		}
		this.firstDayOfWeek = firstDayOfWeek;
		return this;
	}

	/* ---------------------------------------------------------------- */

	@Override
	public final String toString() {
		final StringBuilder sb = new StringBuilder();
		sb.append(getClass().getSimpleName()).append(" [");
		sb.append("timeMillis=").append(getTimeMillis());
		sb.append(", timeZone=").append(timeZone);
		sb.append(", locale=").append(locale);
		sb.append(", firstDayOfWeek=").append(firstDayOfWeek);
		sb.append("] ");
		sb.append(format(FULL, FULL));
		return sb.toString();
	}

	@Override
	public final int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + Long.valueOf(getTimeMillis()).hashCode();
		result = prime * result + timeZone.hashCode();
		result = prime * result + locale.hashCode();
		result = prime * result + firstDayOfWeek;
		return result;
	}

	@Override
	public final boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof DateHelper)) {
			return false;
		}
		final DateHelper other = (DateHelper) obj;
		if (getTimeMillis() != other.getTimeMillis()) {
			return false;
		}
		if (!timeZone.equals(other.timeZone)) {
			return false;
		}
		if (!locale.equals(other.locale)) {
			return false;
		}
		if (firstDayOfWeek != other.firstDayOfWeek) {
			return false;
		}
		return true;
	}

	@Override
	public final int compareTo(final DateHelper dateHelper) {
		final long thisVal = getTimeMillis();
		final long another = dateHelper.getTimeMillis();
		return (thisVal < another ? -1 : (thisVal == another ? 0 : 1));
	}

	/**
	 * @return {@link DateHelper}
	 * @see Object#clone()
	 * @see Cloneable
	 */
	@Override
	public final DateHelper clone() {
		return new DateHelper(getTimeMillis()).setTimeZone(timeZone)
				.setLocale(locale).setFirstDayOfWeek(firstDayOfWeek);
	}

}
