/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.reflect;

import static java.io.File.*;
import static java.util.Collections.*;
import static javax.tools.JavaFileObject.Kind.*;
import static javax.tools.StandardLocation.*;
import static javax.tools.ToolProvider.*;
import static org.mtzky.io.IOUtils.*;
import static org.mtzky.lang.IterableUtils.*;
import static org.mtzky.log.GenericMarker.*;
import static org.slf4j.LoggerFactory.*;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import javax.tools.DiagnosticCollector;
import javax.tools.JavaCompiler;
import javax.tools.JavaFileManager;
import javax.tools.JavaFileObject;
import javax.tools.JavaFileObject.Kind;

import org.mtzky.lang.IterableUtils.Find;
import org.slf4j.Logger;

/**
 * <p>
 * Descriptors for {@link Class} in a {@code package}.
 * </p>
 * 
 * @author mtzky
 * @since 0.1.6
 */
public class PackageDesc {

	private static final Logger LOG = getLogger(PackageDesc.class);

	private static final Set<Kind> KINDS = EnumSet.of(CLASS);

	private static final Find<Class<?>> ALL = new Find<Class<?>>() {
		@Override
		public boolean call(final Class<?> value) {
			return true;
		}
	};

	private final List<Class<?>> classes;

	/**
	 * @param packageName
	 *            a package name
	 * @throws IOException
	 *             if an I/O error occurred
	 * @throws ClassNotFoundException
	 *             if the class cannot be located
	 */
	public PackageDesc(final String packageName) throws IOException,
			ClassNotFoundException {
		this(packageName, ALL);
	}

	/**
	 * @param packageName
	 *            a package name
	 * @param filter
	 * @throws IOException
	 *             if an I/O error occurred
	 * @throws ClassNotFoundException
	 *             if the class cannot be located
	 */
	public PackageDesc(final String packageName, final Find<Class<?>> filter)
			throws IOException, ClassNotFoundException {
		if (packageName == null) {
			throw new NullPointerException("packageName");
		}
		if (filter == null) {
			throw new NullPointerException("filter");
		}
		final JavaCompiler compiler = getSystemJavaCompiler();
		if (compiler == null) {
			throw new IllegalStateException(
					"Running in a JRE, but a JDK is required.");
		}
		final String p = packageName.endsWith(".") ? packageName
				: (packageName + '.');

		/* Gets classpath */
		final StringBuilder classpath = new StringBuilder();
		classpath.append(System.getProperty("java.class.path", ""));
		for (final ClassLoader cl : new ClassLoader[] {
				ClassLoader.getSystemClassLoader(),
				Thread.currentThread().getContextClassLoader() }) {
			if (cl instanceof URLClassLoader) {
				final URL[] urls = ((URLClassLoader) cl).getURLs();
				for (final URL url : urls) {
					try {
						classpath.append(pathSeparatorChar).append(
								new File(url.toURI()).getCanonicalPath());
					} catch (final URISyntaxException e) {
						final String fmt = "FAILED to parse classpath: %s";
						LOG.warn(FAILED_TO_PARSE, String.format(fmt, url), e);
					}
				}
			} else {
				LOG.warn(FAILED_TO_PARSE, "NOT URLClassLoader: {}", cl);
			}
		}

		JavaFileManager manager = null;
		try {
			final List<Class<?>> classes = new ArrayList<Class<?>>();
			manager = compiler.getStandardFileManager(
					new DiagnosticCollector<JavaFileObject>(), null, null);

			/* Adds classpath */
			if (0 < classpath.length()) {
				final List<String> cps = Arrays.asList(classpath.toString());
				if (!manager.handleOption("-classpath", cps.iterator())) {
					LOG.warn(FAILED_TO_ADD, "FAILED to add classpath: '{}'",
							classpath);
				} else if (LOG.isDebugEnabled(ADD)) {
					LOG.debug(ADD, "add classpath: '{}'", classpath);
				}
			}

			for (final JavaFileObject obj : manager.list(CLASS_PATH,
					packageName, KINDS, false)) {
				final String n = obj.getName();
				final Class<?> c = Class.forName(p
						+ n.substring(0, n.lastIndexOf('.')));
				if (filter.call(c)) {
					classes.add(c);
				}
			}
			this.classes = unmodifiableList(classes);
		} finally {
			closeQuietly(manager);
		}
	}

	/**
	 * @return classes in this {@code package} as an
	 *         {@link Collections#unmodifiableList(List) unmodifiable list}
	 */
	public List<Class<?>> getClasses() {
		return classes;
	}

	/**
	 * @param <T>
	 * @param callback
	 * @return a found class in this {@code package}, or {@code null} if not
	 */
	@SuppressWarnings("unchecked")
	public <T> Class<T> findClass(final Find<Class<?>> callback) {
		return (Class<T>) each(classes, callback);
	}

}
