#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#include <unistd.h>
#include <signal.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "vncviewer.h"
#include "image_processing.h"

static int send_histgram (struct client_cb *client_CB);

Display *display;
pthread_mutex_t c_mutex;
pthread_t c_thread = 0;

/**********************************/
/* utility functions              */
/**********************************/
void RGBtoHSV(int r,int g,int b,int *h,int *s,int *v)
{
    int max,min;
    max = r   > g ? r   : g;
    max = max > b ? max : b;
    min = r   < g ? r   : g;
    min = min < b ? min : b;

    if(max == 0)
    {//このときhとsは未定義
        *h = 0;*s = 0;*v = 0;
    }
    else if (max == min)
    {//このときhは未定義
        *h = 0;*s = 0;*v = max;
    }
    else
    {//計算式はWikipediaより引用
        if (max == r)      *h = 60 * (g - b) / (max - min);
        else if (max == g) *h = 60 * (b - r) / (max - min) + 120;
        else               *h = 60 * (r - g) / (max - min) + 240;
    }

    *h /= 2;
    while(*h < 0) *h += 180;
    while(*h >= 180) *h -= 180;
}

void tsInit(TS *ts,int length)
{
    ts->now = 0;
    ts->length = length;
}

void tsPutForward(TS *ts)
{
    ts->now++;
    if(ts->now >= ts->length) ts->now = 0;
}

int tsPrev(TS *ts,int time)
{
    time--;
    while(time < 0         ) time += ts->length;
    while(time >= ts->length) time -= ts->length;
    return time;
}

void tsUpdate(TS *ts, int length)
{
  if (ts->now >= length)	ts->now = length - 1;
  ts->length = length;
}

/**********************************/
/* For client functions           */
/**********************************/

/*
 * クライアント画像処理用スレッド関数
 */
void *image_proc_client (void *arg)
{
    //  宣言部  //
    struct client_cb *client_CB = arg;
    int h,i,j,s,t;

    //スクリーンキャプチャ用
    int screen;
    Window rootWindow;

    //ヒストグラム、時系列情報、フリーズ状態判別用
    TS ts;
    int (*seq_hist_h)[MAX_HIST] = NULL;	//時系列ヒストグラム格納用
    int *seq_matching_hist_h = NULL;   	//マッチング結果格納用
    int flag_freeze;            	//フリーズ状態判別用
    int old_freeze_count = 0;		//更新前フリーズ判定回数(初期状態は0とする。)

    /* block all signal in this thread */
    sigset_t set;
    sigfillset(&set);
    pthread_sigmask(SIG_SETMASK, &set, NULL);

    //  初期化
    XInitThreads();
    display = XOpenDisplay(NULL);
    screen = DefaultScreen(display);
    rootWindow = RootWindow(display, screen);
    flag_freeze = 0;
    client_CB->getimage_interval = MIN_GETIMAGE_INTERVAL;
    memset(&ts, 0, sizeof(ts));

    //サーバに操作状態判別要求メッセージを送信する。
    request_opeStateDecision(client_CB);

    //操作状態判別ループ
    while (1) {

      //画像取得の時間間隔が途中で変わっても対応可能にしておく。
      for (s = 1; s <= client_CB->getimage_interval; s++) {
	sleep(1);
      }

      //サーバから操作状態判別メッセージを受信しないと以後の処理はできない。
      if (client_CB->ope_flag == 0)	continue;
 
      //メインループ内変数宣言
      XImage *image;
      XWindowAttributes attr;
      int root_w,root_h;

      dbg("Client Freeze count: %d", client_CB->freeze_count);
      dbg("Client GetImage Interval: %d", client_CB->getimage_interval);

      //フリーズ判定回数が途中で変わったときの処理。
      if (old_freeze_count != client_CB->freeze_count) {

	if (old_freeze_count == 0) {

	  //初期化
	  seq_hist_h = (int (*)[MAX_HIST])calloc(client_CB->freeze_count*MAX_HIST, sizeof(int));
	  seq_matching_hist_h = (int *)malloc(client_CB->freeze_count*sizeof(int));
	  memset(seq_matching_hist_h, -1, client_CB->freeze_count*sizeof(int));
	  tsInit(&ts, client_CB->freeze_count);
	} else {

	  //一時的なデータ保存用。
	  int tmp_hist_h[MAX_HIST], tmp_freeze_count;
	  int *tmp_matching_hist_h = (int *)malloc(old_freeze_count*sizeof(int));

	  //前のヒストグラムのデータを一時的に保持する。
	  memcpy(tmp_hist_h, seq_hist_h[tsPrev(&ts, ts.now)], sizeof(tmp_hist_h));

	  //一時的なフリーズ判定回数を格納する。
	  tmp_freeze_count = (old_freeze_count < client_CB->freeze_count)? old_freeze_count : client_CB->freeze_count;

	  //フリーズのマッチング結果を一時的に保持する。
	  if (ts.now >= client_CB->freeze_count) {
	    memcpy(tmp_matching_hist_h, seq_matching_hist_h + (ts.now - client_CB->freeze_count + 1), tmp_freeze_count*sizeof(int));
	  } else {
	    memcpy(tmp_matching_hist_h, seq_matching_hist_h, tmp_freeze_count*sizeof(int));
	  }

	  tsUpdate(&ts, client_CB->freeze_count);

	  //メモリの再割り当てを行う。
	  seq_hist_h = (int (*)[MAX_HIST])realloc(seq_hist_h, client_CB->freeze_count*MAX_HIST*sizeof(int));
	  seq_matching_hist_h = realloc(seq_matching_hist_h, client_CB->freeze_count*sizeof(int));

	  //初期化
	  memset(seq_hist_h, 0, client_CB->freeze_count*MAX_HIST*sizeof(int));
	  memset(seq_matching_hist_h, -1, client_CB->freeze_count*sizeof(int));

	  //一時的に保持したヒストグラムのデータをコピー
	  memcpy(seq_hist_h[tsPrev(&ts,ts.now)], tmp_hist_h, MAX_HIST*sizeof(int));

	  //一時的に保持したフリーズ判定用のデータをコピー
	  memcpy(seq_matching_hist_h, tmp_matching_hist_h, tmp_freeze_count*sizeof(int));
	  free(tmp_matching_hist_h);
	}

	old_freeze_count = client_CB->freeze_count;
      }

      //初期化
      memset(seq_hist_h[ts.now],0,sizeof(seq_hist_h[ts.now]));
      seq_matching_hist_h[ts.now] = -1;

      //画像取得
      XGetWindowAttributes(display, rootWindow, &attr);
      root_w = attr.width;root_h = attr.height;
      if (attr.map_state != IsViewable
            || attr.width < 10 || attr.height < 10 //シェードの判定
            || (attr.depth != 16 && attr.depth != 24 && attr.depth != 32)
         )
          image = NULL;
      else
          image = XGetImage(display, rootWindow,//focus,//
                            attr.x, attr.y, attr.width, attr.height,
                            AllPlanes, ZPixmap);

      if (image == NULL)
      {
          printf("failed to get image.\n");
      }
      else
      {
          //ヒストグラム計算
          for (j = 0;j < attr.height;j++)
	  {
            for (i = 0;i < attr.width;i++)
            {
              unsigned char *pix;
              int r = 0;
              int g = 0;
              int b = 0;
              int h,s,v;

              if (attr.depth == 16)
              {
                unsigned long buf = XGetPixel(image,i,j);
                b = buf & 0x1F;
                g = (buf >> 6) & 0x1F;
                r = (buf >> 11);
              }
              else if (attr.depth == 24 ||attr.depth == 32)
              {
                 pix =(unsigned char*)&image->data[(j*attr.width+i)*4];
                 r = *(pix+2);
                 g = *(pix+1);
                 b = *pix;
                 r /= 8;g /= 8;b /= 8;
              }

              RGBtoHSV(r,g,b,&h,&s,&v);

              seq_hist_h[ts.now][h]++;
            }
          }
          XDestroyImage(image);

          //ヒストグラム正規化
          int sum = 0;
          for (h = 0;h < MAX_HIST;h++)
          {
                seq_hist_h[ts.now][h] *=
                     (double)NORMALIZE_HIST
                        / (attr.width * attr.height);
                sum+=seq_hist_h[ts.now][h];
          }

          //マッチング
          int intersection = 0;
          int matching_count = 0;
          for (h = 0;h < MAX_HIST;h++)
          {
                intersection +=
                    seq_hist_h[ts.now][h] < seq_hist_h[tsPrev(&ts,ts.now)][h] ?
                    seq_hist_h[ts.now][h] : seq_hist_h[tsPrev(&ts,ts.now)][h];
          }
          seq_matching_hist_h[ts.now] =
                intersection > NORMALIZE_HIST * MATCHING_RATIO ? 1:0;

	  //FREEZE判定
          for (t = 0;t < client_CB->freeze_count;t++)
          {
            if (seq_matching_hist_h[t] == 1) matching_count++;
          }
          if (matching_count == client_CB->freeze_count) flag_freeze = FREEZE;
          else flag_freeze = NORMAL;
          for (t = 0;t < client_CB->freeze_count;t++)
            printf("%d  ",seq_matching_hist_h[t]);
          printf("> %d\n",flag_freeze);
      }

      pthread_mutex_lock(&c_mutex);
      //memcpy(client_CB->hist_h,seq_hist_h[ts.now],sizeof(seq_hist_h[ts.now]));
      memcpy(client_CB->histogram_h, seq_hist_h[ts.now], sizeof(seq_hist_h[ts.now]));
      //client_CB->flag = flag_freeze;
      client_CB->client_status = flag_freeze;
      send_histgram (client_CB); //TODO
      pthread_mutex_unlock(&c_mutex);

      tsPutForward(&ts);
    }

    free(seq_hist_h);
    free(seq_matching_hist_h);
}

/*
 * ヒストグラム送信関数
 */
static int send_histgram (struct client_cb *client_CB)
{
    struct client_cb ccb;
    HistogramAdv histogramadv;
    int rep;

    if (client_CB->ctldsp == 0) return False;

    //pthread_mutex_lock(&i_mutex);
    memcpy(&ccb, client_CB, sizeof(struct client_cb));
    //pthread_mutex_unlock(&i_mutex);
    
    /* Create Histogram Packet */
    histogramadv.type    = rfbHistogramAdv;
    histogramadv.flag    = 0x4; //TODO
    histogramadv.status  = ccb.client_status;
    histogramadv.padding = 0x0;
    memcpy(histogramadv.histogram_h, ccb.histogram_h, MAX_HIST);
    memcpy(histogramadv.histogram_s, ccb.histogram_s, MAX_HIST);
    memcpy(histogramadv.histogram_v, ccb.histogram_v, MAX_HIST);

    //TODO
    /* Send Histogram */
    //pthread_mutex_lock(&i_mutex);
    rep = send (ccb.ctldsp, &histogramadv, sizeof(HistogramAdv), 0);
    if (rep != sizeof(HistogramAdv)) {
	err("Warning Can't send histogram data.");
	return False;
    }
    //pthread_mutex_unlock(&i_mutex);


    dbg("client_send_histogram client_ID: %s", ccb.client_ID);

    return True;
}

/*
 * 操作状態判別要求メッセージを送信する関数
 */
void request_opeStateDecision (struct client_cb *client_CB)
{
    OperateStateDecisionRequestAdv osdradv;

    bzero(&osdradv, sizeof(osdradv));
    osdradv.type = rfbOperateStateDecisionRequestAdv;
    osdradv.flag = 0x0;

    if (send(client_CB->ctldsp, &osdradv, sizeof(osdradv), 0) == -1){
      err("Warning Can't send operate state decision request message.");
    }
}
