/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/*
 * misc.c - miscellaneous functions.
 */

#include "vncviewer_s.h"
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <pthread.h>

static void CleanupSignalHandler(int sig);
static int CleanupXErrorHandler(Display *dpy, XErrorEvent *error);
static int CleanupXIOErrorHandler(Display *dpy);
static void CleanupChildHandler(int sig);

static XErrorHandler defaultXErrorHandler;
static XIOErrorHandler defaultXIOErrorHandler;


/*
 * ToplevelInitBeforeRealization sets the title, geometry and other resources
 * on the toplevel window.
 */

void
ToplevelInitBeforeRealization()
{
  /* Set handlers for signals and X errors to perform cleanup */

  signal(SIGPIPE, SIG_IGN);
  signal(SIGHUP, CleanupSignalHandler);
  signal(SIGINT, CleanupSignalHandler);
  signal(SIGTERM, CleanupSignalHandler);
  signal(SIGCHLD, CleanupChildHandler);
  defaultXErrorHandler = XSetErrorHandler(CleanupXErrorHandler);
  defaultXIOErrorHandler = XSetIOErrorHandler(CleanupXIOErrorHandler);
}


/*
 * Cleanup - perform any cleanup operations prior to exiting.
 */

void
Cleanup()
{
  int i;
  for (i = 0; i < MAXCLIENT; i++) {
    if ((exDB_flag) && (server_CB.client_grp[i] != NULL)) chStateDB(i, 0);
    if (server_CB.cast_src[i] == EMPTY) continue;

    struct sockaddr_in dest;
    dest.sin_family = AF_INET;
    dest.sin_addr.s_addr = server_CB.casting_ip[i];
    dest.sin_port = server_CB.casting_port;
    sendto ((int)server_CB.cstdsp[i], NULL, 0, 0, (struct sockaddr *)&dest, sizeof(dest));

    if (shm_asym[i] == NULL) continue;
    (*shm_asym[i]).flag = CAST_STOP;
    int res = shmdt(shm_asym[i]);
    int ctlck = shmctl(server_CB.client_grp[i]->shm_id, IPC_RMID, 0);
    printf("shm_deleted.(client[%d]=%d) ctlck=%d\n",i,res,ctlck);

    shutdown (server_CB.cstdsp[server_CB.client_grp[i]->cast_grp], SHUT_RDWR);
    close (server_CB.cstdsp[server_CB.client_grp[i]->cast_grp]);
  }

  system("killall multivnc_client >/dev/null 2>&1");
  system("vncserver -kill :1 >/dev/null 2>&1");
}

static int
CleanupXErrorHandler(Display *dpy, XErrorEvent *error)
{
  fprintf(stderr,"CleanupXErrorHandler called\n");
  Cleanup();
  return (*defaultXErrorHandler)(dpy, error);
}

static int
CleanupXIOErrorHandler(Display *dpy)
{
  fprintf(stderr,"CleanupXIOErrorHandler called\n");
  Cleanup();
  return (*defaultXIOErrorHandler)(dpy);
}

static void
CleanupSignalHandler(int sig)
{
  fprintf(stderr,"CleanupSignalHandler called [server pthread ID=%lu]\n", pthread_self());
  Cleanup();
  exit(1);
}

static void CleanupChildHandler(int sig)
{
  fprintf(stderr,"CleanupChildHandler called, [server pthread ID=%lu]\n", pthread_self());

  int status;

  while (waitpid(-1, &status, WNOHANG) > 0);
}

