/*
 * ata.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ATA device driver
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/conf.h>
#include <sys/poll.h>
#include <sys/uio.h>
#include <machine/DevInterrupt.h>
#include <machine/Cpu.h>
#include <lib/lib.h>
#include <lib/lib_path.h>
#include <kern/lock.h>
#include <kern/proc.h>
#include <kern/time.h>
#include <kern/timer.h>
#include <kern/errno.h>
#include <kern/kmalloc.h>
#include <machine/mp.h>
#include <machine/sp.h>
#include <kern/fs.h>
#include <kern/devfs.h>
#include <dev/console/console.h>
#include <dev/pci/pci.h>
#include <dev/block_partition.h>
#include "ata_dma.h"

#include <kern/debug.h>


//#define DEBUG_ATA 1
#ifdef DEBUG_ATA
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


enum{
	IDENTIFY_SIZE	= 512,		/* ATA ATAPI identify buffer size */
	ATA_SECTOR_SIZE	= 512,		/* ATA disk sector size */

	/* IO register */
	PRIM_BASE = 0x1F0,
	SECN_BASE = 0x170,

	/* status bit */
	BSY_BIT	= 0x80,
	DF_BIT	= 0x20,
	SRV_BIT	= 0x10,
	DRQ_BIT	= 0x8,
	ERR_BIT	= 0x1,
	CHK_BIT	= 0x1,

	/* IRQ number */
	PRIM_IRQ	= 14,
	SECOND_IRQ	= 15,

	/* Transfer mode bit in identify infomation */
	IDENT_PIO4		= 0x2,		/* PIO mode4 */
	IDENT_PIO3		= 0x1,		/* PIO mode3 */
	IDENT_M_DMA2	= 0x4,		/* Multi DMA mode2 */
	IDENT_M_DMA1	= 0x2,		/* Multi DMA mode1 */
	IDENT_M_DMA0	= 0x1,		/* Multi DMA mode0 */
	IDENT_U_DMA6	= 0x40,		/* Ultra DMA mode6 */
	IDENT_U_DMA5	= 0x20,		/* Ultra DMA mode5 */
	IDENT_U_DMA4	= 0x10,		/* Ultra DMA mode4 */
	IDENT_U_DMA3	= 0x8,		/* Ultra DMA mode3 */
	IDENT_U_DMA2	= 0x4,		/* Ultra DMA mode2 */
	IDENT_U_DMA1	= 0x2,		/* Ultra DMA mode1 */
	IDENT_U_DMA0	= 0x1,		/* Ultra DMA mode0 */

	/* ǥХ */
	NODEVICE = 0,
	ATA,
	ATAPI,

	/* ž */
	PIO = 1,	/* PIO transfer mode */
	M_DMA,		/* Multi DMA transfer mode */
	U_DMA,		/* Ultra DMA transfer mode */

	/* Interrupt mode */
	INTR_DISABLE = 0,
	INTR_ENABLE,

	// κž
	IO_ATA_SECTORS_MAX = 256,
	IO_ATAPI_SECTORS_MAX = 65536,
};

/* ATA IO register. */
typedef struct{
	int dtr;	/* ATA_DTR=0 */
	int err;	/* ATA_ERR=1 */
	int ftr;	/* ATA_FTR=1 */
    int scr;	/* ATA_SCR=2 */
    int irr;	/* ATA_IRR=2 */
    int snr;	/* ATA_SNR=3 */
    int clr;	/* ATA_CLR=4 */
    int blr;	/* ATA_BLR=4 */
    int chr;	/* ATA_CHR=5 */
    int bhr;	/* ATA_BHR=5 */
    int dhr;	/* ATA_DHR=6 */
    int str;	/* ATA_STR=7 */
    int cmr;	/* ATA_CMR=7 */
    int ctr;	/* ATA_CTR=0x206 */
    int astr;	/* ATA_ASTR=0x206 */
}ATA_REG;

/* ATA ATAPI identify infomation(ATA-5) */
typedef struct{
	ushort	config;				/* 0 㤤 */
	ushort	n_cyl;				/* 1 ATA only */
	ushort	sub_command;		/* 2 */
	ushort	n_head;				/* 3 ATA only */
	ushort	vender_def1[2];		/* 4 */
	ushort	n_sect;				/* 6 ATA only */
	ushort	reserv1[3];			/* 7 */
	uchar	serial_number[20];	/* 10 */
	ushort	nodef1[3];			/* 20 */
	uchar	firm_ware[8];		/* 23 */
	uchar	model[40];			/* 27 */
	ushort	multi_intr;			/* 47 ATA only */
	ushort	reserv2;			/* 48 */
	ushort	iordy;				/* 49 㤤 */
	ushort	stby_timer;			/* 50 ATA only */
	ushort	nodef2[2];			/* 51 */
	ushort	word_enable;		/* 53 */
	ushort	current_n_cyl;		/* 54 ATA only */
	ushort	current_n_head;		/* 55 ATA only */
	ushort	current_n_sect;		/* 56 ATA only */
	ushort	all_sect[2];		/* 57 ATA only */
	ushort	multiple;			/* 59 ATA only */
	ushort	lba_all_sect[2];	/* 60 ATA only */
	ushort	nodef3;				/* 62 */
	ushort	multi_dma;			/* 63 */
	ushort	pio;				/* 64 */
	ushort	min_mdma_cycl;		/* 65 */
	ushort	rcm_mdma_cycl;		/* 66 */
	ushort	min_flw_pio_cycl;	/* 67 */
	ushort	min_nflw_pio_cycl;	/* 68 */
	ushort	reserv3[2];			/* 69 */
	ushort	pck_bus_release;	/* 71 ATAPI only */
	ushort	bsy_clear_time;		/* 72 ATAPI only */
	ushort	reserv4[2];			/* 73 */
	ushort	max_cue_size;		/* 75 */
	ushort	reserv5[4];			/* 76 */
	ushort	major_num;			/* 80 */
	ushort	minor_num;			/* 81 */
	ushort	cmd1;				/* 82 */
	ushort	cmd2;				/* 83 */
	ushort	cmd_ext;			/* 84 */
	ushort	cmd1_enable;		/* 85 */
	ushort	cmd2_enable;		/* 86 */
	ushort	cmd_ext_enable;		/* 87 */
	ushort	ultra_dma;			/* 88 */
	ushort	secu_erase_time;	/* 89 ATA only */
	ushort	esecu_erase_time;	/* 90 ATA only */
	ushort	pmm_value;			/* 91 ATA only */
	ushort	pass_rev_coad;		/* 92 ATA only */
	ushort	hard_reset_info;	/* 93 */
	ushort	reserv6[32];		/* 94 */
	ushort	atapi_byte_count;	/* 126 ATAPI only */
	ushort	remov_set;			/* 127 */
	ushort	secu_stat;			/* 128 */
	ushort	vender_def2[31];	/* 129 */
	ushort	cfa_pm;				/* 160 */
} IDENTIFY;

/* Conect device. */
typedef struct{
	int type;				/* ATA or ATAPI */
	int mode;				/* PIO=0, Multi DMA=1, Ultra DMA=2 */
	int sectorSize;			/* Secter size */
	uint allSectors;		/* LBA all sectors */
	int flag;				/* Function flag */
	int (*transfer)(int, int, int, int, size_t, int, struct iovec*); /* Tranfer function */
}CONECT_DEV;

/* ATA IO register */
static ATA_REG reg[2]={
	{PRIM_BASE+0,PRIM_BASE+1,PRIM_BASE+1,PRIM_BASE+2,PRIM_BASE+2,PRIM_BASE+3,PRIM_BASE+4,PRIM_BASE+4,
	 PRIM_BASE+5,PRIM_BASE+5,PRIM_BASE+6,PRIM_BASE+7,PRIM_BASE+7,PRIM_BASE+0x206,PRIM_BASE+0x206},
	{SECN_BASE+0,SECN_BASE+1,SECN_BASE+1,SECN_BASE+2,SECN_BASE+2,SECN_BASE+3,SECN_BASE+4,SECN_BASE+4,
	 SECN_BASE+5,SECN_BASE+5,SECN_BASE+6,SECN_BASE+7,SECN_BASE+7,SECN_BASE+0x206,SECN_BASE+0x206}
};
static CONECT_DEV conectDev[2][2];	// Conect device infomation.
static WAIT_INTR waitIntr[2];		// Ԥѹ¤
static uchar irqNum[2] = {			// IRQֹ
	PRIM_IRQ,
	SECOND_IRQ
};

/************************************************************************************************
 *
 * 顼å
 *
 ************************************************************************************************/

/*
 * Check busy flag in status register
 * return : ơ
 */
static int checkBusy(
	const int str)		// ơ쥸
{
	enum{
		TIME_OUT = 5000,		/* Time out ms */
	};
	uchar in;
	uint64 beginClock;

	beginClock = rdtsc();
	while ((in = inb(str)) & BSY_BIT) {
		if (TIME_OUT < getMiliSecondsFromClock(beginClock)) {
			return in;
		}
	}

	return in;
}

/************************************************************************************************
 *
 * ATA ޥ
 *
 ************************************************************************************************/

static int currentIntr[2]		/* Current host interrupt mode,enable=1 or diable=0 */
	={0,0};

/*
 * եȥꥻå
 * return : Error number
 */
static int softReset(int host)
{
	outb(reg[host].ctr,0x4);	/* եȥꥻå */
	mili_timer(5);				/* 5ms wait */
	outb(reg[host].ctr,0x2);	/* ꥻåȲ|߶ػ */
/*	mili_timer(20);*/			/* 20ms wait */
	mili_timer(5);				/* 5ms wait */
	if (checkBusy(reg[host].astr) & BSY_BIT) {
		return -EDBUSY;
	}

	return NOERR;
}

/*
 * Set interrupt mode
 */
static void setIntr(int host,int flag)
{
	if (flag == currentIntr[host]) {
		return;
	}

	if (flag == INTR_DISABLE) {
		set_irq_mask(irqNum[host]);
		outb(reg[host].ctr,0x2);			/* Disable interrupt */
		mili_timer(5);						/* wait */
		currentIntr[host] = INTR_DISABLE;
	}
	else {
		outb(reg[host].ctr,0);				/* Enable interrupt */
		release_irq_mask(irqNum[host]);
		mili_timer(5);						/* wait */
		currentIntr[host] = INTR_ENABLE;
	}
}

/*
 * PIO read data
 * return : Status coad
 */
static int readPioIovec(
	int host,
	int dev,
	int sectorBytes,
	int iovecCnt,			// iovec
	struct iovec *iovec)
{
	int i;

	for (i = 0; i < iovecCnt; ++i) {
		short *readBuf = (short*) iovec[i].iov_base;
		int sectors = iovec[i].iov_len / sectorBytes;
		int j;

		// ñ̤IO
		for (j = 0; j < sectors; ++j) {
			int error = checkBusy(reg[host].str);
			int k;

			if ((error & (BSY_BIT | DRQ_BIT)) != DRQ_BIT) {
				return error;
			}

			// wordñ̤ɤ߹
			for (k = 0; k < sectorBytes / 2; ++k) {
				readBuf[k] = inw(reg[host].dtr);
			}
			readBuf += sectorBytes / 2;
		}
	}

	return checkBusy(reg[host].str);
}

/*
 * PIO write data
 * return : Status coad
 */
static int writePioIovec(
	int host,
	int dev,
	int sectorBytes,
	int iovecCnt,			// iovec
	struct iovec *iovec)
{
	int i;

	for (i = 0; i < iovecCnt; ++i) {
		short *writeBuf = (short*) iovec[i].iov_base;
		int sectors = iovec[i].iov_len / sectorBytes;
		int j;

		// ñ̤IO
		for (j = 0; j < sectors; ++j) {
			int error = checkBusy(reg[host].str);
			int k;

			if ((error & (BSY_BIT | DRQ_BIT)) != DRQ_BIT) {
				return error;
			}

			// wordñ̤ǽ񤭹
			for (k = 0; k < sectorBytes / 2; ++k) {
				outw(reg[host].dtr, writeBuf[k]);
			}
			writeBuf += sectorBytes / 2;
		}
	}

	return checkBusy(reg[host].str);
}

/*
 * PIO read data
 * return : Status coad
 */
static int readPio(
	const int host,
	const size_t bytes,
	void *buf)
{
	short *readBuf = buf;
	int status;
	int i, last;

	status = checkBusy(reg[host].str);
	if ((status & (BSY_BIT | DRQ_BIT)) != DRQ_BIT) {
		return status;
	}

	last = bytes / 2;
	for (i = 0; i < last; ++i) {
		readBuf[i] = inw(reg[host].dtr);
	}

	return checkBusy(reg[host].str);
}

/*
 * ǥХ쥯
 * return : Error number
 */
static int deviceSelect(int host, int dev, int lba24_27)
{
	enum{
		LBA_BIT=0x40,		/* LBA bit in Device_head register */
	};
	int status;

	status = checkBusy(reg[host].astr);
	if (status & DRQ_BIT) {
		return -EDERRE;
	}
	if (status & BSY_BIT) {
		return -EDBUSY;
	}

	/* Device select */
	outb(reg[host].dhr, (uchar) (LBA_BIT | (dev << 4) | lba24_27));
	micro_timer(1);						/* 400ns wait */

	status = checkBusy(reg[host].astr);
	if (status & DRQ_BIT) {
		return -EDERRE;
	}
	if (status & BSY_BIT) {
		return -EDBUSY;
	}

	return NOERR;
}

/*
 * ATA IOޥɤȯ
 *աեϣʲ
 * return : error number
 */
static int transferAtaCmd(
	const int host,
	const int dev,
	const int ioMode,
	const size_t beginSector,	// IOϥ
	const size_t sectors,		// IO
	const int iovcnt,
	struct iovec *iovec)
{
	int error;

	error = deviceSelect(host, dev, beginSector >> 24);
	if (error != 0) {
		return error;
	}
	outb(reg[host].scr, (uchar) sectors);
	outb(reg[host].snr, (uchar) beginSector);
	outb(reg[host].clr, (uchar) (beginSector >> 8));
	outb(reg[host].chr, (uchar) (beginSector >> 16));
	
	if (conectDev[host][dev].mode == PIO){
		// IOؿ
		if (ioMode == UIO_READ) {
			outb(reg[host].cmr, 0x20);
			error = readPioIovec(host, dev, ATA_SECTOR_SIZE, iovcnt, iovec);
		}
		else {
			outb(reg[host].cmr, 0x30);
			error = writePioIovec(host, dev, ATA_SECTOR_SIZE, iovcnt, iovec);
		}
	}
	else{
		// IOؿ
		if (ioMode == UIO_READ) {
			outb(reg[host].cmr, 0xc8);
			readDmaIovec(host, iovcnt, iovec, waitIntr);
		}
		else {
			outb(reg[host].cmr, 0xca);
			writeDmaIovec(host, iovcnt, iovec, waitIntr);
		}
		error = inb(reg[host].str);
	}
	
	if (error & (DRQ_BIT | ERR_BIT)) {
		return -EDERRE;
	}
	if (error & BSY_BIT) {
		return -EDBUSY;
	}

	return NOERR;
}

/*
 * ATA data transfer protocol
 * return : error number
 */
static int transferAta(
	int host,
	int dev,
	int ioMode,
	int dummy,
	size_t i_begin,			// ϥֹ
	int iovecCnt,			// iovec
	struct iovec *iovec)
{
	size_t begin = i_begin;
	size_t bytes;
	int iovecBegin;
	int i;
	int error;

	if (conectDev[host][dev].mode == PIO){
		setIntr(host,INTR_DISABLE);
	}
	else{
		setIntr(host,INTR_ENABLE);
	}

	/*
	 * IO
	 * κIOХȿ IO_ATA_SECTORS_MAX * Хȿ
	 */
	bytes = 0;
	iovecBegin = 0;
	for (i = 0; i < iovecCnt;) {
		bytes += iovec[i].iov_len;
		++i;

		if (IO_ATA_SECTORS_MAX * ATA_SECTOR_SIZE <= bytes) {
			if (IO_ATA_SECTORS_MAX * ATA_SECTOR_SIZE < bytes) {
				bytes -= iovec[--i].iov_len;
			}

			error = transferAtaCmd(host, dev, ioMode, begin, bytes / ATA_SECTOR_SIZE, i - iovecBegin, &iovec[iovecBegin]);
			if (error != NOERR) {
				return error;
			}

			begin += bytes / ATA_SECTOR_SIZE;
			iovecBegin = i;
			bytes = 0;
		}
		else if (iovecCnt <= i) {
			error = transferAtaCmd(host, dev, ioMode, begin, bytes / ATA_SECTOR_SIZE, i - iovecBegin, &iovec[iovecBegin]);
			if (error != NOERR) {
				return error;
			}
		}
	}

	return NOERR;
}

/************************************************************************************************
 *
 * < ATA command >
 *
 ************************************************************************************************/

enum{
	/* Set features subcommand code */
	SET_TRANSFER=0x3,	/* Set transfer mode */

	/* Transfer mode of set features subcommand */
	SUB_PIO_DEF=0x0,
	SUB_PIO_FLO=0x8,
	SUB_M_DMA=0x20,
	SUB_U_DMA=0x40,
};

/*
 * Reset device
 * return : Error number
 */
static int resetDevice(int host,int dev)
{
	int status;

	setIntr(host,INTR_DISABLE);

	/* Device select */
	outb(reg[host].dhr,(uchar)(dev<<4));
	micro_timer(1);			/* 400ns wait */

	outb(reg[host].cmr,0x8);
	micro_timer(1);			/* 400ns wait */

	status = checkBusy(reg[host].astr);
	if (status & ERR_BIT) {
		return -EDERRE;
	}
	if (status & BSY_BIT) {
		return -EDBUSY;
	}

	return NOERR;
}

/*
 * Identif device
 * return : Error number
 */
static int identifyDevice(int host, int dev, int drv, void *buf)
{
	int status;
	int error;

	setIntr(host,INTR_DISABLE);

	error = deviceSelect(host, dev, 0);
	if (error != NOERR) {
		return error;
	}

	outb(reg[host].cmr, (drv == ATA) ? 0xec : 0xa1);
	micro_timer(1);					/* 400ns wait */

	/* Read data */
	status = readPio(host, IDENTIFY_SIZE, buf);
	if (status & (DRQ_BIT | ERR_BIT)) {
		return -EDERRE;
	}
	if (error & BSY_BIT) {
		return -EDBUSY;
	}

	return NOERR;
}

/*
 * Idle device
 * return : Error number
 */
static int idleImmediateDevice(int host, int dev)
{
	int error;
	int status;

	setIntr(host,INTR_DISABLE);

	error = deviceSelect(host, dev, 0);
	if (error != 0) {
		return error;
	}

	outb(reg[host].cmr,0xe1);
	micro_timer(1);			/* 400ns wait */

	status = checkBusy(reg[host].astr);
	if (status & ERR_BIT) {
		return -EDERRE;
	}
	if (status & BSY_BIT) {
		return -EDBUSY;
	}

	return NOERR;
}

/*
 * Initialize device parameters
 * return : Error number
 */
static int initDeviceParam(int host, int dev, uchar head, uchar c_sector)
{
	int error;
	int status;

	if (--head > 0xf){
		return -EINVAL;	/* headϥإåɿ-1ͤȤ롣 */
	}

	setIntr(host,INTR_DISABLE);

	error = deviceSelect(host, dev, head);
	if (error != 0) {
		return error;
	}

	outb(reg[host].scr, c_sector);
	outb(reg[host].cmr, 0x91);
	micro_timer(1);			/* 400ns wait */

	status = checkBusy(reg[host].astr);
	if (status & ERR_BIT) {
		return -EDERRE;
	}
	if (status & BSY_BIT) {
		return -EDBUSY;
	}

	return NOERR;
}

/*
 * ǥХư
 * return : Error number
 */
static int setFeatures(int host, int dev, uchar subcommand, uchar transMode)
{
	int error;
	int status;

	setIntr(host,INTR_DISABLE);

	error = deviceSelect(host, dev, 0);
	if (error != 0) {
		return error;
	}

	outb(reg[host].ftr,subcommand);
	outb(reg[host].scr,transMode);
	outb(reg[host].cmr,0xef);
	micro_timer(1);			/* 400ns wait */

	status = checkBusy(reg[host].astr);
	if (status & ERR_BIT) {
		return -EDERRE;
	}
	if (status & BSY_BIT) {
		return -EDBUSY;
	}

	return NOERR;
}

/************************************************************************************************
 *
 * ѥåȥޥ
 *
 ************************************************************************************************/

enum{
	/* Interrupt reason bit. */
	CD_BIT	= 0x1,
	IO_BIT	= 0x2,
	REL_BIT	= 0x4,

	/* ATAPI function flag. */
	PACK_OVL	= 0x2,			/* Packet feature overlappe flag */
	PACK_DMA	= 0x1,			/* Packet feature DMA flag */
	ATAPI_OVL	= 0x2000,		/* Overlappe flag in ATAPI function */

	/* Packet start stop unit flag */
	/*SSU_STOP=0,*/				/* Disk stop */
	/*SSU_START=0x1,*/			/* Disk start */
	/*SSU_EJECT=0x2,*/			/* Disk eject */
	/*SSU_STANBY=0x30,*/		/* Stanby */
};

/* Packet command parameters */
typedef struct{
	void *buf;				// Block read write buffer
	int size;				// Transfer bytes or Secter size
	int iovecCnt;			// iovec
	struct iovec *iovec;
	uchar feutures;			// Packet command flag
	uchar packet[14];		// Packet command parameters and return packet
} PACKET_PARAM;

/*
 * Issue packet command
 * return : 0 or Error number
 */
static int issuePacketCommand(
	int host,
	int dev,
	PACKET_PARAM *param)
{
	int error;
	int status;
	int i;

	error = deviceSelect(host, dev, 0);
	if (error != 0) {
		return error;
	}

	// ATAޥɤȯ
	outb(reg[host].ftr, param->feutures);
	outb(reg[host].scr, 0);
	outb(reg[host].clr, 0xff);
	outb(reg[host].chr, 0xff);
	outb(reg[host].cmr, 0xa0);
	micro_timer(1);			/* 400ns wait */
	if ((checkBusy(reg[host].str) & (DRQ_BIT | CHK_BIT)) != DRQ_BIT) {
		return -EDERRE;
	}
	if ((inb(reg[host].irr) & (CD_BIT | IO_BIT)) != CD_BIT) {
		return -EDERRE;
	}

	if (param->feutures & PACK_DMA) {
		// DMAž

		setIntr(host, INTR_ENABLE);

		/* Send packet */
		devInitWait(&waitIntr[host]);			// ǥХԤ֥Ȥν
		for (i = 0; i < 6; ++i) {
			outw(reg[host].dtr, ((short*) param->packet)[i]);
		}

		/* Overrapped */
		if (param->feutures & PACK_OVL) {
			devWaitIntr(&waitIntr[host], 2000);

			status = (int) inb(reg[host].str);
			if (status & ERR_BIT) {
				return -EDERRE;		/* Packet command Error */
			}
			if ((status & DRQ_BIT) == 0) {
				return 0;			/* Non data transfer */
			}

			devInitWait(&waitIntr[host]);			// ǥХԤ֥Ȥν
			error = deviceSelect(host, dev, 0);
			if (error != NOERR) {
				return error;
			}
			devWaitIntr(&waitIntr[host], 2000);	/* Wait interrupt */

			outb(reg[host].cmr,0xa2);		/* Issue service command */
			if((checkBusy(reg[host].str)&(BSY_BIT|DRQ_BIT))!=DRQ_BIT)
				return -EDBUSY;
		}

		/* Data transfer */
		if (param->packet[0] == 0x28) {
			readDmaIovec(host, param->iovecCnt, param->iovec, waitIntr);
			
		}
		else if (param->packet[0] == 0x2a) {
			writeDmaIovec(host, param->iovecCnt, param->iovec, waitIntr);
		}
		else {
			struct iovec *iovec = kmalloc(sizeof(*iovec));

			if (iovec == NULL) {
				return -ENOMEM;
			}
			iovec->iov_base = param->buf;
			iovec->iov_len = param->size;
			readDmaIovec(host, 1, iovec, waitIntr);
			kfree(iovec);
		}
		status = inb(reg[host].str);
	}
	else {
		// PIOž

		setIntr(host,INTR_DISABLE);

		/* Send packet */
		for (i = 0; i < 6; ++i) {
			outw(reg[host].dtr, ((short*) param->packet)[i]);
		}

		status = checkBusy(reg[host].str);
		if (status & ERR_BIT) {
			return -EDERRE;		/* Packet command Error */
		}
		if ((status & DRQ_BIT) == 0) {
			return 0;			/* Non data transfer */
		}

		/* Overrapped */
		if (param->feutures & PACK_OVL) {
			devInitWait(&waitIntr[host]);			// ǥХԤ֥Ȥν
			error = deviceSelect(host, dev, 0);
			if (error != NOERR) {
				return error;
			}
			devWaitIntr(&waitIntr[host], 2000);			/* Wait interrupt */

			outb(reg[host].cmr, 0xa2);					/* Issue service command */
			if ((checkBusy(reg[host].str) & (BSY_BIT | DRQ_BIT)) != DRQ_BIT) {
				return -EDBUSY;
			}
		}

		/* Data transfer */
		if (param->packet[0] == 0x28) {
			status = readPioIovec(host, dev, conectDev[host][dev].sectorSize, param->iovecCnt, param->iovec);
		}
		else if (param->packet[0] == 0x2a) {
			status = writePioIovec(host, dev, conectDev[host][dev].sectorSize, param->iovecCnt, param->iovec);
		}
		else {
			status = readPio(host, param->size, param->buf);
		}
	}

	if (status & (DRQ_BIT | ERR_BIT)) {
		return -EDERRE;
	}
	if (status & BSY_BIT) {
		return -EDBUSY;
	}

	return NOERR;
}

/*
 * Test unit ready
 * return : 0 or Error number
 */
static int testUnitReady(int host,int dev)
{
	PACKET_PARAM param;

	/* Set packet parameters */
	param.feutures=0;
	param.size=0;
	memset(param.packet,0,12);
	param.packet[0]=0;

	return issuePacketCommand(host,dev,&param);
}

/*
 * Request sense
 * return : Ready=0 or Retry=1 or Error number
 */
static int requestSense(int host,int dev)
{
	static char buf[14];
	PACKET_PARAM param;

	/* Set packet parameters */
	param.feutures=conectDev[host][dev].mode>>1;
	param.size=14;
	param.buf=buf;
	memset(param.packet,0,12);
	param.packet[0]=0x3;
	param.packet[4]=14;

	if (issuePacketCommand(host,dev,&param)!=0)return -1;
	switch((((uint)buf[2]&0xf)<<16)|((uint)buf[12]<<8)|((uint)buf[13]))
	{
		case 0x62800:return 0;			/* Not ready to ready change */
		case 0x62900:					/* Power on,Reset */
		case 0x20401:return 1;			/* Becoming ready */
		case 0x63a00:return -ENOMEDIUM;	/* Medium not present */
		default:     return -EDERRE;
	}
}

/*
 * Start stop unit
 * return : 0 or Error number
 */
static int startStopUnit(int host,int dev,uchar ope)
{
	PACKET_PARAM param;

	/* Set packet parameters */
	param.feutures=(conectDev[host][dev].flag&ATAPI_OVL)>>12;
	memset(param.packet,0,12);
	param.packet[0]=0x1b;
	param.packet[4]=ope;

	return issuePacketCommand(host,dev,&param);
}

/*
 * Read capacity
 * return : 0 or Error number
 */
static int readCapacity(int host,int dev)
{
	PACKET_PARAM param;
	char tmp,*buf;
	int error;
	int i,j;


	buf=(char*)&conectDev[host][dev].sectorSize;

	/* Set packet parameters */
	param.feutures=conectDev[host][dev].mode>>1;
	param.size=8;
	param.buf=buf;
	memset(param.packet,0,12);
	param.packet[0]=0x25;

	if((error=issuePacketCommand(host,dev,&param))!=0)return error;

	for(i=0,j=7;i<4;++i,--j)
	{
		tmp=buf[i];
		buf[i]=buf[j];
		buf[j]=tmp;
	}

	/*
	 * 512byteñ̤ڼΤ
	 * ɥ饤֤ˤäƤʪξ礬
	 */
	conectDev[host][dev].sectorSize = ROUNDDOWN(conectDev[host][dev].sectorSize, 512);

	return 0;
}

/*
 * ATA IOޥɤȯ
 *աեϣʲ
 * return : error number
 */
static int transferAtapiCmd(
	const int host,
	const int dev,
	const int ioMode,
	const size_t beginSector,	// IOϥ
	const size_t sectors,		// IO
	const int iovcnt,
	struct iovec *iovec)
{
	PACKET_PARAM param;

	/* Set packet parameters */
	param.feutures = ((conectDev[host][dev].flag & ATAPI_OVL) >> 12) | (conectDev[host][dev].mode >> 1);
	param.iovecCnt = iovcnt;
	param.iovec = iovec;
	memset(param.packet, 0, 12);
	param.packet[0] = (ioMode == UIO_READ) ? 0x28 : 0x2a;
	param.packet[2] = (uchar) beginSector >> 24;
	param.packet[3] = (uchar) beginSector >> 16;
	param.packet[4] = (uchar) beginSector >> 8;
	param.packet[5] = (uchar) beginSector;
	param.packet[7] = (uchar) sectors >> 8;
	param.packet[8] = (uchar) sectors;

	return issuePacketCommand(host, dev, &param);
}

/*
 * ATAPIǡɣ
 * rturn : Error number
 */
static int transferAtapi(
	int host,
	int dev,
	int ioMode,
	int dummy,
	size_t i_begin,			// ϥֹ
	int iovecCnt,			// iovec
	struct iovec *iovec)
{
	size_t begin = i_begin;
	size_t bytes;
	int iovecBegin;
	int i;
	int error;

	/*
	 * IO
	 * κIOХȿ IO_SECTORS_MAX * Хȿ
	 */
	bytes = 0;
	iovecBegin = 0;
	for (i = 0; i < iovecCnt;) {
		bytes += iovec[i].iov_len;
		++i;

		if (IO_ATAPI_SECTORS_MAX * ATA_SECTOR_SIZE <= bytes) {
			if (IO_ATAPI_SECTORS_MAX * ATA_SECTOR_SIZE < bytes) {
				bytes -= iovec[--i].iov_len;
			}

			error = transferAtapiCmd(host, dev, ioMode, begin, bytes / conectDev[host][dev].sectorSize, i - iovecBegin, &iovec[iovecBegin]);
			if (error != NOERR) {
				return error;
			}

			begin += bytes / ATA_SECTOR_SIZE;
			iovecBegin = i;
			bytes = 0;
		}
		else if (iovecCnt <= i) {
			error = transferAtapiCmd(host, dev, ioMode, begin, bytes / conectDev[host][dev].sectorSize, i - iovecBegin, &iovec[iovecBegin]);
			if (error != NOERR) {
				return error;
			}
		}
	}

	return NOERR;
}

/************************************************************************************************
 *
 * < ATA functions calss >
 *
 ************************************************************************************************/

/*
 * IDENTIFY⡼ɤDMA⡼ɤ
 * return : DMA⡼ or NO_DMAMODE
 */
STATIC int getDmaModeFromIdentifyMode(
	const int mode,
	const IDENTIFY *identify)
{
	switch (mode) {
	case U_DMA:
		if (identify->ultra_dma & IDENT_U_DMA5) {
			return UDMA5;
		}
		else if (identify->ultra_dma & IDENT_U_DMA4) {
			return UDMA4;
		}
		else if (identify->ultra_dma & IDENT_U_DMA3) {
			return UDMA3;
		}
		else if (identify->ultra_dma & IDENT_U_DMA2) {
			return UDMA2;
		}
		else if (identify->ultra_dma & IDENT_U_DMA1) {
			return UDMA1;
		}
		else if (identify->ultra_dma & IDENT_U_DMA0) {
			return UDMA0;
		}
		// 
	case M_DMA:
		if (identify->multi_dma & IDENT_M_DMA2) {
			return WDMA2;
		}
		else if (identify->multi_dma & IDENT_M_DMA1) {
			return WDMA1;
		}
		else if (identify->multi_dma & IDENT_M_DMA0) {
			return WDMA0;
		}

		break;
	default:
		ASSERT(0);
	}

	return NO_DMAMODE;
}

/*
 * DMA⡼ɤž⡼ɤ
 * return : ž⡼
 */
STATIC uchar getModeFromDmaMode(
	const int dmaMode)
{
	switch (dmaMode) {
	case UDMA5:
	case UDMA4:
	case UDMA3:
	case UDMA2:
	case UDMA1:
	case UDMA0:
		return U_DMA;
	case WDMA2:
	case WDMA1:
	case WDMA0:
		return M_DMA;
	default:
		ASSERT(0);
	}
	return 0;
}

/*
 * DMA⡼ɤTRANSFER⡼ɤ
 * return : TRANSFER⡼
 */
STATIC uchar getTransModeFromDmaMode(
	const int dmaMode)
{
	switch (dmaMode) {
	case WDMA0:
		return SUB_M_DMA | 0;
	case WDMA1:
		return SUB_M_DMA | 1;
	case WDMA2:
		return SUB_M_DMA | 2;
	case UDMA0:
		return SUB_U_DMA | 0;
	case UDMA1:
		return SUB_U_DMA | 1;
	case UDMA2:
		return SUB_U_DMA | 2;
	case UDMA3:
		return SUB_U_DMA | 3;
	case UDMA4:
		return SUB_U_DMA | 4;
	case UDMA5:
		return SUB_U_DMA | 5;
	default:
		ASSERT(0);
	}

	return 0;
}

/*
 * ž⡼ɤꤹ
 * ž⡼ɤǤʤвΥ٥ž⡼ɤꤹ
 * return : error number
 */
static int changeMode(
	const int host,
	const int dev,
	const int i_mode)
{
	uchar subcm;
	IDENTIFY *identify;
	int mode;
	int error;

	// Ǥ
	if (conectDev[host][dev].mode == i_mode) {
		return NOERR;
	}

	/* Read identify infomation */
	identify = kmalloc(IDENTIFY_SIZE);
	if (identify == NULL) {
		return -EDNOMEM;
	}
	error = identifyDevice(host, dev, conectDev[host][dev].type, identify);
	if (error != NOERR) {
		return error;
	}

	// ƥ⡼ɤ
	switch (i_mode) {
	case U_DMA:
	case M_DMA: {
		int dmaMode = getDmaModeFromIdentifyMode(i_mode, identify);

		if (dmaMode != NO_DMAMODE) {
			error = initDma(host, dev, dmaMode, &dmaMode);
			if (error == NOERR) {
				subcm = getTransModeFromDmaMode(dmaMode);
				mode = getModeFromDmaMode(dmaMode);
				break;
			}
		}
	
		// DMA⡼ɤǤʤPIO⡼ɤꤹ
		// 
	}
	case PIO:
		if (identify->pio & IDENT_PIO4) {
			subcm = SUB_PIO_FLO | 4;
		}
		else if (identify->pio & IDENT_PIO3) {
			subcm = SUB_PIO_FLO | 3;
		}
		else {
			subcm = SUB_PIO_DEF;
		}
		mode = PIO;

		break;
	default:
		return -EDINVAL;
	}

	kfree(identify);
	conectDev[host][dev].mode = mode;

	return setFeatures(host, dev, SET_TRANSFER, subcm);
}

/************************************************************************************************
 *
 * < interrupt handler class >
 *
 ************************************************************************************************/

/*
 * Primary ATA interrupt handler
 * return : Task switch on
 */
static int primIntrHandler(int irq)
{
/***************************************
	printk("IRQ14\n");
***************************************/
	return devWakeIntr(&waitIntr[0]);
}

/*
 * Secondary ATA interrupt handler
 * return : Task switch on
 */
static int secondIntrHandler(int irq)
{
/***************************************
	printk("IRQ15\n");
***************************************/
	return devWakeIntr(&waitIntr[1]);
}

/************************************************************************************************
 *
 * < System call interface class >
 *
 ************************************************************************************************/

/* ԤWait queue */
static MODIFY_WAIT_LOCK waitQueue[2];

/*
 * Test atapi and read capacity
 * return : Redy=0 or Error number
 */
static int testAtapi(dev_t dev, int host, int devNum)
{
	int error;

	for (; testUnitReady(host, devNum) != 0;) {
		error = requestSense(host, devNum);
		if (error == 0) {
			break;
		}
		else if (error < 0) {
			return error;
		}
	}

	/* Read sector size and max sector number */
	if ((error = readCapacity(host, devNum)) != 0) {
		return error;
	}
	dev->beginSector = 0;
	dev->sectSize = conectDev[host][devNum].sectorSize;
	dev->sectors = conectDev[host][devNum].allSectors;
//	hd_info[host][devNum].last_blk = conectDev[host][devNum].allSectors - 1;
//	hd_info[host][devNum].sector_size = conectDev[host][devNum].sectorSize;

	return 0;
}

/*
 * File operation interface
 */
static int open(int host, int devNum, dev_t dev, int oflags, int devtype, d_thread_t *proc)
{
	dev->si_drv1 = (void*) host;
	dev->si_drv2 = (void*) devNum;

	switch (conectDev[host][devNum].type) {
	case ATA:
//		dev->beginSector = 0;
		dev->sectSize = ATA_SECTOR_SIZE;
//		dev->sectors = conectDev[host][devNum].allSectors;
		return 0;
	case ATAPI:
		return testAtapi(dev, host, devNum);
	default:
		return -EDNODEV;
	}
}

static int openHda(dev_t dev, int oflags, int devtype, d_thread_t *proc)
{
	return open(0, 0, dev, oflags, devtype, proc);
}

static int openHdb(dev_t dev, int oflags, int devtype, d_thread_t *proc)
{
	return open(0, 1, dev, oflags, devtype, proc);
}

static int openHdc(dev_t dev, int oflags, int devtype, d_thread_t *proc)
{
	return open(1, 0, dev, oflags, devtype, proc);
}

static int openHdd(dev_t dev, int oflags, int devtype, d_thread_t *proc)
{
	return open(1, 1, dev, oflags, devtype, proc);
}

static int close(dev_t dev, int fflag, int devtype, d_thread_t *proc)
{
	return 0;
}

static int io(dev_t dev, struct uio *uio, int ioflag)
{
	int host = (int)dev->si_drv1;
	int devNum = (int)dev->si_drv2;
	size_t beginSector = uio->uio_offset / dev->sectSize;
	int rest;

	ASSERT(uio->uio_offset % dev->sectSize == 0);
	ASSERT(uio->uio_iov->iov_len % dev->sectSize == 0);

	if (conectDev[host][devNum].allSectors <= beginSector) {
		return -EIO;
	}

	uio->uio_resid = 0;

	modifyWaitLock(&waitQueue[host]);
	{
		rest = conectDev[host][devNum].transfer(host, devNum, uio->uio_rw, dev->sectSize, beginSector, uio->uio_iovcnt, uio->uio_iov);
	}
	modifyWaitUnlock(&waitQueue[host]);

	return rest;
}

static int ioctl(dev_t dev, u_long cmd, caddr_t data, int fflag, d_thread_t *p)
{
	return 0;
}

static int poll(dev_t dev, int events, d_thread_t *p)
{
	return POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
}

static struct cdevsw hd_info[2][2]={
	{{openHda, close, io, io, ioctl, poll, NULL, NULL, "hda", 0, NULL,  NULL, D_DISK,},
	 {openHdb, close, io, io, ioctl, poll, NULL, NULL, "hdb", 0, NULL,  NULL, D_DISK,}},
	{{openHdc, close, io, io, ioctl, poll, NULL, NULL, "hdc", 0, NULL,  NULL, D_DISK,},
	 {openHdd, close, io, io, ioctl, poll, NULL, NULL, "hdd", 0, NULL,  NULL, D_DISK,}}
};

/************************************************************************************************
 *
 * < ATA initialize class >
 *
 ************************************************************************************************/

/*
 * ɤΥӥåǥȥ륨ǥѤ
 * return : string address
 */
static char *convertBigToLittle(char *str,int len)
{
	char c;
	int count;
	int i;

	/*
	 * ' 'ĤĤŤ'\0'
	 */
	count = 0;
	for (i = 0; i < len; ++i) {
		c = str[i];
		str[i] = str[i + 1];
		str[i + 1] = c;

		if (str[i]==' ') ++count;
		else count = 0;
		if (count == 2) break;
		if(str[++i]==' ')++count;
		else count = 0;
		if (count == 2) break;
	}
	str[--i] = '\0';

	return str;
}

//================================== PUBLIC =============================================

/*
 * ATAν
 * return : 0 or Error number
 */
int initAta()
{
	uchar cl, ch;
	int error;
	IDENTIFY *identify;
	int i, j, k;

	identify = kmalloc(IDENTIFY_SIZE);
	if (identify == NULL) {
		return -EDNOMEM;
	}

	// ȥåν
	initModifyWaitLock(&waitQueue[0]);
	initModifyWaitLock(&waitQueue[1]);

	/* ³ǥХȽꤹ */
	for (i = 0; i < 2; ++i) {
		/* եȥꥻå */
		if (softReset(i) != NOERR) {
			continue;
		}

		for (j = 0; j < 2; ++j) {
			/*
			 * եȥꥻåȸξ֡
			 *  ATA:ATA_CLR=0 ATA_CHR=0, ATAPI:ATA_CLR=0x14 ATA_CHR=0xeb
			 */
			outb(reg[i].dhr,j << 4);
			mili_timer(5);
			cl = inb(reg[i].clr);
			ch = inb(reg[i].chr);

			/* ATA  */
			if ((cl == 0) && (ch == 0)) {
				identify->model[0] = 0xff;
				if (identifyDevice(i, j, ATA, identify) != NOERR) {
					continue;		/* Read identify infomation */
				}
				if (identify->model[0] == 0xff)continue;				/* ɤ߽ФƤ뤫å */

				/* Print device infomation */
				convertBigToLittle(identify->model, 40);
				printk("%s : %s, %s\n", hd_info[i][j].d_name, identify->model, "ATA DISK drive");

				/* LBA all sectors */
				conectDev[i][j].allSectors = (uint) identify->lba_all_sect[1] << 16 | (uint) identify->lba_all_sect[0];
				if (conectDev[i][j].allSectors == 0) {
					printk("This device is not support LBA. Stop initialize!");
					continue;
				}
				conectDev[i][j].type = ATA;
				conectDev[i][j].transfer = transferAta;

				/* Init device parameters */
				initDeviceParam(i, j, (uchar)identify->n_head, identify->n_sect);

//				hd_info[i][j].last_blk=conectDev[i][j].allSectors;
//				hd_info[i][j].sector_size=ATA_SECTOR_SIZE;
			}
			/* ATAPI device. */
			else if ((cl == 0x14) && (ch == 0xeb)) {
				char *media;

				if (identifyDevice(i, j, ATAPI, identify) != NOERR) {
					continue;	/* Read identify infomation */
				}

				/* Print device infomation */
				convertBigToLittle(identify->model, 40);
				switch ((identify->config >> 8) & 0x1f) {		/* Medium name */
				case 0x5:
					media="ATAPI CDROM drive";
					break;
				default:
					media="ATAPI OTHER drive";
					break;
				}
				printk("%s : %s, %s\n", hd_info[i][j].d_name, identify->model, media);

				conectDev[i][j].flag = identify->iordy;
				conectDev[i][j].type = ATAPI;
				conectDev[i][j].transfer = transferAtapi;
			}
			else {
				continue;
			}

			/* Idle device */
			idleImmediateDevice(i, j);

			/* Set DMA or PIO mode. */
			if (changeMode(i, j, M_DMA) != NOERR){
				error = changeMode(i, j, PIO);
				if (error != NOERR) {
					printk("Transfer mode set error : %x\n", error);
				}
			}
		}
	}

	kfree(identify);

	/*
	 * ⤦ٳǧ
	 * ǥХˤäƤϡƱۥȤ¸ߤʤӥ֤ΤޤޤˤʤäƤޤ
	 */
	for (i = 0; i < 2; ++i){
		for (j = 0; j < 2; ++j){
			outb(reg[i].dhr, j << 4);
			mili_timer(5);
			if ((inb(reg[i].str) & BSY_BIT) == BSY_BIT){
				if ((error = softReset(i)) != NOERR){	/* soft reset */
					return error;
				}
				for (k = 0; k < 2; ++k){
					changeMode(i, k, conectDev[i][k].mode);	/* set transfer mode */
				}
			}
		}
	}

	/*
	 * ߤ
	 * 8259PICξޥƤߤݻƤΤǡޥθ
	 * ޡơϥɥ˳ߤȯ롣
	 */
	release_irq_mask(PRIM_IRQ);
	release_irq_mask(SECOND_IRQ);
	mili_timer(2);

	irq_entry[IRQ14] = primIntrHandler;
	irq_entry[IRQ15] = secondIntrHandler;

	/* ǥХե륷ƥϿ */
	for (i = 0; i < 2; ++i) {
		for (j = 0; j < 2; ++j) {
			dev_t dev = NULL;

			if (conectDev[i][j].type != NODEVICE) {
				dev = makeBlkDev(&hd_info[i][j], 0, conectDev[i][j].allSectors, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
				if (dev != NULL) {
					// ǥХե
					makeDevf(hd_info[i][j].d_name, dev, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);

					if (conectDev[i][j].type == ATA) {
						/* ѡƥǥϿ */
						registBlockPartition(dev);
					}
				}
				else {
					printk("initAta() host:%d dev:%d is Failed!\n", i, j );
				}

			}
		}
	}

	return 0;
}
