/*
 * ata_dma.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * PCI BUS MASTER DMA
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <dev/pci/pci.h>
#include <kern/device.h>
#include "ata_dma.h"

#include <kern/debug.h>


//#define DEBUG_ATA_DMA 1
#ifdef DEBUG_ATA_DMA
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


//==================================  ===========================================

//================================== Х륤ݡ ===============================

//================================== PRIVATE ============================================

enum{
	PCI_CONF_BMBASE = 0x20,		/* IDE Bus Master IO base address register in PCI Configration */
	IDE_BMIO_SECOND = 0x8,		/* ꡼ۥȤξ˥쥸ͤ˥ץ饹 */

	/* IDE Bus Master IO register address */
	BMIC_ADDR = 0x0,			/* Command register */
	BMIS_ADDR = 0x2,			/* Status register */
	BMID_ADDR = 0x4,			/* Descriptor Table Pointer */

	BMIC_READ		= 0x8,
	BMIC_WRITE		= 0,
	BMIC_START		= 0x1,
	BMIS_ERROR		= 0x2,
	BMIS_INTERRUPT	= 0x4,

	PRD_EOT	= 1 << 31,			/* PRD EOT bit */

	IDE_BM_TIMEOUT = 7000,		/* Bus Master time out ms. */
};

/* Physical Region Descriptor for IDE Busmaster */
typedef struct{
	void *physAddr;		/* Physical address */
	uint count;			/* EOT bit(31)|Transfer count by bytes,Max 64kbyte */
}PRD;

static int busMasterReg[2];		/* IDE Bus Master IO base address */

/*
 * DMAž
 */
STATIC INLINE void dmaIo(
	const int host,
	const int bmicMode,		// Хޥž⡼ɥޥ
	const int iovecCnt,
	struct iovec *iovec,
	WAIT_INTR *waitIntr)
{
	iovec[iovecCnt - 1].iov_len |= PRD_EOT;
	outl(busMasterReg[host] + BMID_ADDR, (uint) iovec);					// Set PRD
	outb(busMasterReg[host] + BMIC_ADDR, bmicMode);						// ž
	outb(busMasterReg[host] + BMIS_ADDR, BMIS_ERROR | BMIS_INTERRUPT);	// Clear interrupt bit and error bit.
	devInitWait(&waitIntr[host]);										// ǥХԤ֥Ȥν
	outb(busMasterReg[host] + BMIC_ADDR, inb(busMasterReg[host] + BMIC_ADDR) | BMIC_START);		// Start Bus Master.
	devWaitIntr(&waitIntr[host], IDE_BM_TIMEOUT);						// Wait interrupt.
	outb(busMasterReg[host] + BMIC_ADDR, inb(busMasterReg[host] + BMIC_ADDR) & ~BMIC_START);	// Stop Bus Master.

	// PRD_EOT򥯥ꥢƸ᤹
	iovec[iovecCnt - 1].iov_len ^= PRD_EOT;
}

//================================== PUBLIC =============================================

/*
 * DMA read data
 */
void readDmaIovec(
	const int host,
	const int iovecCnt,
	struct iovec *iovec,
	WAIT_INTR *waitIntr)
{
	dmaIo(host, BMIC_READ, iovecCnt, iovec, waitIntr);
}

/*
 * DMA write data
 */
void writeDmaIovec(
	const int host,
	const int iovecCnt,			// iovec
	struct iovec *iovec,
	WAIT_INTR *waitIntr)
{
	dmaIo(host, BMIC_WRITE, iovecCnt, iovec, waitIntr);
}

/*
 * Хޥ
 * return : error number
 */
int initDma(
	const int host,
	const int dev,
	const int dmaMode,	// DMA⡼
	int *o_dmaMode)		// DMA⡼
{
	PCI_INFO pci;
	ushort com;
	int error;

	// PCI Configuration Registers IDE Controller Registers õ
	pci.bus = pci.dev = pci.func = 0;
	if (search_pci_class(PCI_CLASS_IDE_BSMST, pci, &pci) < 0) {
		return -ENODEV;
	}

	// Test Bus Master enable bit on
	com = read_pci_config(pci, PCI_CONF_COMMAND);
	writew_pci_config(pci, PCI_CONF_COMMAND, com | PCI_COMMAND_BSMST);
	if ((read_pci_config(pci, PCI_CONF_COMMAND) & PCI_COMMAND_BSMST) == 0) {
		return -EDNOFUNC;
	}

	// DMA⡼ɤꤹ롢ƥåץåȤȤϤɲäƤ
	error = initDmaIntel(host, dev, pci, dmaMode, o_dmaMode);
	if (error != NOERR) {
		if (error != -ENODEV) {
			return error;
		}
	}
	
	if (error != NOERR) {
		// бåץå̵
		return error;
	}

	/* Get IO base address */
	if ((busMasterReg[0] = read_pci_config(pci, PCI_CONF_BMBASE)) == 0) {
		return -EDNOFUNC;
	}
	busMasterReg[0] &= 0xfff0;
	busMasterReg[1] = busMasterReg[0] + IDE_BMIO_SECOND;

	/* Reset Bus Master */
//	outb(busMasterReg[host] + BMIC_ADDR, 0);

	return NOERR;
}
