/*
 * ata_intel.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ƥåץåȤPCI DMAν
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <dev/pci/pci.h>
#include "ata_dma.h"

#include <kern/debug.h>


//#define DEBUG_ATA_INTEL 1
#ifdef DEBUG_ATA_INTEL
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


//==================================  ===========================================

//================================== Х륤ݡ ===============================

//================================== PRIVATE ============================================

/*
 * 80ĥ֥뤬ҤäƤ뤫
 * return : YES or NO
 */
STATIC INLINE int isSet80Cable(
	const int host,			// ATAۥֹ
	const int dev,			// ATAǥХֹ
	const PCI_INFO pci)
{
	int cableBit = 4 + host * 2 + dev;
	u_int32_t value = read_pci_config(pci, 0x54);

	return (value & ((u_int32_t) 1 << cableBit)) ? YES : NO;
}

/*
 * ե졼쥸ž⡼ɤꤹ
 */
STATIC void setConfigReg(
	const int host,			// ATAۥֹ
	const int dev,			// ATAǥХֹ
	const PCI_INFO pci,
	const int modePos)
{
	u_int32_t value = read_pci_config(pci, 0x54);

	value |= 1 << modePos;
	writew_pci_config(pci, 0x54, value);
}

/*
 * ե졼쥸ž⡼ɤ
 */
STATIC void resetConfigReg(
	const int host,			// ԣۥֹ
	const int dev,			// ԣǥХֹ
	const PCI_INFO pci,
	const int modePos)
{
	u_int32_t value = read_pci_config(pci, 0x54);

	value &= ~((u_int32_t) 1 << modePos);
	writew_pci_config(pci, 0x54, value);
}

/*
 * 󥯥ʥģͣ쥸ꤹ
 */
STATIC void setSynchronousReg(
	const int host,			// ATAۥֹ
	const int dev,			// ATAǥХֹ
	const PCI_INFO pci,
	const u_int32_t timingBit)
{
	int synchroCntrPos = host * 2 + dev;
	int synchroTimingPos = 16 + (host * 2 + dev) * 4;
	u_int32_t value = read_pci_config(pci, 0x48);

	value |= (u_int32_t) 1 << synchroCntrPos;
	value &= ~((u_int32_t) 3 << synchroTimingPos);
	value |= timingBit << synchroTimingPos;
	writedw_pci_config(pci, 0x48, value);
}

/*
 * 󥯥ʥģͣ쥸
 */
STATIC void resetSynchronousReg(
	const int host,			// ATAۥֹ
	const int dev,			// ATAǥХֹ
	const PCI_INFO pci)
{
	int synchroCntrPos = host * 2 + dev;
	u_int32_t value = read_pci_config(pci, 0x48);

	value &= ~((u_int32_t) 1 << synchroCntrPos);
	writedw_pci_config(pci, 0x48, value);
}

/*
 * ATA100ž⡼ɤꤹ
 */
STATIC INLINE void setAta100(
	const int host,			// ATAۥֹ
	const int dev,			// ATAǥХֹ
	const PCI_INFO pci)
{
	int ata100Pos = 12 + host * 2 + dev;

	setConfigReg(host, dev, pci, ata100Pos);
	setSynchronousReg(host, dev, pci, 1);
}

/*
 * ATA100ž⡼ɤ
 */
STATIC INLINE void resetAta100(
	const int host,			// ԣۥֹ
	const int dev,			// ԣǥХֹ
	const PCI_INFO pci)
{
	int ata100Pos = 12 + host * 2 + dev;

	resetConfigReg(host, dev, pci, ata100Pos);
}

/*
 * ATA66ž⡼ɤꤹ
 */
STATIC INLINE void setAta66(
	const int host,			// ATAۥֹ
	const int dev,			// ATAǥХֹ
	const PCI_INFO pci)
{
	int ata66Pos = host * 2 + dev;

	setConfigReg(host, dev, pci, ata66Pos);
	setSynchronousReg(host, dev, pci, 2);
}

/*
 * ATA66ž⡼ɤ
 */
STATIC INLINE void resetAta66(
	const int host,			// ԣۥֹ
	const int dev,			// ԣǥХֹ
	const PCI_INFO pci)
{
	int ata66Pos = host * 2 + dev;

	resetConfigReg(host, dev, pci, ata66Pos);
}

/*
 * ATA33ž⡼ɤꤹ
 */
STATIC INLINE void setAta33(
	const int host,			// ATAۥֹ
	const int dev,			// ATAǥХֹ
	const PCI_INFO pci)
{
	setSynchronousReg(host, dev, pci, 2);
}

/*
 * ATA33ž⡼ɤ
 */
STATIC INLINE void resetAta33(
	const int host,			// ԣۥֹ
	const int dev,			// ԣǥХֹ
	const PCI_INFO pci)
{
	resetSynchronousReg(host, dev, pci);
}

/*
 * IDEߥ󥰤ꤹ
 */
STATIC void setIdeTiming(
	const int host,			// ATAۥֹ
	const int dev,			// ATAǥХֹ
	const PCI_INFO pci)
{
	uint32_t value;
	uint32_t mask40, new40, mask44, new44;

	// ߥ󥰥쥸ޥȥ졼֤ʬꤹ
	value = read_pci_config(pci, 0x40);
	if ((value & (0x4000 << host * 16)) == 0) {
		// ߥͤ򥹥졼ѥ쥸˥ԡ
		new40 = value;
		new40 |= 0x4000 << 16 * host;
		writedw_pci_config(pci, 0x40, new40);

		new44 = read_pci_config(pci, 0x44);
		new44 &= ~((uint32_t) 0xf << 4 * host);
		new44 |= (new40 & (0x300 << 16 * host)) >> (8 + 12 * host);
		new44 |= (new40 & (0x3000 << 16 * host)) >> (10 + 12 * host);
		writedw_pci_config(pci, 0x44, new44);
	}

	// ߥ󥰥ӥåȤ
	if (dev == 0) {
		value = read_pci_config(pci, 0x40);
		mask40 = 0x330f << 16 * host;
		new40 = 0x2307 << 16 * host;
		writedw_pci_config(pci, 0x40, (value & ~mask40) | new40);
	}
	else {
		value = read_pci_config(pci, 0x44);
		mask44 = 0xf << 4 * host;
		new44 = 0xb << 4 * host;
		writedw_pci_config(pci, 0x44, (value & ~mask44) | new44);
	}
}

//================================== PUBLIC =============================================

/*
 * return : error number -ENODEVʤ鳺åץåȤ̵ä
 */
int initDmaIntel(
	const int host,			// ԣۥֹ
	const int dev,			// ԣǥХֹ
	const PCI_INFO pci,
	const int mode,			// ⡼
	int *o_mode)			// ѥ⡼
{
	switch (pci.vidDid) {
	case 0x24db8086:	/* Intel ICH5 */
	case 0x24d18086:	/* Intel ICH5 SATA */
	case 0x24ca8086:	/* Intel ICH4 mobile */
	case 0x24cb8086:	/* Intel ICH4 */
	case 0x248a8086:	/* Intel ICH3 mobile */ 
	case 0x248b8086:	/* Intel ICH3 */
	case 0x244a8086:	/* Intel ICH2 mobile */ 
	case 0x244b8086:	/* Intel ICH2 */
		if (UDMA5 <= mode) {
			if (isSet80Cable(host, dev, pci) == YES) {
				*o_mode = UDMA5;
				setAta100(host, dev, pci);
				break;
			}
		}

		/* make sure eventual ATA100 mode from the BIOS is disabled */
		resetAta100(host, dev, pci);

		/* FALLTHROUGH */
	case 0x24118086:	/* Intel ICH */
	case 0x76018086:	/* Intel ICH */
		if (UDMA4 <= mode) {
			if (isSet80Cable(host, dev, pci) == YES) {
				*o_mode = UDMA4;
				setAta66(host, dev, pci);
				break;
			}
		}		

		/* make sure eventual ATA66 mode from the BIOS is disabled */
		resetAta66(host, dev, pci);

		/* FALLTHROUGH */
	case 0x71118086:	/* Intel PIIX4 */
	case 0x84CA8086:	/* Intel PIIX4 */
	case 0x71998086:	/* Intel PIIX4e */
	case 0x24218086:	/* Intel ICH0 */
		if (UDMA2 <= mode) {
			*o_mode = UDMA2;
			setAta33(host, dev, pci);
			break;
		}

		/* make sure eventual ATA33 mode from the BIOS is disabled */
		resetAta33(host, dev, pci);

		/* FALLTHROUGH */
	case 0x70108086:	/* Intel PIIX3 */
		if (WDMA2 <= mode) {
			*o_mode = WDMA2;
			setIdeTiming(host, dev, pci);
			break;
		}

		return -EDINVAL;
	default:
		return -ENODEV;
	}
	
	return NOERR;
}
