/*
 * pci.c (C) 2002 Minoru Murashima
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * PCI
 */


#include <sys/param.h>
#include <sys/types.h>
#include <lib/lib.h>
#include <kern/errno.h>
#include <dev/pci/pci.h>


//================================== PRIVATE ============================================

enum{
	/* IO port */
	CONFIG_ADDR = 0xcf8,
	CONFIG_DATA = 0xcfc,
};

//================================== PUBLIC =============================================

/*
 * ꡼IOѥɥ쥹Ƥ롣
 * ˳ƤƤ륢ɥ쥹
 * IO APIC:		0xfec000000xfec00040
 * LOCAL APIC:	0xfee000000xfee00400,
 * return : memory address
 */
uint getPciMemAddress(
	int size)			// ׵᥵
{
	// ꡼IOѥɥ쥹
	static uint pciMem = 0xfef00000;

	int addr;


	/* 0x100000Ȥ˥꡼Ƥ롣 */
	addr = pciMem;
	pciMem += ROUNDUP(size, 0x100000);

	return addr;
}

/**************************************************************************
 *
 * PCI CONFIGURATION
 *
 **************************************************************************/

/*
 * Read pci config register
 * parameters : PCI_INFO,Register
 * return : Read value
 */
uint read_pci_config(
	const PCI_INFO pci, 
	const int reg)
{
	uint value;

	outl(CONFIG_ADDR, (1 << 31) | (pci.bus << 16) | (pci.dev << 11) | (pci.func << 8) | (reg & ~3));
	value = inl(CONFIG_DATA);
	outl(CONFIG_ADDR, 0);

	return value >> ((reg & 3) * 8);
}


/*
 * Write double word value to pci config register
 */
void writedw_pci_config(
	const PCI_INFO pci,
	const int reg,
	const uint value)
{
	outl(CONFIG_ADDR, (1 << 31) | (pci.bus << 16) | (pci.dev << 11) | (pci.func << 8) | (reg & ~3));
	outl(CONFIG_DATA + (reg & 3), value);
	outl(CONFIG_ADDR, 0);
}

/*
 * Write word value to pci config register
 */
void writew_pci_config(
	const PCI_INFO pci,
	const int reg,
	const ushort value)
{
	outl(CONFIG_ADDR, (1 << 31) | (pci.bus << 16) | (pci.dev << 11) | (pci.func << 8) | (reg & ~3));
	outw(CONFIG_DATA + (reg & 3), value);
	outl(CONFIG_ADDR, 0);
}

/*
 * Write byte value to pci config register
 */
void writeb_pci_config(
	const PCI_INFO pci,
	const int reg,
	const uchar value)
{
	outl(CONFIG_ADDR, (1 << 31) | (pci.bus << 16) | (pci.dev << 11) | (pci.func << 8) | (reg & ~3));
	outb(CONFIG_DATA + (reg & 3), value);
	outl(CONFIG_ADDR, 0);
}

/*
 * Search PCI device from class coad.
 * Ϳ줿PCI饹ͤ鸡Ϥ롣
 * return : error number
 */
int search_pci_class(
	const int class, 
	const PCI_INFO i_pci,
	PCI_INFO *m_pci)
{
	PCI_INFO pci = i_pci;

	for(;pci.bus < 255; ++pci.bus){
		for(;pci.dev < 32; ++pci.dev){
			for(;pci.func < 8; ++pci.func){
				if ((ushort)read_pci_config(pci, PCI_CONF_VENDER) == 0xffff) {
					continue;
				}
				if(read_pci_config(pci, PCI_CONF_CLASS) == class) {
					pci.vidDid = read_pci_config(pci, PCI_CONF_VENDER);
					*m_pci = pci;

					return NOERR;
				}
			}
			pci.func = 0;
		}
		pci.dev = 0;
	}

	return -ENODEV;
}
/*
// VenderID  DeviceIDõС
int search_pci_class(
	uint VenderID, 
	uint DeviceID, 
	const PCI_INFO inf)
{
	uint vidDid = (VenderID & 0xffff) | (DeviceID << 16);

	for (;inf.bus < 255; ++inf.bus){
		for (;inf.dev < 32; ++inf.dev){
			for (;inf.func < 8; ++inf.func){
				if ((ushort)read_pci_config(inf,PCI_CONF_VENDER) == 0xffff){
					continue;
				}
				if (read_pci_config(inf, PCI_CONF_VENDER) == vidDid){
					inf.vidDid = vidDid;
					inf.class = read_pci_config(inf, PCI_CONF_CLASS);
					return 0;
				}
			}
			inf.func = 0;
		}
		inf.dev = 0;
	}

	return -ENODEV;
}
*/

/***************************************************************************************************
 *
 * PCI IRQ Routing Table
 *
 ***************************************************************************************************/

typedef struct{
	uchar  bus;							/* PCI Bus Number. */
	uchar  dev;							/* PCI Device Number (in upper five bits). */
	struct{
		uchar  value;					/* Link Value. */
		ushort bitmap;					/* IRQ Bitmap. */
	} __attribute__((packed)) irq[4];
	uchar  number;						/* Slot Number. */
	char   reserve;						/* Reserved. */
}SLOT_ENTRY;

typedef struct{
	uint       sign;		/* Signature. */
	ushort     version;		/* Version. */
	ushort     tblsize;		/* Table Size. */
	uchar      bus;			/* PCI Interrupt Router's Bus. */
	uchar      devfunc;		/* PCI Interrupt Router's DevFunc. */
	ushort     excirq;		/* PCI Exclusive IRQs. */
	uint       router;		/* Compatible PCI Interrupt Router. */
	uint       mport;		/* Miniport Data. */
	char       reserve[11];	/* Reserved (Zero). */
	uchar      checksum;	/* Checksum. */
	SLOT_ENTRY entry[0];	/* Slot Entry. */
}PCI_IRQ_ROUT_TABLE;

/* PRIVATE */
static PCI_IRQ_ROUT_TABLE *pciIrqTbl;

/*
 * PRIVATE
 * Search PCI IRQ Routing Table.
 * return : address or failed=NULL
 */
PCI_IRQ_ROUT_TABLE *searchPciIrqTable()
{
	enum{
		BEGIN_ADDR = 0xf0000,
		LAST_ADDR = 0xfffff,
		SIG1 = '$',
		SIG2 = 'P',
		SIG3 = 'I',
		SIG4 = 'R',
	};

	char *p;

	for (p = (char*)BEGIN_ADDR; p < (char*)(LAST_ADDR - 3);){
		if ((*p++ == SIG1) && (*p == SIG2) && (*++p == SIG3) && (*++p == SIG4)){
			return (PCI_IRQ_ROUT_TABLE*)(p - 3);
		}
	}

	return NULL;
}

/*
 * PUBLIC
 * Init PCI IRQ Routing Table.
 * return : 0
 */
static int initPciIrqTable()
{
	pciIrqTbl = searchPciIrqTable();

	return 0;
}

/*
 * Get PCI IRQ number.
 * return : PCI IRQ number or failed=-1
 */
int getPciIrq(
	const PCI_INFO pci)
{
	int pin;
	int i,last;
	SLOT_ENTRY *entry;

	if(pciIrqTbl==NULL)return -1;

	pin = read_pci_config(pci, PCI_CONF_INTRPIN) & 0xff;
	if (pin == 0){
		return -1;
	}
	last = (pciIrqTbl->tblsize - sizeof(PCI_IRQ_ROUT_TABLE)) / sizeof(SLOT_ENTRY);
	entry = pciIrqTbl->entry;
	for (i = 0; i < last; ++i){
		if ((entry[i].bus == pci.bus) && (entry[i].dev >> 3 == pci.dev)){
			return entry[i].irq[pin - 1].value & 0x3;
		}
	}

	return -1;
}

/**************************************************************************
 *
 * 
 *
 **************************************************************************/

/*
 * Init pci.
 * return : 0
 */
int initPci()
{
	initPciIrqTable();

	return 0;
}

/*************************************************************************************/
void test_pci()
{
}
