/*-
 * Copyright (c) 1997 Michael Smith
 * Copyright (c) 1998 Jonathan Lemon
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/i386/i386/bios.c,v 1.29.2.4 2003/08/09 16:21:17 luoqi Exp $
 */

/*
 * Code for dealing with the BIOS in x86 PC systems.
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <machine/bios.h>
#include <lib/lib.h>

#include <kern/debug.h>


//================================== PRIVATE ============================================


#define BIOS_START	0xe0000
#define BIOS_SIZE	0x20000


struct bios32_SDentry PCIbios;
static u_int bios32_SDCI;


// bios32_SDlookup
//
// Query the BIOS32 Service Directory for the service named in (ent),
// returns nonzero if the lookup fails.  The caller must fill in
// (ent->ident), the remainder are populated on a successful lookup.
// return : 0 or 1 = error
static int bios32_SDlookup(struct bios32_SDentry *ent)
{
	struct bios_regs args;

	if (bios32_SDCI == 0){
		return (1);
	}

	args.eax = ent->ident.id;		/* set up arguments */
	args.ebx = args.ecx = args.edx = 0;
	bios32(&args, bios32_SDCI);
	if ((args.eax & 0xff) == 0) {	/* success? */
		ent->base = args.ebx;
		ent->len = args.ecx;
		ent->entry = args.edx;
		ent->ventry = (ent->base + ent->entry);
		return (0);			/* all OK */
	}
	return (1);				/* failed */
}


//================================== PUBLIC =============================================


// bios_sigsearch
//
// Search some or all of the BIOS region for a signature string.
//
// (start)	Optional offset returned from this function 
//		(for searching for multiple matches), or NULL
//		to start the search from the base of the BIOS.
//		Note that this will be a _physical_ address in
//		the range 0xe0000 - 0xfffff.
// (sig)	is a pointer to the byte(s) of the signature.
// (siglen)	number of bytes in the signature.
// (paralen)	signature paragraph (alignment) size.
// (sigofs)	offset of the signature within the paragraph.
//
// Returns the _physical_ address of the found signature, 0 if the
// signature was not found.
// return : address or 0 = error
u_int32_t bios_sigsearch(u_int32_t start, u_char *sig, int siglen, int paralen, int sigofs)
{
	u_char	*sp, *end;

	/* compute the starting address */
	if ((start >= BIOS_START) && (start <= (BIOS_START + BIOS_SIZE))) {
		sp = (u_char *)start;
	}
	else if (start == 0) {
		sp = (u_char *)BIOS_START;
	}
	else {
		return 0;	/* bogus start address */
	}

	/* compute the end address */
	end = (u_char *)BIOS_START + BIOS_SIZE;

	/* loop searching */
	while ((sp + sigofs + siglen) < end) {
		/* compare here */
		if (!strncmp(sp + sigofs, sig, siglen)) {
			/* convert back to physical address */
			return((u_int32_t)sp);
		}
		sp += paralen;
	}

	return(0);
}


/*
 * bios32_init
 *
 * Locate various bios32 entities.
 */
void bios32_init()
{
	u_long					sigaddr;
	struct bios32_SDheader	*sdh;
	u_int8_t				ck, *cv;
	int						i;

	/* look for the signature */
	if ((sigaddr = bios_sigsearch(0, "_32_", 4, 16, 0)) != 0) {
		/* get a virtual pointer to the structure */
		sdh = (struct bios32_SDheader *)(uintptr_t)(sigaddr);
		for (cv = (u_int8_t *)sdh, ck = 0, i = 0; i < (sdh->len * 16); i++) {
			ck += cv[i];
		}
		/* If checksum is OK, enable use of the entrypoint */
		if ((ck == 0) && (sdh->entry < (BIOS_START + BIOS_SIZE))) {
			bios32_SDCI = (sdh->entry);
			if (bootverbose) {
				printf("bios32: Found BIOS32 Service Directory header at %p\n", sdh);
				printf("bios32: Entry = 0x%x (%x)  Rev = %d  Len = %d\n", sdh->entry, bios32_SDCI, sdh->revision, sdh->len);
			}
			/* See if there's a PCI BIOS entrypoint here */
			PCIbios.ident.id = 0x49435024;	/* PCI systems should have this */
			if (!bios32_SDlookup(&PCIbios) && bootverbose){
				printf("pcibios: PCI BIOS entry at 0x%x\n", PCIbios.entry);
			}
		}
		else {
			printf("bios32: Bad BIOS32 Service Directory\n");
		}
	}
}
