/*
 * debug.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 *ԳסCPUݡȥǥХåǽ
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <lib/bitmap.h>
#include <i386/interrupt.h>
#include <i386/Entry.h>
#include <i386/debug.h>

#include <kern/debug.h>


//#define DEBUG_I386_DEBUG 1
#ifdef DEBUG_I386_DEBUG
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


//==================================  ===========================================

//================================== Х륤ݡ ===============================

//================================== PRIVATE ============================================

enum {
	// ֥졼ݥȥ֥͡ե饰
	DEBUG_LOCAL		= 1 << 0,		// åȤ˽
	DEBUG_GLOBAL	= 1 << 1,

	NUMBERS_OF_BREAKPOINT = 4,		// ֥졼ݥǽ
};

/*
 * ǥХå쥸READ
 * return : 쥸
 */
STATIC uint readDr0()
{
	uint reg;

	asm volatile(
		"movl	%%dr0, %%eax"
		:"=a"(reg):
	);

	return reg;
}

STATIC uint readDr1()
{
	uint reg;

	asm volatile(
		"movl	%%dr1, %%eax"
		:"=a"(reg):
	);

	return reg;
}

STATIC uint readDr2()
{
	uint reg;

	asm volatile(
		"movl	%%dr2, %%eax"
		:"=a"(reg):
	);

	return reg;
}

STATIC uint readDr3()
{
	uint reg;

	asm volatile(
		"movl	%%dr3, %%eax"
		:"=a"(reg):
	);

	return reg;
}

STATIC uint readDr6()
{
	uint reg;

	asm volatile(
		"movl	%%dr6, %%eax"
		:"=a"(reg):
	);

	return reg;
}

STATIC uint readDr7()
{
	uint reg;

	asm volatile(
		"movl	%%dr7, %%eax"
		:"=a"(reg):
	);

	return reg;
}

/*
 * ǥХå쥸WRITE
 */
STATIC void writeDr0(
	const uint i_value)
{
	asm volatile(
		"movl	%%eax, %%dr0"
		::"a"(i_value)
	);
}

STATIC void writeDr1(
	const uint i_value)
{
	asm volatile(
		"movl	%%eax, %%dr1"
		::"a"(i_value)
	);
}

STATIC void writeDr2(
	const uint i_value)
{
	asm volatile(
		"movl	%%eax, %%dr2"
		::"a"(i_value)
	);
}

STATIC void writeDr3(
	const uint i_value)
{
	asm volatile(
		"movl	%%eax, %%dr3"
		::"a"(i_value)
	);
}

STATIC INLINE void writeDr6(
	const uint i_value)
{
	enum {
		DR6_RESERVE_BIT = 0xffff0ff0,	// DR6ͽӥå
	};
	uint value = i_value;

	value |= DR6_RESERVE_BIT;
	asm volatile(
		"movl	%%eax, %%dr6"
		::"a"(value)
	);
}

STATIC INLINE void writeDr7(
	const uint i_value)
{
	enum {
		DR7_RESERVE_BIT = 0x400,		// DR7ͽӥå
	};
	uint value = i_value;

	value |= DR7_RESERVE_BIT;
	asm volatile(
		"movl	%%eax, %%dr7"
		::"a"(value)
	);
}

static uint (*readbreakAddr[NUMBERS_OF_BREAKPOINT])() = {
	readDr0,
	readDr1,
	readDr2,
	readDr3
};

static void (*writebreakAddr[NUMBERS_OF_BREAKPOINT])(const uint) = {
	writeDr0,
	writeDr1,
	writeDr2,
	writeDr3
};

/*
 * ̤ѤΥǥХå쥸ֹ򸡺
 * return : TURUE or FALSE
 */
STATIC int getUnusedRegNumber(
	int *o_number)
{
	uint dr7 = readDr7();
	int number;
	
	for (number = 0; number < NUMBERS_OF_BREAKPOINT; ++number) {
		if ((dr7 & (0x3 << (number * 2))) == 0) {
			*o_number = number;
			return TRUE;
		}
	}
	
	return FALSE;
}

/*
 * ե饰ꤹ
 *ճ֥졼ݥֹե饰ϥꥢƤ뤳
 */
STATIC void setCondition(
	const int number,		// ֥졼ݥֹ
	const int condition)	// ɤߡ񤭡ɥ쥹Ĺե饰
{
	uint dr7;

	ASSERT((0 <= number) && (number < NUMBERS_OF_BREAKPOINT));
	
	dr7 = readDr7();
	dr7 |= condition << (number * 4);
	dr7 |= DEBUG_GLOBAL << (number * 2);
	writeDr7(dr7);
}

/*
 * ե饰򥯥ꥢ
 */
STATIC void clearCondition(
	const int number)		// ֥졼ݥֹ
{
	uint dr7;

	ASSERT((0 <= number) && (number < NUMBERS_OF_BREAKPOINT));

	dr7 = 0x3 << (number * 2);		// 롦Х륤֥͡ե饰
	dr7 |= 0xf000 << (number * 4);	// ɤߡ񤭡ɥ쥹Ĺե饰
	dr7 = readDr7() & ~dr7;

	writeDr7(dr7);
}

/*
 * 㳰򵯤֥졼ݥֹ
 * return : ֥졼ݥֹ
 */
STATIC INLINE int getBreakNumber()
{
	uint dr6 = readDr6();
	
	return getBitmapPos(0, 4, (const u_int8_t*) &dr6);
}

/*
 * ơե饰򥯥ꥢ
 */
STATIC void clearState()
{
	writeDr6(0);
}

//================================== PUBLIC =============================================

/*
 * ǥХå
 * return : NOERR or ERR
 */
int setDebug(
	const uint breakAddr,		// ֥졼ݥȥ˥ɥ쥹
	const int condition)		// ֥졼
{
	int number;
	int eflags;
	int error;

	eflags = enterCli();
	{
		// ֥졼ݥֹõ
		if (getUnusedRegNumber(&number) == FALSE) {
			error = ERR;
			goto EXIT;
		}

		// ֥졼˥ɥ쥹
		writebreakAddr[number](breakAddr);

		// ֥졼ݥֹꤹ
		clearCondition(number);
		setCondition(number, condition);
		
		// ơե饰򥯥ꥢ
		clearState();
		
		error = NOERR;
	}
EXIT:
	exitCli(eflags);

	return error;
}

/*
 * ǥХå
 */
void resetDebug(
	uint breakAddr)		// ֥졼ݥȥ˥ɥ쥹
{
	int i;
	
	for (i = 0; i < NUMBERS_OF_BREAKPOINT; ++i) {
		if (readbreakAddr[i]() == breakAddr) {
			// ֥졼ݥե饰򥯥ꥢ
			clearCondition(i);
		}
	}
}

/*
 * ǥХåϥɥ
 *ճԲĤǸƤӽФ롣
 */
void debugHandler(
	TaskContext frame)
{
	int number = getBreakNumber();
	uint breakAddr = readbreakAddr[number]();

	// eip㳰򵯤̿μ̿ᥢɥ쥹
	printk("\nBreak point trap! : break point=0x%x write value=0x%x eip=0x%x\n",
		breakAddr,
		*(uint*) breakAddr,
		frame.eip);

	// ǥХåȯȤ˥ꥢɬפ
	clearState();
}

/*
 * ֥졼ݥȥϥɥ
 *ճԲĤǸƤӽФ롣
 */
void excBreakpoint()
{
	printk("Breakpoint!\n");
}

//*******************************************************************
void test_debug(
	const uint breakAddr)
{
	static uint lastBreakAddr = 0;

	resetDebug(lastBreakAddr);

	if (setDebug(breakAddr, DEBUG_WRITE | DEBUG_4BYTE) == ERR) {
		printk("setDebug error!\n");
		return;
	}
	
	lastBreakAddr = breakAddr;
}
