/*
 * ProcSignal.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/signal.h>
#include <lib/lib.h>
#include <kern/errno.h>
#include <kern/Wait.h>
#include <kern/Thread.h>
#include <kern/ProcSignal.h>

#include <kern/debug.h>


//#define DEBUG_SIGNAL 1
#ifdef DEBUG_SIGNAL
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif


//=====================================  ===================================================

/*
 * ʥ빽¤
 *ԱƶեСѹProcSignalObjΥѹ뤳
 */
typedef struct {
	SIG_ACTION	sigAction[NSIG];
	void		*returnMethod;		// 桼ʥϥɥ餫᥽å
} ProcSignal;

struct DefaultHandler {
	SA_HANDLER handler[NSIG];
};

//===================================== Х륤ݡ =======================================

extern ProcSignalObj *procGetProcSignal(const void*);
extern void *getCurrentProc();

//===================================== PRIVATE ====================================================

//--------------------------------------------------------------------------------------------------
// Getter
//--------------------------------------------------------------------------------------------------

STATIC INLINE SIG_HANDLER getHandler(
	ProcSignal *this,
	const int sigNum)
{
	return (SIG_HANDLER) this->sigAction[sigNum].sa_handler;
}

STATIC INLINE sigset_t *getMask(
	ProcSignal *this,
	const int sigNum)
{
	return &this->sigAction[sigNum].sa_mask;
}

STATIC INLINE void *getReturnMethod(
	ProcSignal *this)
{
	return this->returnMethod;
}

//--------------------------------------------------------------------------------------------------
// Setter
//--------------------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------------------
// Yes or No
//--------------------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------------------
// Search
//--------------------------------------------------------------------------------------------------

/*
 * Signal default actino. Terminate process.
 * return : task switch YES or NO
 */
STATIC int terminate(int signum)
{
	exit(0, signum);
	return exitSchedule();
}

/*
 * Signal default actino. Abnormal terminate process.
 * return : task switch YES or NO
 */
STATIC int abnormalTerminate(int signum)
{
	exit(0, signum);
	return exitSchedule();
}

/*
 * Signal default actino. Stop process.
 * return : task switch YES or NO
 */
STATIC int stopProcess(int signum)
{
	procSetExitSignum(getCurrentProc(), signum);

	return stopSchedule();
}

static struct DefaultHandler defaultHandler = {
	.handler[0]			= NULL,
	.handler[SIGABRT]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGALRM]	= (SA_HANDLER) terminate,
	.handler[SIGBUS]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGCHLD]	= SIG_IGN,
	.handler[SIGCONT]	= SIG_DFL,
	.handler[SIGFPE]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGHUP]	= (SA_HANDLER) terminate,
	.handler[SIGILL]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGINT]	= (SA_HANDLER) terminate,
	.handler[SIGKILL]	= (SA_HANDLER) terminate,
	.handler[SIGPIPE]	= (SA_HANDLER) terminate,
	.handler[SIGPOLL]	= (SA_HANDLER) terminate,
	.handler[SIGPROF]	= (SA_HANDLER) terminate,
	.handler[SIGQUIT]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGSEGV]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGSTOP]	= (SA_HANDLER) stopProcess,
	.handler[SIGSYS]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGTERM]	= (SA_HANDLER) terminate,
	.handler[SIGTRAP]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGTSTP]	= (SA_HANDLER) stopProcess,
	.handler[SIGTTIN]	= (SA_HANDLER) stopProcess,
	.handler[SIGTTOU]	= (SA_HANDLER) stopProcess,
	.handler[SIGURG]	= SIG_IGN,
	.handler[SIGUSR1]	= (SA_HANDLER) terminate,
	.handler[SIGUSR2]	= (SA_HANDLER) terminate,
	.handler[SIGVTALRM]	= (SA_HANDLER) terminate,
	.handler[SIGXCPU]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGXFSZ]	= (SA_HANDLER) abnormalTerminate,
	.handler[SIGWINCH]	= SIG_IGN,
	.handler[SIGINFO]	= SIG_IGN,
	.handler[SIGWAKE]	= SIG_IGN,
};

//================================== PUBLIC =============================================

int ProcSignalInit()
{
#ifdef DEBUG
	if (sizeof(ProcSignal) != sizeof(ProcSignalObj)) {
		printk("ProcSignalInit() object size error! ProcSignal=%d ProcSignalObj=%d\n", sizeof(ProcSignal), sizeof(ProcSignalObj));
		idle();
	}
#endif

	return NOERR;
}

void ProcSignalConstructor(
	ProcSignalObj *i_this)
{
	ProcSignal *this = (ProcSignal*) i_this;
	int i;

	memset(this, 0, sizeof(*this));

	// ǥեȥϥɥ򥳥ԡ
	for (i = 0; i < NSIG; ++i) {
		this->sigAction[i].sa_handler = (SA_HANDLER) defaultHandler.handler[i];
	}
}

void ProcSignalCopyConstructor(
	ProcSignalObj *i_this,
	ProcSignalObj *i_from)
{
	memcpy(i_this, i_from, sizeof(ProcSignal));
}

void ProcSignalDestructor(
	ProcSignalObj *i_this)
{
	// ⤷ʤ
}

//--------------------------------------------------------------------------------------------------
// Getter
//--------------------------------------------------------------------------------------------------

SIG_HANDLER ProcSignalGetHandler(
	ProcSignalObj *i_this,
	const int i_sigNum)
{
	ProcSignal *this = (ProcSignal*) i_this;
	return getHandler(this, i_sigNum);
}

sigset_t *ProcSignalGetHandlerMask(
	ProcSignalObj *i_this,
	const int i_sigNum)
{
	ProcSignal *this = (ProcSignal*) i_this;
	return getMask(this, i_sigNum);
}

void *ProcSignalGetReturnMethod(
	ProcSignalObj *i_this)
{
	ProcSignal *this = (ProcSignal*) i_this;
	return getReturnMethod(this);
}

//--------------------------------------------------------------------------------------------------
// Setter
//--------------------------------------------------------------------------------------------------

/*
 *ոƤӽФϼץΤ
 */
void ProcSignalSetMask(
	const int i_sigNum)
{
	void *proc = getCurrentProc();
	sigset_t *mask = procGetSignalMask(proc);
	setSignal(mask, i_sigNum);
	procSetSignalMask(proc, mask);
}

//--------------------------------------------------------------------------------------------------
// Yes or No
//--------------------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------------------
// Search
//--------------------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------------------
// ƥॳ
//--------------------------------------------------------------------------------------------------

int sys_sigaction(
	int signum, 
	const struct sigaction *act, 
	struct sigaction *oldact, 
	void *returnMethod)
{
	ProcSignal *procSignal = (ProcSignal*) procGetProcSignal(getCurrentProc());

	if (signum == 0){
		return 0;
	}
	if ((NSIG < signum) || (signum < 0)){
		return -EINVAL;
	}

	if (act != NULL){
		if (checkMem(act, sizeof(*act)) == ERR) {
			return -EFAULT;
		}

		if ((signum == SIGKILL) || (signum == SIGSTOP)){
			return -EINVAL;		// SIGKILLSIGSTOPѹԲ
		}

		if (act->sa_handler == SIG_DFL){
			procSignal->sigAction[signum].sa_handler = defaultHandler.handler[signum];
		}
		else if (act->sa_handler == SIG_IGN){
			procSignal->sigAction[signum].sa_handler = SIG_IGN;
		}
		else if (USER_BEG <= (uint)act->sa_handler) {
			int error;

			procSignal->sigAction[signum].sa_handler = act->sa_handler;

			// 桼᥽åɥݥ󥿤
			procSignal->returnMethod = returnMethod;

			// ʥϥɥѲۥ꡼
			error = procCreatUserHandlerStack(getCurrentProc());
			if (error != NOERR) {
				return error;
			}
		}
		else {
			return -EINVAL;		/* ϥɥ饢ɥ쥹 */
		}

		procSignal->sigAction[signum].sa_mask = act->sa_mask;
		if ((act->sa_flags & SA_NODEFER) == 0) {
			setSignal(&procSignal->sigAction[signum].sa_mask, signum);
		}

		procSignal->sigAction[signum].sa_flags = act->sa_flags;
	}

	if (oldact != NULL) {
		if (checkMem(oldact, sizeof(*oldact)) == ERR) {
			return -EFAULT;
		}

		if (procSignal->sigAction[signum].sa_handler == SIG_IGN) {
			oldact->sa_handler = SIG_IGN;
		}
		else if (procSignal->sigAction[signum].sa_handler == defaultHandler.handler[signum]) {
			oldact->sa_handler = SIG_DFL;
		}
		else {
			oldact->sa_handler = (SA_HANDLER) procSignal->sigAction[signum].sa_handler;
		}

		oldact->sa_mask = procSignal->sigAction[signum].sa_mask;
	}

	return 0;
}

int sys_sigprocmask(
	int how, 
	sigset_t *set,
	sigset_t *oldset)
{
	sigset_t *mask;

	if (checkMem(set, sizeof(*set)) == ERR) {
		return -EFAULT;
	}

	mask = procGetSignalMask(getCurrentProc());
	if (oldset != NULL) {
		if (checkMem(oldset, sizeof(*oldset)) == ERR) {
			return -EFAULT;
		}
		*oldset = *mask;
	}

	switch (how) {
	case SIG_BLOCK:
		// SIGKILLSIGSTOPϥ֥åԲ
		if ((isMaskSignal(set, SIGKILL) == YES) || (isMaskSignal(set, SIGSTOP) == YES)) {
			return -EINVAL;
		}
		addMask(mask, set);
		break;
	case SIG_UNBLOCK:
		delMask(mask, set);
		break;
	case SIG_SETMASK:
		// SIGKILLSIGSTOPϥ֥åԲ
		if ((isMaskSignal(set, SIGKILL) == YES) || (isMaskSignal(set, SIGSTOP) == YES)) {
			return -EINVAL;
		}
		*mask = *set;
		break;
	}

	return 0;
}

int sys_kill(
	pid_t pid, 
	int sigNum)
{
	if (NSIG <= (uint) sigNum) {
		return -EPERM;
	}

	if (0 < pid) {
		void *proc = procGetProcPid(pid);
		if (proc == NULL) {
			return -ESRCH;
		}
		return procSendSignal(proc, sigNum);
	}
	else if (pid == 0) {		/* callץƱ롼ס */
		procSendSignalPgid(getPgid(getCurrentProc()), sigNum);
	}
	else if (pid == -1) {		/* ץ */
		sendSignalAllProc(sigNum);
	}
	else {						/* pid<-1 -pidΥ롼ס */
		procSendSignalPgid(-pid, sigNum);
	}

	return 0;
}

int sys_sigpending(
	sigset_t *set,
	int flag)
{
	/* ݥ󥿤å */
	if (checkMem(set, sizeof(*set)) == ERR) {
		return -EINVAL;
	}

	switch(flag) {
	case SIG_PENDING_SOTORE:
		*set = *procGetSignalPending(getCurrentProc());
		break;
	case SIG_PENDING_SET:
		procSetSignalPending(getCurrentProc(), set);
		break;
	}

	return 0;
}

int sys_sigsuspend(
	sigset_t *i_mask)
{
	void *proc;
	sigset_t *beforeMask;

	if (checkMem(i_mask, sizeof(*i_mask)) == ERR) {
		return -EFAULT;
	}

	// SIGKILLSIGSTOPϥ֥åԲ
	if ((isMaskSignal(i_mask, SIGKILL) == YES) || (isMaskSignal(i_mask, SIGSTOP) == YES)) {
		return -EINVAL;
	}

	proc = getCurrentProc();
	beforeMask = procGetSignalMask(proc);
	procSetSignalMask(proc, i_mask);

	// suspend
	procSendSignal(proc, SIGSTOP);
	waitTask();

	procSetSignalMask(proc, beforeMask);

	return -EINTR;
}
