/*
 * block_cache.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ֥åǥХå
 *
 *ա
 *  ǥХϥ512ХȤˤƤΤǡ¾Υ
 *  ǥХоݤˤˤϡɬס
 *ԸƤ
 * ̥ǥХʣƱߤб롣
 * 	ٱߥåɤʣưФǤ롩
 * 	iovWrite򥹥åɤȤ˻롣
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/uio.h>
#include <netinet/in.h>
#include <kern/vm.h>
#include <kern/kmalloc.h>
#include <kern/lock.h>
#include <kern/Thread.h>
#include <kern/device.h>
#include <kern/devfs.h>
#include <lib/lib.h>
#include <kern/Wait.h>
#include <kern/listFunc.h>

#include <kern/debug.h>


//#define DEBUG_BLOCK_CACHE 1
#ifdef DEBUG_BLOCK_CACHE
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE inline
#endif


//==================================  ===========================================

/*
 * ֥åå幽¤
 */
typedef struct BLOCK_CACHE {
	struct specinfo		*dev;			// ǥХ
	uint				sector;			// ǥХֹCACHE_SECTORS
	char				*cache;			// å
	u_int8_t			bitmap;			// åӥåȥޥåסӥåȡᣱ
	u_int8_t			dirtyBitmap;	// ӥåȥޥåסӥåȡᣱ

	OBJ_LIST			seqList;		// ǥХinodeֹ桦ֹ縡å۴ĥꥹ

	OBJ_LIST			updateList;		// ǿå۴ĥꥹ

	OBJ_LIST			writeList;		// ᤷԤ۴ĥꥹ
	OBJ_LIST			writeSeqList;	// ǥХinodeֹ桦ֹᤷԤ۴ĥꥹ
	int 				writeBusy;		// ᤷե饰
	struct BLOCK_CACHE	*cacheNext;		// ᤷԤꥹȥ
	u_int8_t			writeBitmap;	// ᤷӥåȥޥåסӥåȡᣱ
} BLOCK_CACHE;

enum {
	CACHE_NUM_MAX = 20,					// IOå֥åѹġ
	CACHE_SIZE = PAGE_SIZE,				// IOå֥å
	CACHE_SECTOR_SIZE = 512,			// 
	CACHE_SECTORS = CACHE_SIZE / CACHE_SECTOR_SIZE,	// åΥ
};

// ֥åå奤󥹥
static BLOCK_CACHE blkCacheInstance[CACHE_NUM_MAX];

/***************************************************************************************************
 * ӥåȥޥå
 ***************************************************************************************************/

typedef struct {
	uchar sectorNum;	// åֹ
	uchar sectorCnt;	// å⥻
	uchar sectorHigh;	// åǾֹ̥
} CACHE_SECTOR;

//================================== PRIVATE ============================================

static const CACHE_SECTOR bitmapSeq[] = {
	{0,0,0},{0,1,1},{1,1,2},{0,2,2},{2,1,3},{0,1,3},{1,2,3},{0,3,3},{3,1,4},{0,1,4},{1,1,4},{0,2,4},{2,2,4},{0,1,4},{1,3,4},{0,4,4},
	{4,1,5},{0,1,5},{1,1,5},{0,2,5},{2,1,5},{0,1,5},{1,2,5},{0,3,5},{3,2,5},{0,1,5},{1,1,5},{0,2,5},{2,3,5},{0,1,5},{1,4,5},{0,5,5},
	{5,1,6},{0,1,6},{1,1,6},{0,2,6},{2,1,6},{0,1,6},{1,2,6},{0,3,6},{3,1,6},{0,1,6},{1,1,6},{0,2,6},{2,2,6},{0,1,6},{1,3,6},{0,4,6},
	{4,2,6},{0,1,6},{1,1,6},{0,2,6},{2,1,6},{0,1,6},{1,2,6},{0,3,6},{3,3,6},{0,1,6},{1,1,6},{0,2,6},{2,4,6},{0,1,6},{1,5,6},{0,6,6},
	{6,1,7},{0,1,7},{1,1,7},{0,2,7},{2,1,7},{0,1,7},{1,2,7},{0,3,7},{3,1,7},{0,1,7},{1,1,7},{0,2,7},{2,2,7},{0,1,7},{1,3,7},{0,4,7},
	{4,1,7},{0,1,7},{1,1,7},{0,2,7},{2,1,7},{0,1,7},{1,2,7},{0,3,7},{3,2,7},{0,1,7},{1,1,7},{0,2,7},{2,3,7},{0,1,7},{1,4,7},{0,5,7},
	{5,2,7},{0,1,7},{1,1,7},{0,2,7},{2,1,7},{0,1,7},{1,2,7},{0,3,7},{3,1,7},{0,1,7},{1,1,7},{0,2,7},{2,2,7},{0,1,7},{1,3,7},{0,4,7},
	{4,3,7},{0,1,7},{1,1,7},{0,2,7},{2,1,7},{0,1,7},{1,2,7},{0,3,7},{3,4,7},{0,1,7},{1,1,7},{0,2,7},{2,5,7},{0,1,7},{1,6,7},{0,7,7},
	{7,1,8},{0,1,8},{1,1,8},{0,2,8},{2,1,8},{0,1,8},{1,2,8},{0,3,8},{3,1,8},{0,1,8},{1,1,8},{0,2,8},{2,2,8},{0,1,8},{1,3,8},{0,4,8},
	{4,1,8},{0,1,8},{1,1,8},{0,2,8},{2,1,8},{0,1,8},{1,2,8},{0,3,8},{3,2,8},{0,1,8},{1,1,8},{0,2,8},{2,3,8},{0,1,8},{1,4,8},{0,5,8},
	{5,1,8},{0,1,8},{1,1,8},{0,2,8},{2,1,8},{0,1,8},{1,2,8},{0,3,8},{3,1,8},{0,1,8},{1,1,8},{0,2,8},{2,2,8},{0,1,8},{1,3,8},{0,4,8},
	{4,2,8},{0,1,8},{1,1,8},{0,2,8},{2,1,8},{0,1,8},{1,2,8},{0,3,8},{3,3,8},{0,1,8},{1,1,8},{0,2,8},{2,4,8},{0,1,8},{1,5,8},{0,6,8},
	{6,2,8},{0,1,8},{1,1,8},{0,2,8},{2,1,8},{0,1,8},{1,2,8},{0,3,8},{3,1,8},{0,1,8},{1,1,8},{0,2,8},{2,2,8},{0,1,8},{1,3,8},{0,4,8},
	{4,1,8},{0,1,8},{1,1,8},{0,2,8},{2,1,8},{0,1,8},{1,2,8},{0,3,8},{3,2,8},{0,1,8},{1,1,8},{0,2,8},{2,3,8},{0,1,8},{1,4,8},{0,5,8},
	{5,3,8},{0,1,8},{1,1,8},{0,2,8},{2,1,8},{0,1,8},{1,2,8},{0,3,8},{3,1,8},{0,1,8},{1,1,8},{0,2,8},{2,2,8},{0,1,8},{1,3,8},{0,4,8},
	{4,4,8},{0,1,8},{1,1,8},{0,2,8},{2,1,8},{0,1,8},{1,2,8},{0,3,8},{3,5,8},{0,1,8},{1,1,8},{0,2,8},{2,6,8},{0,1,8},{1,7,8},{0,8,8}
};

static const u_int8_t cacheBitmap[8][9] = {
	{0, 0x1,  0x3,  0x7,  0xf,  0x1f, 0x3f, 0x7f, 0xff},
	{0, 0x2,  0x6,  0xe,  0x1e, 0x3e, 0x7e, 0xfe, 0xfe},
	{0, 0x4,  0xc,  0x1c, 0x3c, 0x7c, 0xfc, 0xfc, 0xfc},
	{0, 0x8,  0x18, 0x38, 0x78, 0xf8, 0xf8, 0xf8, 0xf8},
	{0, 0x10, 0x30, 0x70, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0},
	{0, 0x20, 0x60, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0},
	{0, 0x40, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0},
	{0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80}
};

//================================== PROTECT ============================================

/*
 * ¸ߤƤ륭å⥻
 *ահֲ̤ΥӥåȤ̤³ƤӥåȰ֤ȥӥåȿ֤
 * return : CACHE_SECTOR
 */
STATIC INLINE CACHE_SECTOR getExistCacheSector(
	const u_int8_t bitmap)
{
	return bitmapSeq[bitmap];
}

/*
 * ¸ߤƤ륭å⥻
 * return : ӥåȥޥå
 */
STATIC INLINE u_int8_t getBitmap(
	const int sectorNum,	// åХֹ
	const int sectorCnt)	// å⥻
{
	return cacheBitmap[sectorNum][sectorCnt];
}

/***************************************************************************************************
 * UIO
 ***************************************************************************************************/

//================================== PROTECT ============================================

STATIC INLINE void setUioParam(
	const dev_t dev,
	const int iovecCnt,		// struct iovec
	enum uio_rw uioMode,	// UIO_READ or UIO_WRITE
	const uint topSector,	// ǥХƬֹ
	struct iovec *iovec,
	struct uio *m_uio)		// ꤹstruct uio
{
	setUio(
		iovecCnt, 
		uioMode, 
		(topSector + dev->beginSector) * dev->sectSize,
		0,
		iovec,
		m_uio);
}

/**************************************************************************************************
 * ǥɤ߹
 ***************************************************************************************************/

//================================== PRIVATE ============================================

// ǥХIOiovec
static struct iovec *iovRead;

//================================== PROTECT ============================================

/*
 * 
 * return : error number
 */
STATIC int initReadDevice()
{
	/*
	 * ǥХIOiovecγ
	 * iovecϥǥХuiouio_iovľܻȤΤǡ
	 * 4KХȶ֤롣
	 */
	iovRead = kmalloc(PAGE_SIZE);
	if (iovRead == NULL) {
		return -ENOMEM;
	}
	
	return NOERR;
}

/*
 * ǥХ饭åɤ߹ࡣ
 * return : error number
 */
STATIC int readDevice(
	const int sectorNum,		// åХֹ
	const int sectorCnt,		// ɤ߹ߥ
	BLOCK_CACHE *m_blkCache)
{
	dev_t dev = m_blkCache->dev;
	u_int8_t readBitmap = getBitmap(sectorNum, sectorCnt);
	CACHE_SECTOR cacheSector;

	for (readBitmap &= ~m_blkCache->bitmap; readBitmap != 0; readBitmap &= ~getBitmap(cacheSector.sectorNum, cacheSector.sectorCnt)) {
		struct uio uio;
		int error;

		cacheSector = getExistCacheSector(readBitmap);
		iovRead[0].iov_base = m_blkCache->cache + cacheSector.sectorNum * CACHE_SECTOR_SIZE;
		iovRead[0].iov_len = cacheSector.sectorCnt * CACHE_SECTOR_SIZE;

		// ǥХIO
		setUioParam(dev, 1, UIO_READ, m_blkCache->sector + cacheSector.sectorNum, iovRead, &uio);
		error = dev->si_devsw->d_read(dev, &uio, 0);
		if (error != NOERR) {
			return error;
		}
	}

	return NOERR;
}

/**************************************************************************************************
 * ǥᤷ
 ***************************************************************************************************/

//================================== PRIVATE ============================================

enum {
	IOVEC_MAX = PAGE_SIZE / sizeof(struct iovec),	// iovec
};

static int writeLock;			// ٱ񤭹Ԥå
static OBJ_LIST *writeHead;		// ֥ååٱ񤭹Ԥꥹȥإå
static OBJ_LIST *writeSeqHead;	// ֥ååٱ񤭹ԤǥХinodeֹ桦ֹꥹȥإå
static void *writeTask;			// ٱ񤭹ѥ
static struct iovec *iovWrite;	// ǥХIOiovec

/*
 *Թ
 * ӥåȥޥåפᤷӥåȥޥåפ˲ä
 */
STATIC INLINE void setWriteBitmap(
	const u_int8_t upBitmap,	// ӥåȥޥå
	BLOCK_CACHE *m_blkCache)
{
	m_blkCache->writeBitmap |= upBitmap;
}

/*
 *Թ
 * ӥåȥޥåפᤷӥåȥޥåפ˲ä
 */
STATIC INLINE void resetWriteBitmap(
	BLOCK_CACHE *m_blkCache)
{
	m_blkCache->writeBitmap = 0;
}

/*
 * å夫ǥХ˽᤹
 * return : error number
 */
STATIC int writeDevice(
	BLOCK_CACHE *i_blkCache)	// IOƬ֥åå
{
	dev_t dev = i_blkCache->dev;
	struct uio uio;
	BLOCK_CACHE *blkCache;
	uint topSector;				// Ƭ
	int c_iovec;

	/*
	 * å󥯤iovWriteꤷIO
	 *աեå󥯤ϥϢ³Ƥ뤳ȤˤƤ
	 */
	c_iovec = 0;
	for (blkCache = i_blkCache; blkCache != NULL; blkCache = blkCache->cacheNext) {
		uchar bitmap;
		CACHE_SECTOR cacheSector;

		for (bitmap = blkCache->writeBitmap; bitmap != 0; bitmap &= ~getBitmap(cacheSector.sectorNum, cacheSector.sectorCnt)) {
			cacheSector = getExistCacheSector(bitmap);
			iovWrite[c_iovec].iov_base = blkCache->cache + cacheSector.sectorNum * CACHE_SECTOR_SIZE;
			iovWrite[c_iovec].iov_len = cacheSector.sectorCnt * CACHE_SECTOR_SIZE;
			++c_iovec;

			if (1 == c_iovec) {
				topSector = blkCache->sector + cacheSector.sectorNum;
			}

			if ((IOVEC_MAX <= c_iovec) || (cacheSector.sectorNum + cacheSector.sectorCnt < CACHE_SECTORS)){
				int error;

				// ǥХIO
				setUioParam(dev, c_iovec, UIO_WRITE, topSector, iovWrite, &uio);
				error = dev->si_devsw->d_write(dev, &uio, 0);
				if (error != NOERR) {
					return error;
				}
				c_iovec = 0;
			}
		}
	}

	if (0 < c_iovec) {
		setUioParam(dev, c_iovec, UIO_WRITE, topSector, iovWrite, &uio);
		return dev->si_devsw->d_write(dev, &uio, 0);
	}
	else {
		return NOERR;
	}
}

/*
 * ٱ񤭹Ԥե饰򥻥åȤ
 */
STATIC INLINE void setWriteBusy(
	BLOCK_CACHE *m_blkCache)
{
	m_blkCache->writeBusy = 1;
}

/*
 * ٱ񤭹Ԥե饰ꥻåȤ
 */
STATIC INLINE void resetWriteBusy(
	BLOCK_CACHE *m_blkCache)
{
	m_blkCache->writeBusy = 0;
}

/*
 *աwriteLockǥåƤ椫ƤФ뤳
 * ᤷԤ۴ĥꥹȥ椫
 * return : YES or NO
 */
STATIC INLINE int isWriteListLink(
	BLOCK_CACHE *blkCache)
{
	if (refNextList(OFFSETOF(BLOCK_CACHE, writeList), &blkCache->writeList) != blkCache) {
		return YES;
	}
	else if (refHeadObj(OFFSETOF(BLOCK_CACHE, writeList), writeHead) == blkCache) {
		return YES;
	}
	return NO;
}

/*
 *Թ
 * ҤäƤ륭å󥯤
 * return : Ҥä : TRUE, Ҥʤ : FALSE
 */
STATIC INLINE int linkCache(
	BLOCK_CACHE *m_dstCache,
	BLOCK_CACHE *m_srcCache)		// ³å
{
	CACHE_SECTOR cacheDst;
	CACHE_SECTOR cacheSrc;

	cacheDst = getExistCacheSector(m_dstCache->writeBitmap);
    cacheSrc = getExistCacheSector(m_srcCache->writeBitmap);

	ASSERT(0 < cacheDst.sectorCnt);

	if (m_dstCache->sector + cacheDst.sectorHigh == m_srcCache->sector + cacheSrc.sectorNum) {
		m_dstCache->cacheNext = m_srcCache;

		return TRUE;
	}
	else {
		return FALSE;
	}
}

/*
 *Թ
 *աwriteLockǥåƤ椫ƤФ뤳
 * ٱ񤭹ԤǥХinodeֹ桦ֹꥹȤ˲ä
 */
STATIC void addWriteSeq(
	BLOCK_CACHE *m_blkCache)
{
	BLOCK_CACHE *head = refHeadObj(OFFSETOF(BLOCK_CACHE, writeSeqList), writeSeqHead);
	
	if (head == NULL) {
		insertHeadCrclList(&writeSeqHead, &m_blkCache->writeSeqList);
	}
	else if (m_blkCache->dev < head->dev) {
		insertHeadCrclList(&writeSeqHead, &m_blkCache->writeSeqList);
	}
	else if ((m_blkCache->dev == head->dev) && (m_blkCache->sector < head->sector)) {
		insertHeadCrclList(&writeSeqHead, &m_blkCache->writeSeqList);
	}
	else {
		BLOCK_CACHE *blkCache;

		for (blkCache = refNextList(OFFSETOF(BLOCK_CACHE, writeSeqList), &head->writeSeqList);
			blkCache != head;
			blkCache = refNextList(OFFSETOF(BLOCK_CACHE, writeSeqList), &blkCache->writeSeqList)) {
			
			ASSERT((m_blkCache->dev != blkCache->dev) || (m_blkCache->sector != blkCache->sector));
			
			if (m_blkCache->dev < blkCache->dev) {
				break;
			}
			else if ((m_blkCache->dev == head->dev) && (m_blkCache->sector < blkCache->sector)) {
				break;
			}
		}
		insertPrevList(&blkCache->writeSeqList, &m_blkCache->writeSeqList);
	}
}

/*
 *Թ
 * ᤷԤ۴ĥꥹȤκǸΥåȥҤäƤ륭å󥯤ƼФ
 * return : ᤷԤå or NULL
 */
STATIC BLOCK_CACHE *getWriteWait()
{
	BLOCK_CACHE *linkTop;

	enter_spinlock(&writeLock);
	{
		BLOCK_CACHE *writeCache = getHeadPrevList(OFFSETOF(BLOCK_CACHE, writeList), &writeHead);
		BLOCK_CACHE *linkEnd;
		BLOCK_CACHE *limit;
		BLOCK_CACHE *blkCache;
		BLOCK_CACHE *next;
		BLOCK_CACHE *prev;

		if (writeCache == NULL) {
			exit_spinlock(&writeLock);
			return NULL;
		}

		linkTop = writeCache;
		limit = refHeadObj(OFFSETOF(BLOCK_CACHE, writeSeqList), writeSeqHead);

		ASSERT(limit != NULL);

		for (blkCache = refPrevList(OFFSETOF(BLOCK_CACHE, writeSeqList), &writeCache->writeSeqList);; blkCache = prev) {
			prev = refPrevList(OFFSETOF(BLOCK_CACHE, writeSeqList), &blkCache->writeSeqList);
			
			if ((blkCache->dev == linkTop->dev) && (linkCache(blkCache, linkTop) == TRUE)) {
				removeOutHeadList(&writeSeqHead, &blkCache->writeSeqList);
				removeOutHeadList(&writeHead, &blkCache->writeList);
				linkTop = blkCache;
			}
			else {
				break;
			}

			if (blkCache == limit) {
				break;
			}
		}

		linkEnd = writeCache;
		limit = refHeadPrevList(OFFSETOF(BLOCK_CACHE, writeSeqList), writeSeqHead);

		ASSERT(limit != NULL);

		for (blkCache = refNextList(OFFSETOF(BLOCK_CACHE, writeSeqList), &writeCache->writeSeqList);; blkCache = next) {
			next = refNextList(OFFSETOF(BLOCK_CACHE, writeSeqList), &blkCache->writeSeqList);

			if ((blkCache->dev == linkEnd->dev) && (linkCache(linkEnd, blkCache) == TRUE)) {
				removeOutHeadList(&writeSeqHead, &blkCache->writeSeqList);
				removeOutHeadList(&writeHead, &blkCache->writeList);
				linkEnd = blkCache;
			}
			else {
				break;
			}
			
			if (blkCache == limit) {
				break;
			}
		}
		
		removeOutHeadList(&writeSeqHead, &writeCache->writeSeqList);
	}
	exit_spinlock(&writeLock);

	return linkTop;
}

/*
 *Թ
 * ٱ񤭹߽λ
 */
STATIC void endWriteCache(
	BLOCK_CACHE *m_blkCache)
{
	BLOCK_CACHE *blkCache;
	BLOCK_CACHE *next;

	for (blkCache = m_blkCache; blkCache != NULL; blkCache = next) {
		next = blkCache->cacheNext;
		blkCache->cacheNext = NULL;
		resetWriteBitmap(blkCache);
		resetWriteBusy(blkCache);	// ӥե饰¾ܤꥻåȤ˥ꥻå
	}
}

/*
 * ǥХؤٱ񤭹ߥ
 */
STATIC void delayWriteTask()
{
	// 롼
	for (;;) {
		// 񤭹ԤФ
		BLOCK_CACHE *blkCache = getWriteWait();

		if (blkCache != NULL){
			if (writeDevice(blkCache) != NOERR) {
				printk("Device error in delay write function!\n");
			}
			endWriteCache(blkCache);
		}
		else{
			waitTask();
		}
	}
}

/*
 * ٱ񤭹ߥεư
 */
STATIC void startDelayWriteTask()
{
	switch (forkKernelThread()) {
		case -1:
			printk("Faild startDelayWriteTask\n");
			break;
		case 0:
			writeTask = getCurrentTask();

			// λ륷ʥ򤹤٤ƥޥ
			setSigAllMask(TaskGetTaskSignal(writeTask));

			delayWriteTask();

			// äƤʤ
	}
}

//================================== PROTECT ============================================

/*
 * 
 * return : error number
 */
STATIC int initWriteDevice()
{
	int i;

	for (i = 0; i < CACHE_NUM_MAX; ++i){
		initCirculList(&blkCacheInstance[i].writeList);
		initCirculList(&blkCacheInstance[i].writeSeqList);
		blkCacheInstance[i].cacheNext = NULL;
		blkCacheInstance[i].writeBusy = 0;
	}

	/*
	 * ǥХIOiovecγ
	 * iovecϥǥХuiouio_iovľܻȤΤǡ
	 * 4KХȶ֤롣
	 */
	iovWrite = kmalloc(PAGE_SIZE);
	if (iovWrite == NULL) {
		return -ENOMEM;
	}
	
	// ٱ񤭹ѥεư
	startDelayWriteTask();

	return NOERR;
}

/*
 * ٱ񤭹Ԥ
 * return : YES or NO
 */
STATIC INLINE int isWriteBusy(
	const BLOCK_CACHE *blkCache)
{
	return (blkCache->writeBusy == 1) ? YES : NO;
}

/*
 * ٱ񤭹Ԥ
 */
STATIC INLINE void waitWriteBusy(
	BLOCK_CACHE *blkCache)
{
	if (isWriteBusy(blkCache) == YES) {
		// ٱ񤭹ߥ򵯤
		activeTask(getWaitTask(writeTask));

		waitFlag(1, &blkCache->writeBusy);
	}
}

/*
 *Թ
 * ٱ񤭹ԤꥹȤ˲ä
 */
STATIC void addWriteWait(
	const u_int8_t upBitmap,	// ӥåȥޥå
	BLOCK_CACHE *m_blkCache)
{
	enter_spinlock(&writeLock);
	{
		if (isWriteListLink(m_blkCache) == YES) {
			removeOutHeadList(&writeHead, &m_blkCache->writeList);
		}
		else {
			waitWriteBusy(m_blkCache);

			addWriteSeq(m_blkCache);
			setWriteBusy(m_blkCache);
		}

		// ᤷӥåȥޥåפΥå
		setWriteBitmap(upBitmap, m_blkCache);

		insertHeadCrclList(&writeHead, &m_blkCache->writeList);
	}
	exit_spinlock(&writeLock);

	// ٱ񤭹ߥ򵯤
	activeTask(getWaitTask(writeTask));
}

/*
 *Թ
 * å˥ǡ򥳥ԡ
 * DMAžǤΥǥᤷΥå˽񤭹ߤȥǥᤷԶ礬ȯΤ
 * ǥᤷΥå˽񤭹ߤʤ褦椹롣
 */
STATIC void synchroCopyCache(
	const uint offset,
	const size_t size,
	char *writeBuf,
	BLOCK_CACHE *m_blkCache)
{
	if (isWriteBusy(m_blkCache) == YES) {
		enter_spinlock(&writeLock);
		{
			if (isWriteListLink(m_blkCache) == NO) {
				waitWriteBusy(m_blkCache);
			}
			memcpy(m_blkCache->cache + offset, writeBuf, size);
		}
		exit_spinlock(&writeLock);
	}
	else {
		memcpy(m_blkCache->cache + offset, writeBuf, size);
	}
}

/*
 *Ի
 * ᤷԤΥå夬ĤäƤ뤫
 * return : YES or NO
 */
STATIC int isExistWaitCache()
{
	return (refHeadObj(OFFSETOF(BLOCK_CACHE, writeList), writeHead) != NULL) ? YES : NO;
}

/**************************************************************************************************
 * Ȼ
 ***************************************************************************************************/

//================================== PRIVATE ============================================

static OBJ_LIST *updateHead;	// ֥åå幹ꥹȥإå

//================================== PROTECT ============================================

/*
 *Թ
 * ǿå
 */
STATIC void initUpdate()
{
	int i;

	for (i = 0; i < CACHE_NUM_MAX; ++i){
		initCirculList(&blkCacheInstance[i].updateList);

		// ǿ󥯤ˤĤʤ
		insertHeadCrclList(&updateHead, &blkCacheInstance[i].updateList);
	}
}

/*
 *Թ
 * ǿåˤ
 */
STATIC void updateCache(
	BLOCK_CACHE *m_blkCache)
{
	// ǿ˹
	removeOutHeadList(&updateHead, &m_blkCache->updateList);
	insertHeadCrclList(&updateHead, &m_blkCache->updateList);
}

/*
 *Ի
 * ǿ饭å򻲾Ȥ
 */
STATIC BLOCK_CACHE *refNewBelowCache(
	const int order)		// ǿν֡ǿ
{
	BLOCK_CACHE *new;
	int i;
	
	new = refHeadObj(OFFSETOF(BLOCK_CACHE, updateList), updateHead);
	for (i = 1; i < order; ++i) {
		new = refNextList(OFFSETOF(BLOCK_CACHE, updateList), &new->updateList);
	}
	
	return new;
}

/*
 *Ի
 * ָŤå򻲾Ȥ
 */
STATIC INLINE BLOCK_CACHE *refOldCache()
{
	return refHeadPrevList(OFFSETOF(BLOCK_CACHE, updateList), updateHead);
}

/**************************************************************************************************
 * 
 ***************************************************************************************************/

//================================== PRIVATE ============================================

static OBJ_LIST *seqHead;		// ֥åå帡ꥹȥإå

/*
 *Ի
 * ༡̵а˼Υ֥åå֤
 * return : ֥åå or NULL
 */
STATIC BLOCK_CACHE *searchCache(
	dev_t dev,			// ǥХinode
	const uint sector,		// ֹ
	BLOCK_CACHE **o_next)	// Υ֥åå
{
	BLOCK_CACHE *head = refHeadObj(OFFSETOF(BLOCK_CACHE, seqList), seqHead);
	BLOCK_CACHE *find;
	BLOCK_CACHE *next;

	if (head == NULL) {
		*o_next = NULL;
		return NULL;
	}

	/*
	 * 󥯤򻲾ȤƥǥХinodeȥֹ椬פå
	 */
	find = NULL;	// 줿å
	*o_next = NULL;
	next = head;
	do {
		if (dev < next->dev) {
			*o_next = next;
			break;
		}
		
		if (next->dev == dev) {
			if (sector < next->sector) {
				*o_next = next;
				break;
			}
			else if ((next->sector <= sector) && (sector < next->sector + CACHE_SECTORS)) {
				find = next;
				break;
			}
		}
		
		next = refNextList(OFFSETOF(BLOCK_CACHE, seqList), &next->seqList);
	} while (next != head);
	
	return find;
}

//================================== PROTECT ============================================

/*
 * 
 */
STATIC void initFind()
{
	int i;

	for (i = 0; i < CACHE_NUM_MAX; ++i){
		initCirculList(&blkCacheInstance[i].seqList);
	}
}

/*
 *Ի
 * 
 * return : ֥åå or NULL
 */
STATIC INLINE BLOCK_CACHE *findCache(
	const dev_t dev,		// ǥХinode
	const uint sector)		// ֹ
{
	BLOCK_CACHE *next;

	return searchCache(dev, sector, &next);
}

/*
 *Թ
 * å򸡺󥯤ˤĤʤ
 * 󥯤ϥǥХinode硦֥å˥Ȥ褦ˤĤʤ
 */
STATIC void addFindCache(
	BLOCK_CACHE *m_blkCache)
{
	BLOCK_CACHE *next;

	// Ƥʤä˸ƤФΤǸĤϤϤʤ
	ASSERT(searchCache(m_blkCache->dev, m_blkCache->sector, &next) == NULL);

	if (next == NULL) {
		BLOCK_CACHE *prev = refHeadPrevList(OFFSETOF(BLOCK_CACHE, seqList), seqHead);
		
		if (prev == NULL) {				// 󥯤˲ʤ
			insertHeadCrclList(&seqHead, &m_blkCache->seqList);
		}
		else {
			insertNextList(&prev->seqList, &m_blkCache->seqList);
		}
	}
	else {
		BLOCK_CACHE *head = refHeadObj(OFFSETOF(BLOCK_CACHE, seqList), seqHead);

		if (next == head) {
			insertHeadCrclList(&seqHead, &m_blkCache->seqList);
		}
		else {
			insertPrevList(&next->seqList, &m_blkCache->seqList);
		}
	}
}

/*
 *Թ
 * å򸡺󥯤
 */
STATIC void removeFindCache(
	BLOCK_CACHE *m_blkCache)
{
	removeOutHeadList(&seqHead, &m_blkCache->seqList);
}

/**************************************************************************************************
 * å
 ***************************************************************************************************/

enum {
	WRITE_BACK_ORDER = 4,	// ǥ˽᤹åκǿν
};

static int cacheLock;		// ֥ååå

/*
 * ֥ååν
 */
STATIC void initCache(
	BLOCK_CACHE *m_blkCache)
{
	m_blkCache->dev = NULL;
	m_blkCache->sector = 0;
	m_blkCache->bitmap = 0;
	m_blkCache->dirtyBitmap = 0;
}

/*
 * ֥åå
 * return : error number
 */
STATIC int constructBlockCache()
{
	int i;

	// ֥åå奤󥹥󥹤ν
	for (i = 0; i < CACHE_NUM_MAX; ++i){
		initCache(&blkCacheInstance[i]);

		blkCacheInstance[i].cache = kmalloc(CACHE_SIZE);
		if (blkCacheInstance[i].cache == NULL){
			return -ENOMEM;
		}
	}
	
	return NOERR;
}

/*
 *Թ
 * å⥻򥭥åӥåȥޥåפɲä
 */
STATIC INLINE void addCacheSector(
	const int sectorNum,	// åХֹ
	const int sectorCnt,	// å⥻
	BLOCK_CACHE *m_blkCache)
{
	ASSERT(sectorNum < 8);
	ASSERT(sectorCnt <= 8);

	m_blkCache->bitmap |= getBitmap(sectorNum, sectorCnt);
}

/*
 *Թ
 * å⥻򹹿ӥåȥޥåפɲä
 */
STATIC INLINE void setDirtySector(
	const int sectorNum,	// åХֹ
	const int sectorCnt,	// å⥻
	BLOCK_CACHE *m_blkCache)
{
	m_blkCache->dirtyBitmap |= getBitmap(sectorNum, sectorCnt);
}

/*
 *Թ
 * ӥåȥޥåפꥻåȤ
 */
STATIC INLINE void resetDirtyBitmap(
	BLOCK_CACHE *m_blkCache)
{
	m_blkCache->dirtyBitmap = 0;
}

/*
 *Ի
 * ӥåȥޥåפåȤƤ뤫
 * return : YES or NO
 */
STATIC INLINE int isDirtyCache(
	const BLOCK_CACHE *blkCache)
{
	return (blkCache->dirtyBitmap != 0) ? YES : NO;
}

/*
 *Թ
 * åǥ˽᤹
 */
STATIC void writeBackCache(
	BLOCK_CACHE *m_blkCache)
{
	// ٱԤ󥯤³
	addWriteWait(m_blkCache->dirtyBitmap, m_blkCache);

	resetDirtyBitmap(m_blkCache);
}

/*
 *Թ
 * ˥å
 * return : å
 */
STATIC BLOCK_CACHE *getCache(
	const dev_t dev,		// ǥХinode
	const uint i_sector)	// ֹ
{
	const uint sector = ROUNDDOWN(i_sector, CACHE_SECTORS);
	BLOCK_CACHE *blkCache;
	
	// ָŤå򻲾Ȥ
	blkCache = refOldCache();

	// ָŤåϽᤷƤϤ
	ASSERT((isWriteBusy(blkCache) == YES) || (isDirtyCache(blkCache) == NO));
	
	// ߽λԤ
	waitWriteBusy(blkCache);
	
	// ֥ååν
	initCache(blkCache);
	blkCache->dev = dev;
	blkCache->sector = sector;

	// 󥯤
	removeFindCache(blkCache);

	// 󥯤³
	addFindCache(blkCache);

	return blkCache;
}

//================================== PUBLIC =============================================

/*
 * ǥХå奷ƥ
 * return : error number
 */
int initBlockCache()
{
	int error;

	error = constructBlockCache();
	if (error != NOERR) {
		return error;
	}
	
	// ν
	initFind();
	
	// ǿȴν
	initUpdate();
	
	// ǥɤ߹ߤν
	error = initReadDevice();
	if (error != NOERR) {
		return error;
	}

	// ǥᤷν
	error = initWriteDevice();
	if (error != NOERR) {
		return error;
	}

	return NOERR;
}

/*
 *աճߥϥɥ餫θƤӽФԲ
 *ԸƤreadCache()ʣåɤ߹ߤб롣
 * return : error number
 */
int read_cache(
	void *i_dev,				// ǥХ
	void *i_readBuf,			// readХåե
	const size_t i_sectors,		// read
	const size_t i_begin)		// readϥ
{
	const dev_t dev = i_dev;
	char *readBuf = i_readBuf;
	uint sector;
	uint sectorLast;
	uint sectorBound;

	// ǥ¤Υå
	if (dev->sectors < i_begin + i_sectors){
		return -EIO;
	}

	/*
	 * åPAGE_SIZEñ̤IO
	 * sector : åñ̤ɤ߹߳ϥֹ
	 * sectorLast : 饹IOֹ
	 * sectorBound : PAGE_SIZEΥֹ
	 * unreadCnt : ǥХ̤ɤ߹ߥå
	 * unreadCache : ǥХ̤ɤ߹ߥåƬ
	 */
	sectorLast = i_begin + i_sectors;
	sectorBound = ROUNDDOWN(i_begin, CACHE_SECTORS);
	for (sector = i_begin; sector < sectorLast;) {
		BLOCK_CACHE *blkCache;	// IOå
		uint sectorNext;		// PAGE_SIZEΥֹ
		uint sectorCnt;			// Хåեԡ
		int error;

		enter_spinlock(&cacheLock);
		{
			BLOCK_CACHE *backCache;

			sectorNext = ROUNDDOWN(sector + CACHE_SECTORS, CACHE_SECTORS);

			// ɣϥ׻
			if (sectorLast < sectorNext) {
				sectorCnt = sectorLast - sector;
			}
			else {
				sectorCnt = sectorNext - sector;
			}

			// å򸡺
			blkCache = findCache(dev, sector);
			if (blkCache == NULL) {
				// åˤʤп˳Ƥ
				blkCache = getCache(dev, sector);
			}

			// ǥɤ߹ޤƤʤɤ߹
			error = readDevice(sector - sectorBound, sectorCnt, blkCache);
			if (error != NOERR) {
				// read顼ϥǥץ졼Τ
				printk("readCache() error! : %d\n", error);
			}
			addCacheSector(sector - sectorBound, sectorCnt, blkCache);

			// å夫饳ԡ
			memcpy(readBuf, blkCache->cache + (sector - sectorBound) * CACHE_SECTOR_SIZE, sectorCnt * CACHE_SECTOR_SIZE);

			// ǿå˹
			updateCache(blkCache);

			// ٱᤷ
			backCache = refNewBelowCache(WRITE_BACK_ORDER);
			if (isDirtyCache(backCache) == YES) {
				writeBackCache(backCache);
			}
		}
		exit_spinlock(&cacheLock);

		readBuf += sectorCnt * CACHE_SECTOR_SIZE;
		sector = sectorNext;
		sectorBound = sectorNext;
	}

	return NOERR;
}

/*
 *ա ߥϥɥ餫θƤӽФԲ
 * return : error number
 */
int write_cache(
	void *i_dev,				// ǥХ
	void *i_writeBuf,			// writeХåե
	const size_t i_sectors, 	// write
	const size_t i_begin)		// writeϥ
{
	const dev_t dev = i_dev;
	char *writeBuf = i_writeBuf;
	uint sector;
	uint sectorLast;
	uint sectorBound;

	// ǥ¤Υå
	if (dev->sectors < i_begin + i_sectors){
		return -EIO;
	}

	/*
	 * åPAGE_SIZEñ̤IO
	 * sector : åñ̤ɤ߹߳ϥֹ
	 * sectorLast : 饹IOֹ
	 * sectorBound : PAGE_SIZEΥֹ
	 */
	sectorLast = i_begin + i_sectors;
	sectorBound = ROUNDDOWN(i_begin, CACHE_SECTORS);
	for (sector = i_begin; sector < sectorLast;) {
		BLOCK_CACHE *blkCache;	// IOå
		uint sectorNext;		// PAGE_SIZEΥֹ
		uint sectorCnt;			// Хåեԡ

		enter_spinlock(&cacheLock);
		{
			BLOCK_CACHE *backCache;

			sectorNext = ROUNDDOWN(sector + CACHE_SECTORS, CACHE_SECTORS);

			// ɣϥ׻
			if (sectorLast < sectorNext) {
				sectorCnt = sectorLast - sector;
			}
			else {
				sectorCnt = sectorNext - sector;
			}

			// å򸡺
			blkCache = findCache(dev, sector);
			if (blkCache == NULL) {
				// åˤʤп˳Ƥ
				blkCache = getCache(dev, sector);
			}

			// å˥ԡ
			synchroCopyCache((sector - sectorBound) * CACHE_SECTOR_SIZE, sectorCnt * CACHE_SECTOR_SIZE, writeBuf, blkCache);

			setDirtySector(sector - sectorBound, sectorCnt, blkCache);
			addCacheSector(sector - sectorBound, sectorCnt, blkCache);

			// ǿå˹
			updateCache(blkCache);

			// ٱᤷ
			backCache = refNewBelowCache(WRITE_BACK_ORDER);
			if (isDirtyCache(backCache) == YES) {
				writeBackCache(backCache);
			}
		}
		exit_spinlock(&cacheLock);

		writeBuf += sectorCnt * CACHE_SECTOR_SIZE;
		sector = sectorNext;
		sectorBound = sectorNext;
	}

	return NOERR;
}

/*
 * ߤꥭå֥å򤹤٤ٱߤ롣
 */
void writeBackAllCache()
{
	enter_spinlock(&cacheLock);
	{
		int i;
		
		for (i = 0; i < CACHE_NUM_MAX; ++i) {
			if ((isWriteBusy(&blkCacheInstance[i]) == NO) && (isDirtyCache(&blkCacheInstance[i]) == YES)) {
				writeBackCache(&blkCacheInstance[i]);
			}
		}
		
		while (isExistWaitCache() == YES) {
			wait_task();
		}
	}
	exit_spinlock(&cacheLock);
}

/*
 * å夫ǥХ˥ǡ᤹
 */
int sys_sync()
{
	writeBackAllCache();

	return 0;
}

#ifdef DEBUG
int testBlock(
	void *i_task)
{
	return (i_task == writeTask) ? YES : NO;
}
#endif
