/*
 * fs.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ۥե륷ƥࡣ
 * ե륷ƥ
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/limits.h>
#include <sys/utime.h>
#include <sys/resource.h>
#include <sys/conf.h>
#include <machine/syscall.h>
#include <net/net.h>
#include <lib/lib.h>
#include <lib/lib_path.h>
#include <kern/kmalloc.h>
#include <kern/proc.h>
#include <kern/Thread.h>
#include <kern/errno.h>
#include <kern/time.h>
#include <kern/ProcSignal.h>
#include <kern/devfs.h>
#include <kern/vfs.h>
#include <kern/fs.h>
#include <kern/timer.h>

#include <kern/debug.h>


//#define DEBUG_FS 1
#ifdef DEBUG_FS
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


//================================== PRIVATE ============================================

/*
 * եǥץֹ
 * return : fd number or error number
 */
STATIC int getEmptyFdNum(FILE_STRUCT *fstruct, const uint min)
{
	int num;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= min) {
		return -EINVAL;
	}

	for (num = min; fstruct->fd[num] != NULL;) {
		if (_POSIX_OPEN_MAX == ++num){
			num = 0;
		}
		if (num == min){
			return -ENFILE;
		}
	}

	return num;
}

//------------------ ȥǥ쥯ȥΥåȡꥻå ---------

/*
 *աղۥȥ꡼ޥȥȥ꡼Ǥ뤳ȡ
 * Set current directory.
 */
STATIC void setCurrentDir(void *vent)
{
	FILE_STRUCT *fstruct = getFileStructProc(getCurrentProc());

	if (fstruct->current_dir != NULL) {
		closeVent(fstruct->current_dir);
	}

	fstruct->current_dir = vent;
}

//------------------ եǥץե饰μå ---------

/*
 * Get file descriptor flag.
 * return : flag or error number
 */
STATIC int getFdFlag(uint num)
{
	FILE_STRUCT *fstruct;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= num) {
		return -EBADF;
	}

	fstruct = getFileStructProc(getCurrentProc());
	if (fstruct->fd[num] == NULL) {
		return -EBADF;
	}

	return fstruct->flag[num];
}

/*
 * Set file descriptor flag.
 * return : 0 or error number
 */
STATIC int setFdFlag(uint num,int flag)
{
	FILE_STRUCT *fstruct;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= num) {
		return -EBADF;
	}
	fstruct = getFileStructProc(getCurrentProc());
	if (fstruct->fd[num] == NULL) {
		return -EBADF;
	}
	fstruct->flag[num] = flag;

	return 0;
}

/*
 * Get file open mode flag.
 * returm : open mode flag
 */
STATIC int getOpenFlag(uint num)
{
	FILE_STRUCT *fstruct;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= num){
		return -EINVAL;
	}
	fstruct = getFileStructProc(getCurrentProc());
	if (fstruct->fd[num] == NULL) {
		return -EBADF;
	}

	return fstruct->fd[num]->accessMode;
}

/*
 * Set file open mode flag.
 * returm : open mode flag
 */
STATIC int setOpenFlag(uint num,int flag)
{
	FILE_STRUCT *fstruct;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= num) {
		return -EBADF;
	}
	fstruct = getFileStructProc(getCurrentProc());
	if (fstruct->fd[num] == NULL) {
		return -EBADF;
	}

	fstruct->fd[num]->accessMode &= ~(O_APPEND | O_NONBLOCK);
	fstruct->fd[num]->accessMode |= flag & (O_APPEND | O_NONBLOCK);

	return 0;
}

/*
 * Copy file descriptor.
 * return : file descriptor number or error number
 */
STATIC int copyFd(int num1,int num2)
{
	FILE_STRUCT *fstruct = getFileStructProc(getCurrentProc());

	// ǥץֹ
	if (_POSIX_OPEN_MAX <= (uint)num2) {
		num2 = 0;
	}
	num2 = getEmptyFdNum(fstruct,num2);

	// եǥץ򥳥ԡ
	fstruct->fd[num2] = fstruct->fd[num1];
	++fstruct->fd_count;
	++fstruct->fd[num2]->refCount;
	fstruct->flag[num2] = 0;			// Clear close-on-exec flag.
	linkVent(fstruct->fd[num2]->vent);

	return num2;
}

/*
 * file descriptorλȥ󥿤򸺤餹
 * return : error number
 */
STATIC int unlinkFd(FILE_STRUCT *fstruct, int fdNum)
{
	F_DSC *fd;

	fd = fstruct->fd[fdNum];
	--fd->refCount;
	if (fd->form == FD_FORM_NETSOKET){
		if (fd->refCount <= 0){
			closeSocket((SOCKET*)fd);
		}
	}
	else{
		closeVent(fd->vent);
		if (fd->refCount <= 0){
			kfree(fd);
		}
	}

	fstruct->fd[fdNum] = NULL;
	fstruct->next_fd = fdNum;
	--fstruct->fd_count;

	return NOERR;
}

//================================== PUBLIC =============================================

/*
 * fork˥եطι¤Τƥץ饳ԡ롣
 * return : child file structure or error=NULL
 */
FILE_STRUCT *copyFileStruct(FILE_STRUCT *fstruct)
{
	FILE_STRUCT *cp_struct;
	int i, cnt;

	cp_struct = kmalloc(sizeof(FILE_STRUCT));
	if (cp_struct == NULL){
		return NULL;
	}
	memcpy(cp_struct, fstruct, sizeof(FILE_STRUCT));

	// եǥץλȥ󥿤1䤹
	for (cnt = fstruct->fd_count, i = 0; 0 < cnt; ++i){
		if (cp_struct->fd[i] != NULL){
			++cp_struct->fd[i]->refCount;
			if (cp_struct->fd[i]->form == FD_FORM_LOCALFILE){
				linkVent(cp_struct->fd[i]->vent);
			}
			--cnt;
		}
	}

	return cp_struct;
}

/*
 * եطι¤Τ롣
 */
void releaseFileStruct(FILE_STRUCT *fstruct)
{
	int i;

	for (i = 0; i < _POSIX_OPEN_MAX; ++i){
		if (fstruct->fd[i] != NULL){
			unlinkFd(fstruct, i);
			if (fstruct->fd_count <= 0){
				break;
			}
		}
	}

	kfree(fstruct);
}

/*
 * ե빽¤Τν
 */
void initExecFs()
{
	FILE_STRUCT *fstruct = getFileStructProc(getCurrentProc());
	int i, cnt;

	// Close file descriptor.
	for (cnt = fstruct->fd_count, i = 0; 0 < cnt; ++i) {
		if (fstruct->fd[i] != NULL){
			if ((fstruct->flag[i] & FD_CLOEXEC) == FD_CLOEXEC) {
				unlinkFd(fstruct,i);
			}
			--cnt;
		}
	}
}

/*
 * ɥץѥե빽¤Τν
 * return : error number
 */
int initFileStruct(
	void *proc)
{
	FILE_STRUCT *fstruct;

	fstruct = kmalloc(sizeof(FILE_STRUCT));
	if (fstruct == NULL) {
		return -ENOMEM;
	}
	memset(fstruct, 0, sizeof(FILE_STRUCT));
	fstruct->current_dir = getRootEnt();
	setFileStructProc(proc, fstruct);

	return NOERR;
}

/*
 * եǥץν
 */
void initFd(void *vent, const int form, const int amode, F_DSC *m_fd)
{
	m_fd->form = form;
	m_fd->accessMode = amode;
	m_fd->f_pos = 0;
	m_fd->refCount = 1;
	m_fd->vent = vent;
}

/*
 * եǥץϿ롣
 * return : number of file descriptor
 */
int setFd(F_DSC *fd, const int flag)
{
	int num;
	FILE_STRUCT *fstruct = getFileStructProc(getCurrentProc());

	num = getEmptyFdNum(fstruct, fstruct->next_fd);
	fstruct->next_fd = (num + 1 < _POSIX_OPEN_MAX)? num + 1 : 0;
	++fstruct->fd_count;
	fstruct->fd[num] = fd;
	fstruct->flag[num] = flag;

	return num;
}

/*
 * File descriptor롣
 * return : file descriptor or NULL
 */
F_DSC *getFd(int num)
{
	if (_POSIX_OPEN_MAX <= (uint)num){
		return NULL;
	}
	return ((FILE_STRUCT*) getFileStructProc(getCurrentProc()))->fd[num];
}

/*
 * inodeȥ꡼
 * return : inodeȥ꡼
 */
void *getVentFromFdsc(
	const F_DSC *i_fdsc)
{
	return i_fdsc->vent;
}

/*
 * ץե礫
 * return : YES or NO
 */
int isFileOpenMax()
{
	FILE_STRUCT *fstruct = getFileStructProc(getCurrentProc());

	return  (_POSIX_OPEN_MAX <= fstruct->fd_count) ? YES : NO;
}

/*
 * Get current directory entry.
 * return : current directory entry
 */
void *getCurrentDir()
{
	return ((FILE_STRUCT*) getFileStructProc(getCurrentProc()))->current_dir;
}

/************************************************************************************
 *
 * ƥॳ
 *
 ************************************************************************************/

//================================== PROTECTED ==========================================

/*
 * fd_set¤ƱΤOR黻
 */
STATIC void orFdset(const int lastFdNum, fd_set *dstFdSet, fd_set *srcFdSet)
{
	int last = (lastFdNum < FD_SETSIZE)? lastFdNum / CHAR_BIT + 1 : FD_BITMAP_LEN;
	int i;
	
	for (i = 0; i < last; ++i){
		dstFdSet->bitmap[i] |= srcFdSet->bitmap[i];
	}
}

/*
 * fd_set¤ΤΥǥץֹʾꤵƤǥץֹ֤
 * return : number of discriptor or -1 = ʤ
 */
STATIC int getFdFromFdset(const int lastFdNum, const int firstFd, const fd_set *fdSet)
{
	int fdNum;
	int array;
	int last = (lastFdNum < FD_SETSIZE)? lastFdNum / CHAR_BIT + 1 : FD_BITMAP_LEN;
	
	for (array = firstFd / CHAR_BIT, fdNum = firstFd % CHAR_BIT; array < last; ++array){
		if (fdSet->bitmap[array] != 0){
			for (; fdNum < CHAR_BIT; ++fdNum){
				if (fdSet->bitmap[array] & (1 << fdNum)){
					return array * CHAR_BIT + fdNum;
				}
			}
		}
		fdNum = 0;
	}
	return -1;
}

//================================== PUBLIC =============================================

/*
 * ¹ԥե륪ץ̾Υץδάؿ
 * return : error number
 */
int exec_open(const char *i_path, EXEC_FD *fd)
{
	void *vent;
	int error;

	if (*i_path == '\0') {
		return -ENOENT;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}

	error = openObj(i_path, O_RDONLY, &vent);
	if (error != NOERR) {
		return error;
	}
	if (getTypeVent(vent) != S_IFREG) {
		return -EACCES;
	}
	fd->vent = vent;

	return NOERR;
}

/*
 * ¹ԥեɤ߹ߡ"sys_read"δάؿ
 * parameters : open file struct,buffer,read size
 * return : read bytes or error number
 */
int exec_read(EXEC_FD *fd, void *buf, size_t size)
{
	return readVent(fd->vent, size, fd->offset, buf);
}

/*
 * ¹ԥեΥ
 */
void exec_close(EXEC_FD *fd)
{
	releaseVent(fd->vent);
}

int sys_mount(const char *devPath, const char *fsName, const char *i_path)
{
	void *mountRoot;
	void *mountDevDsc;
	FS *mountFs;
	void *vent;
	int error;

	if (PATH_MAX < strnlen(devPath, PATH_MAX + 1)){
		return -ENAMETOOLONG;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)){
		return -ENAMETOOLONG;
	}

	/* Search file system. */
	mountFs = search_fs(fsName);
	if (mountFs == NULL){
		return -ENOENT;
	}

	/* ǥХץ */
	error = openDevice(devPath, &mountDevDsc);
	if (error < 0){
		return error;
	}

	// ޥȴؿƤӽФ
	error = mountFs->mount(mountDevDsc, &mountRoot);
	if (error < 0){
		return error;
	}

	error = openPath(i_path, O_RDWR, &vent);
	if (error != NOERR) {
		goto ERR;
	}

	// ۥǥ쥯ȥޥȤ
	error = mountToVfs(mountFs, mountDevDsc, mountRoot, vent);
	if (error != NOERR){
		goto ERR2;
	}

	return NOERR;
ERR2:
	closeVent(vent);
ERR:
	mountFs->umount(mountDevDsc, mountRoot);
	return error;
}

int sys_umount(const char *i_path)
{
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)){
		return -ENAMETOOLONG;
	}

	return umountPath(i_path);
}

off_t sys_lseek(int fdNum, off_t offset, int whence)
{
	struct stat stat;
	F_DSC *fd;
	off_t rest;
	int error;

	fd = getFd(fdNum);
	if (fd == NULL) {
		return -EBADF;
	}

	error = statVent(fd->vent, &stat);
	if (error != NOERR) {
		return error;
	}

	switch(whence){
	case SEEK_CUR:
		rest = fd->f_pos + offset;
		break;
	case SEEK_SET:
		rest = offset;
		break;
	case SEEK_END:
		rest = stat.st_size - offset;
		break;
	default:
		return -EINVAL;
	}
	if (rest < 0) {
		return -EINVAL;
	}

	return fd->f_pos = rest;
}

/*
 *ԸƤFIFOեǤΥץ
 * return : file descriptor number or error number
 */
int sys_open(const char *i_path, const int i_oflag, const mode_t i_mode)
{
	void *vent;
	F_DSC *fd;
	int fdNum;
	int error;

	// ѥĹθ
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}
	if(*i_path == '\0') {
		return -ENOENT;
	}

	// ץΥå
	if (isFileOpenMax() == YES){
		return -EMFILE;
	}

	if ((i_oflag & (O_CREAT | O_EXCL)) == (O_CREAT | O_EXCL)) {
		mode_t mode = i_mode | ((i_oflag & O_FIFO) ? S_IFIFO : S_IFREG | S_IRUSR | S_IWUSR);

		// ¥եκ
		error = creatObj(i_path, mode);
		if (error != NOERR) {
			goto ERR;
		}
	}
	
	// եΥץ
	error = openPath(i_path, i_oflag, &vent);
	switch (error) {
	case NOERR:
		break;
	case -ENOENT:
		if ((i_oflag & O_CREAT) != 0) {
			mode_t mode = i_mode | ((i_oflag & O_FIFO) ? S_IFIFO : S_IFREG | S_IRUSR | S_IWUSR);

			// ¥եκ
			error = creatObj(i_path, mode);
			if (error != NOERR) {
				goto ERR;
			}
			error = openPath(i_path, i_oflag, &vent);
			if (error != NOERR) {
				goto ERR;
			}
			break;
		}
		else {
			goto ERR;
		}
	default:
		goto ERR;
	}

	// ǥ쥯ȥɤ߹ߤΤ
	if (getTypeVent(vent) == S_IFDIR) {
		if ((i_oflag & O_RDONLY) == 0) {
			error = -EISDIR;
			goto ERR2;
		}
	}

	// TRUNCATEǥץ
	if ((i_oflag & O_TRUNC) == O_TRUNC) {
		if (getTypeVent(vent) == S_IFREG) {
			error = ioctlVent(vent, I_TRUNC, NULL, 0);
			if (error < 0) {
				goto ERR2;
			}
		}
	}

	// FIFOեǤΥץ
/*	if ((i_oflag & O_FIFO) == O_FIFO) {
		vent->type = S_IFIFO;
	}
*/
	// եǥץν
	fd = kmalloc(sizeof(F_DSC));
 	if (fd == NULL) {
		error = -ENOMEM;
		goto ERR2;
	}
	initFd(vent, FD_FORM_LOCALFILE, i_oflag, fd);
	
	// եǥץϿ
	fdNum = setFd(fd, FD_CLOEXEC);

	// APPENDǥץ
	if ((i_oflag & O_APPEND) == O_APPEND) {
		sys_lseek(fdNum, 0, SEEK_END);
	}

	return fdNum;
ERR2:
	closeVent(vent);
ERR:
	return error;
}

int sys_close(int fdNum)
{
	F_DSC *fd;
	FILE_STRUCT *fstruct;

	if ((fd = getFd(fdNum)) == NULL) {
		return -EBADF;
	}
	fstruct = (FILE_STRUCT*) getFileStructProc(getCurrentProc());

	return unlinkFd(fstruct, fdNum);
}

int sys_read(int fdNum, void *buf, size_t size)
{
	F_DSC *fd;
	int rest;

	/* Хåեå */
	if (checkMem(buf, size) == ERR) {
		return -EFAULT;
	}

	fd = getFd(fdNum);
	if (fd == NULL) {
		return -EBADF;
	}

	/* ⡼ɤΥå */
	if ((fd->accessMode & (O_RDONLY | O_RDWR)) == 0) {
		return -EBADF;
	}

	if (getTypeVent(fd->vent) != S_IFIFO) {
		rest = readVent(fd->vent, size, fd->f_pos, buf);
		if (0 < rest) {
			fd->f_pos += rest;
		}
	}
	else{
		rest = readVent(fd->vent, size, 0, buf);
	}

	return rest;
}

int sys_write(int fdNum, void *buf, size_t size)
{
	F_DSC *fd;
	int rest;

	/* Хåեå */
	if (checkMem(buf, size) == ERR){
		return -EFAULT;
	}

	fd = getFd(fdNum);
	if (fd == NULL){
/****************************************************************************************************************/
((char*) buf)[size] = '\0';
printk("%s", buf);
return size;
/****************************************************************************************************************/
		return -EBADF;
	}

	/* ⡼ɤΥå */
	if ((fd->accessMode & (O_WRONLY | O_RDWR)) == 0){
		return -EBADF;
	}

	if (getTypeVent(fd->vent) == S_IFIFO) {
		rest = readVent(fd->vent, size, 0, buf);
	}
	else{
		rest = writeVent(fd->vent, size, fd->f_pos, buf);
		if (0 < rest) {
			fd->f_pos += rest;
		}
	}

	return rest;
}

int sys_ioctl(int fdNum, int cmd, caddr_t parm, int fflag)
{
	F_DSC *fd;

	if ((fd = getFd(fdNum)) == NULL) {
		return -EBADF;
	}

	return ioctlVent(fd->vent, cmd, parm, fflag);
}

int sys_mkdir(const char *i_path, const mode_t mode)
{
	if (*i_path == '\0'){
		return -ENOENT;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)){
		return -ENAMETOOLONG;
	}

	return mkdirPath(i_path, mode);
}

int sys_unlink(const char *i_path)
{
	if (*i_path == '\0') {
		return -ENOENT;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)){
		return -ENAMETOOLONG;
	}

	return deleteObj(i_path, S_IFREG | S_IFLNK);
}

int sys_rmdir(const char *i_path)
{
	if (*i_path == '\0') {
		return -ENOENT;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}

	return deleteObj(i_path, S_IFDIR);
}

int sys_opendir(const char *i_path)
{
	void *vent;
	uint block;
	int index;
	DIR_STRAEM *dir;
	int fdNum;
	int error;

	if (*i_path == '\0') {
		return -ENOENT;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}

	error = openPath(i_path, O_RDONLY, &vent);
	if (error != NOERR) {
		goto ERR;
	}
	error = opendirVent(vent, &block, &index);
	if (error != NOERR){
		goto ERR2;
	}

	// ǥ쥯ȥꥹȥ꡼κ
	dir = kmalloc(sizeof(DIR_STRAEM));
	if (dir == NULL){
		error = -ENOMEM;
		goto ERR2;
	}
	initFd(vent, FD_FORM_DIRSTREAM, 0, (F_DSC*)dir);
	fdNum = setFd((F_DSC*)dir, FD_CLOEXEC);
	dir->block = block;
	dir->index = index;

	return fdNum;
ERR2:
	closeVent(vent);
ERR:
	return error;
}

int sys_readdir(int dirNum, char *name)
{
	DIR_STRAEM *dir;

	dir = (DIR_STRAEM*)getFd(dirNum);
	if (dir == NULL){
		return -EBADF;
	}
	if (dir->fd.form != FD_FORM_DIRSTREAM) {
		return -EBADF;
	}

	return readdirVent(dir->fd.vent, &dir->block, &dir->index, name);
}

int sys_rewinddir(int dirNum)
{
	DIR_STRAEM *dir;

	dir = (DIR_STRAEM*)getFd(dirNum);
	if(dir == NULL){
		return -EBADF;
	}
	if (dir->fd.form != FD_FORM_DIRSTREAM) {
		return -EBADF;
	}
	dir->index = 0;

	return 0;
}

int sys_rename(const char *oldPath, const char *newPath)
{
	if (strcmp(oldPath, newPath) == 0){
		return 0;
	}

	return renamePath(oldPath, newPath);
}

int sys_chdir(const char *i_path)
{
	void *vent;
	int error;

	if (*i_path == '\0') {
		return -ENOENT;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}

	/* ե빽¤Τ˿ȥǥ쥯ȥϿ */
	vent = getChdirVentPath(i_path, &error);
	if (error != NOERR) {
		return error;
	}
	setCurrentDir(vent);

	return NOERR;
}

int sys_fchdir(int fdNum)
{
	F_DSC *fd;
	void *vent;
	int error;

	fd = getFd(fdNum);
	if (fd == NULL){
		return -EBADF;
	}

	/* ե빽¤Τ˿ȥǥ쥯ȥϿ */
	vent = getChdirVent(fd->vent, &error);
	if (error != NOERR) {
		return error;
	}
	setCurrentDir(vent);

	return NOERR;
}

int sys_getcwd(char *strBuf, size_t size)
{
	int len;
	void *vent;

	if (size == 0){
		return -EINVAL;
	}

	/* Хåեå */
	if (checkMem(strBuf, size) == ERR){
		return -EFAULT;
	}

	vent = getCurrentDir();
	len = copyRootPath(vent, size, strBuf);
	if (len == size) {
		return -ERANGE;
	}
	strBuf[len] = '\0';

	return 0;
}

int sys_stat(const char *i_path, struct stat *m_stat)
{
	if (checkMem(m_stat, sizeof(struct stat)) == ERR) {
		return -EFAULT;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}

	return statPath(i_path, m_stat);
}

int sys_fstat(int fdNum, struct stat *m_stat)
{
	F_DSC *fd;

	if (checkMem(m_stat, sizeof(struct stat)) == ERR){
		return -EFAULT;
	}
	if ((fd = getFd(fdNum)) == NULL){
		return -EBADF;
	}
	m_stat->st_crtpos = fd->f_pos;

	return statVent(fd->vent, m_stat);
}

int sys_lstat(const char *i_path, struct stat *m_stat)
{
	if (checkMem(m_stat, sizeof(struct stat)) == ERR) {
		return -EFAULT;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}

	return statPathSymlink(i_path, m_stat);
}

/*
 * 롼ȥե륷ƥ򥢥ޥȤ롣
 */
int sys_umount_root()
{
	umountAllFs();

	return 0;
}

int sys_fcntl(int fdNum, int cmd, uint arg)
{
	switch(cmd){
	case F_DUPFD:{
		if (isFileOpenMax() == YES){
			return -EMFILE;
		}
		if (getFd(fdNum) == NULL){			/* File descriptorΥå */
			return -EBADF;
		}
		return copyFd(fdNum,arg);			/* Copy File descriptor. */
	}
	case F_GETFD:
		return getFdFlag(fdNum);
	case F_SETFD:
		return setFdFlag(fdNum,arg);
	case F_GETFL:
		return getOpenFlag(fdNum);
	case F_SETFL:
		return setOpenFlag(fdNum, arg);
	case F_GETLK:
		//
	case F_SETLK:
		//
	case F_SETLKW:
		//
	case F_GETOWN:
		//
	case F_SETOWN:
		//
	default:
		return -ENOTSUP;
	}
}

int sys_chmod(const char *i_path, const mode_t mode)
{
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}

	return chmodPath(i_path, mode);
}

int sys_fchmod(const int fildes, const mode_t mode)
{
	F_DSC *fd;

	fd = getFd(fildes);
	if (fd == NULL) {
		return -EBADF;
	}

	return chmodVent(fd->vent, mode);
}

int sys_chown(const char *i_path, const uid_t owner, const gid_t group)
{
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}

	return chownPath(i_path, owner, group);
}

int sys_fchown(const int fildes, const uid_t owner, const gid_t group)
{
	F_DSC *fd;

	fd = getFd(fildes);
	if (fd == NULL) {
		return -EBADF;
	}

	return chownVent(fd->vent, owner, group);
}

int sys_utimes(const char *i_path, const struct timeval *times)
{
	return utimesPath(i_path, times);
}

int sys_link(const char *fromPath, const char *toPath)
{
	if (PATH_MAX < strnlen(fromPath, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}
	if (PATH_MAX < strnlen(toPath, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}

	return linkPath(fromPath, toPath);
}

/*
 * Хѥꤹ
 */
int sys_symlink(char *linkPath, const char *srcPath)
{
	int len;

	if (PATH_MAX < strnlen(srcPath, PATH_MAX + 1)) {
		return -ENAMETOOLONG;
	}
	if (*srcPath == '\0') {
		return -ENOENT;
	}

	len = strnlen(linkPath, PATH_MAX + 1);
	if (PATH_MAX < len) {
		return -ENAMETOOLONG;
	}
	return creatSymlink(srcPath, S_IFLNK | S_IRWXU | S_IRWXG | S_IRWXO, linkPath);
}

int sys_readlink(const char *i_path, char *readBuf, size_t bufSize)
{
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)){
		return -ENAMETOOLONG;
	}
	if (checkMem(readBuf, bufSize) == ERR){
		return -EFAULT;
	}

	return readlinkPath(i_path, bufSize, readBuf);
}

int sys_select(const int lastFdNum, fd_set *m_fdsRead, fd_set *m_fdsWrite, fd_set *m_fdsError, struct timeval *timeout)
{
	enum{
		FDS_LAST = 3,
	};
	int miliSeconds;
	fd_set fdSetAll;
	fd_set *fdSet[] = {m_fdsRead, m_fdsWrite, m_fdsError};
	int pollEvents[] = {POLLIN, POLLOUT, POLLERR};
	int isFds;
	int rest;
	int i;

	if (lastFdNum < 0){
		return -EINVAL;
	}

	// ॢȻ֤
	if (timeout == NULL){
		// ʥȯޤǥ֥å
		miliSeconds = -1;
	}
	else{
		miliSeconds = (timeout->tv_sec * 1000) + (timeout->tv_usec / 1000);
	}

	// FD
	FD_ZERO(&fdSetAll);
	isFds = NO;
	for (i = 0; i < FDS_LAST; ++i){
		if (fdSet[i] != NULL){
			if (checkMem(fdSet[i], sizeof(fd_set)) == ERR){
				return -EFAULT;
			}
			isFds = YES;
			orFdset(lastFdNum, &fdSetAll, fdSet[i]);
		}
	}

	if (isFds == YES){
		// ٥Ȥθ
		for (;;){
			int fdNum;

			for (fdNum = 0; (fdNum = getFdFromFdset(lastFdNum, fdNum, &fdSetAll)) != -1;){
				int c_update = 0;
				int events;
				F_DSC *fd = getFd(fdNum);

				if (fd == NULL){
					return -EBADF;
				}
				// Ϥ٥Ȥ
				events = 0;
				for (i = 0; i < FDS_LAST; ++i){
					if (FD_ISSET(fdNum, fdSet[i])){
						events |= pollEvents[i];
					}
				}
				rest = pollVent(fd->vent, events);
				
				// ǽFDΥ
				for (i = 0; i < FDS_LAST; ++i){
					if ((rest & pollEvents[i]) == pollEvents[i]){
						if (fdSet[i] != NULL){
							if (FD_ISSET(fdNum, fdSet[i])){
								c_update += 1;
							}
						}
					}
				}
				// FD_SET˥٥Ȥ
				if (0 < c_update){
					for (i = 0; i < FDS_LAST; ++i){
						if ((rest & pollEvents[i]) == pollEvents[i]){
							if (fdSet[i] != NULL){
								if (FD_ISSET(fdNum, fdSet[i])){
									FD_ZERO(fdSet[i]);
									FD_SET(fdNum, fdSet[i]);
								}
								else{
									FD_ZERO(fdSet[i]);
								}
							}
						}
					}
					return c_update;
				}
				fdNum += 1;
			}
			
			if (miliSeconds == 0){
				// ǥץ̵
				for (i = 0; i < FDS_LAST; ++i){
					if (fdSet[i] != NULL){
						FD_ZERO(fdSet[i]);
					}
				}
				return 0;
			}
			
			// Ԥ
			waitTimeSignal(miliSeconds);

			// ʥߤ
			if(isSigint(TaskGetTaskSignal(getCurrentTask())) == YES) {
				return -EINTR;
			}

			// ܤԤ֤򥯥ꥢ
			miliSeconds = 0;
		}
	}
	else{
		// ʥԤ
		waitTimeSignal(miliSeconds);

		return 0;
	}
}

int sys_poll(
	struct pollfd *m_pollfd,
	const nfds_t i_nfd,
	const int i_miliSeconds)
{
	int c_complete;			// λǥץ
	int miliSeconds;

	if (RLIMIT_NOFILE < i_nfd){
		return -EINVAL;
	}

	// ꡼å
	if (checkMem(m_pollfd, sizeof(*m_pollfd) * i_nfd) == ERR) {
		return -EFAULT;
	}
	
	// ॢȻ֤
	if (i_miliSeconds < 0){
		miliSeconds = -1;
	}
	else{
		miliSeconds = i_miliSeconds;
	}

	// ٥Ȥθ
	for (;;){
		int isWait = NO;
		int rest;
		int i;

		c_complete = 0;
		for (i = 0; i < i_nfd; ++i){
			// -1ʤΡå
			if (m_pollfd[i].fd == -1) {
				m_pollfd[i].revents = 0;
			}
			else {
				F_DSC *fd = getFd(m_pollfd[i].fd);
				if (fd == NULL){
					m_pollfd[i].revents |= POLLNVAL;
					continue;
				}

				m_pollfd[i].revents = 0;
				rest = pollVent(fd->vent, m_pollfd[i].events);
				if (rest & (POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI)){
					if (m_pollfd[i].events & (POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI)){
						m_pollfd[i].revents |= rest;
						c_complete += 1;
						continue;
					}
				}
				else if (rest & (POLLOUT | POLLWRNORM | POLLWRBAND)){
					if (m_pollfd[i].events & (POLLOUT | POLLWRNORM | POLLWRBAND)){
						m_pollfd[i].revents |= rest;
						c_complete += 1;
						continue;
					}
				}
				else if (rest & (POLLERR | POLLHUP)){
					m_pollfd[i].revents |= (rest & (POLLERR | POLLHUP));
					continue;
				}
				else if (rest < 0){
					m_pollfd[i].revents |= POLLNVAL;
					continue;
				}
				isWait = YES;
			}
		}

		if (0 < c_complete) {
			break;
		}

		if (isWait == NO) {
			break;
		}
		if (miliSeconds == 0){
			break;
		}
		waitTimeSignal(miliSeconds);

		// ʥߤ
		if(isSigint(TaskGetTaskSignal(getCurrentTask())) == YES) {
			return -EINTR;
		}
	}

	return c_complete;
}

int sys_getstatfs(struct statfs *m_statfs, long bufsize, int flags)
{
	struct statfs *statfs = m_statfs;
	uint count;

	count = (statfs == NULL) ? INT_MAX : bufsize / sizeof(struct statfs);
	if (statfs != NULL) {
		if (checkMem(statfs, sizeof(*statfs) * count) == ERR){
			return -EFAULT;
		}
	}

	return searchMount(statfs, count);
}

int sys_statfs(const char *i_path, struct statfs *m_statfs)
{
	if (*i_path == '\0'){
		return -ENOENT;
	}
	if (PATH_MAX < strnlen(i_path, PATH_MAX + 1)){
		return -ENAMETOOLONG;
	}
	if (checkMem(m_statfs,sizeof(struct statfs)) == ERR){
		return -EFAULT;
	}

	return statfsPath(i_path, m_statfs);
}

/*
 * եǥץü̾롣
 * return : error number
 */
int sys_ttyname(
	int fdnum, 
	char *buf)
{
	F_DSC *fd;

	if ((fd = getFd(fdnum)) == NULL) {
		return -EBADF;
	}

	return getTtynameVent(fd->vent, buf);
}

/************************************* ̤ ***************************************/
int sys_lock()
{
	return 0;
}


/************************************************************************************
 *
 * 
 *
 ************************************************************************************/

//================================== PUBLIC =============================================

/*
 * Init process0 file struct.
 * return : error number
 */
int setInitProcFs()
{
	int fdNum;

	/*
	 * ɸϤ򳫤
	 */
	fdNum = sys_open("/dev/console", O_RDONLY, 0);	/* ɸϡ */
	if (fdNum < 0){
		return fdNum;
	}
	fdNum = setFdFlag(fdNum, 0);						/* No exec fd close. */
	if (fdNum < 0){
		return fdNum;
	}
	fdNum = sys_open("/dev/console", O_WRONLY, 0);	/* ɸϡ */
	if (fdNum < 0){
		return fdNum;
	}
	fdNum = setFdFlag(fdNum, 0);
	if (fdNum < 0){
		return fdNum;
	}
	fdNum = sys_open("/dev/console", O_WRONLY, 0);	/* ɸ२顼 */
	if (fdNum < 0){
		return fdNum;
	}
	fdNum = setFdFlag(fdNum, 0);
	if (fdNum < 0){
		return fdNum;
	}

	return NOERR;
}

/*************************************************************************************************/

void testFs(int num, int nest)
{
	printk("testFs() %d %d\n", num, nest);
}
