/*
 * term.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/types.h>
#include <sys/param.h>
#include <kern/proc.h>
#include <kern/Thread.h>
#include <kern/ProcSignal.h>
#include <kern/term.h>
#include <lib/lib.h>
#include <kern/proc.h>
#include <kern/timer.h>

#include <kern/debug.h>


//#define DEBUG_TERM 1
#ifdef DEBUG_TERM
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif


/***************************************************************************************
 *
 * ü
 *
 ***************************************************************************************/

//================================== PRIVATE ============================================

/* ե󥯥󥭡 */
#define	FUNK_HOME	"\033[H"
#define	FUNK_UP		"\033[A"
#define	FUNK_DOWN	"\033[B"
#define	FUNK_LEFT	"\033[D"
#define	FUNK_RIGHT	"\033[C"

/*
 * ץ򵯾롣
 */
STATIC INLINE void wakeTermTask(
	TERM_CTL *m_ctl)
{
	if (m_ctl->waitTask != NULL) {
		void *waitTask = m_ctl->waitTask;

		m_ctl->waitTask = NULL;
		activeTaskSoonSignal(getWaitTask(waitTask));
	}
}

/*
 * 1ʸɲ
 */
STATIC void putToBuf(char c,LINE_BUF *lbuf)
{
	if (lbuf->crt == LINE_BUF_SIZE) {
		return;
	}

	/* Хåե */
	if (lbuf->last == LINE_BUF_SIZE) {
		--lbuf->last;
	}
	memCpyBack(lbuf->buf + lbuf->crt + 1, lbuf->buf + lbuf->crt, lbuf->last - lbuf->crt);
	lbuf->buf[lbuf->crt] = c;
	++lbuf->last;
	++lbuf->crt;
}

/*
 * Ȱ֤򺸤˰ư
 s*/
STATIC void leftCursorBuf(LINE_BUF *lbuf)
{
	lbuf->pastCrt = lbuf->crt;
	if (lbuf->crt > lbuf->lineTop) {
		--lbuf->crt;
	}
}

/*
 * Ȱ֤򱦤˰ư
 */
STATIC void rightCursorBuf(LINE_BUF *lbuf)
{
	lbuf->pastCrt = lbuf->crt;
	if (lbuf->crt < lbuf->last) {
		++lbuf->crt;
	}
}

/*
 * ʸɲ
 */
STATIC void inputStrBuf(char *str,size_t n,LINE_BUF *lbuf)
{
	int i;
	
	lbuf->pastCrt = lbuf->crt;
	lbuf->pastLast = lbuf->last;
	for (i = 0; i < n; ++i) {
		putToBuf(str[i],lbuf);
	}
}

/*
 * +1ʹߤʸ˰ưǸ˶
 */
STATIC void deleteBuf(LINE_BUF *lbuf)
{
	lbuf->pastLast = lbuf->last;
	if(lbuf->last > lbuf->crt) {
		memcpy(lbuf->buf + lbuf->crt, lbuf->buf + lbuf->crt + 1, lbuf->last - lbuf->crt);
		lbuf->buf[lbuf->last - 1] = '\0';
		--lbuf->last;
	}
}

/*
 * Ȱʹߤʸ˰ưǸ˶
 */
STATIC void backSpaceBuf(LINE_BUF *lbuf)
{
	lbuf->pastCrt = lbuf->crt;
	lbuf->pastLast = lbuf->last;
	if(lbuf->crt>lbuf->lineTop)
	{
		memcpy(lbuf->buf+lbuf->crt - 1, lbuf->buf + lbuf->crt, lbuf->last-lbuf->crt);
		lbuf->buf[lbuf->last - 1] = '\0';
		--lbuf->crt;
		--lbuf->last;
	}
}

/*
 * 饤õ
 */
STATIC void eraseLineBuf(LINE_BUF *lbuf)
{
	lbuf->pastCrt = lbuf->crt;
	lbuf->pastLast = lbuf->last;
	memset(lbuf->buf + lbuf->lineTop, 0, lbuf->last - lbuf->lineTop);
	lbuf->last = lbuf->lineTop;
	lbuf->crt = lbuf->lineTop;
}

/*
 * 
 */
STATIC void newLineBuf(LINE_BUF *lbuf)
{
	if(lbuf->last < LINE_BUF_SIZE) {
		lbuf->buf[lbuf->last++] = '\n';
	}
	lbuf->crt = lbuf->last;
	lbuf->lineTop = lbuf->last;
}

/*
 * EOF
 */
STATIC void eof(LINE_BUF *lbuf)
{
	if(lbuf->lineTop == lbuf->last) {	/* ʸʤ */
		lbuf->buf[lbuf->last++] = TERM_EOF;
		lbuf->crt = lbuf->last;
		lbuf->lineTop = lbuf->last;
	}
}

//================================== PUBLIC =============================================

/*
 * ȥץüΥե饦ɥ롼פ
 * return : YES or NO
 */
int isForeground(
	const void *proc)
{
	if (getCtltermProc(proc) == NULL) {
		return NO;
	}
	if (((TERM_CTL*) getCtltermProc(proc))->ctlProc == NULL) {
		return YES;
	}
	return ((TERM_CTL*) getCtltermProc(proc))->foregrnd == getPgid(proc) ? YES : NO;
}

/*
 * ȥץüץ
 * return : YES or NO
 */
int isCtlproc(
	const void *proc)
{
	if (getCtltermProc(proc) == NULL) {
		return NO;
	}
	return (((TERM_CTL*) getCtltermProc(proc))->ctlProc == proc) ? YES : NO;
}

/*
 * ߥʥ륳ȥץ
 */
void releaseTerm(
	void *proc)
{
	if (((TERM_CTL*) getCtltermProc(proc))->ctlProc == proc) {
		((TERM_CTL*) getCtltermProc(proc))->ctlProc = NULL;
	}
}

/*
 * 󥽡Хåեɤ߹߲ǽ
 * return : YES or NO
 */
int canReadTerm(
	struct termios *tio,
	TERM_CTL *ctl,			// üȥ빽¤
	LINE_BUF *lbuf)
{
	int i;

	if ((uchar)*lbuf->buf == TERM_EOF) {
		return YES;
	}

	// Υ˥⡼
	if (tio->c_lflag & ICANON) {
		for (i = 0; i < lbuf->last; ++i) {
			if (lbuf->buf[i] == '\n') {
				return YES;
			}
		}
	}
	// Υ󥫥Υ˥⡼
	else {
		if (0 < lbuf->last) {
			return YES;
		}
	}

	ctl->waitTask = getCurrentTask();
	return NO;
}

/*
 * üХåեɤ߹ࡣ
 * ɤ߹ǡ̵ХȤ롣
 * return : ϥХȿ or EOF = 0 or Ϥʤ = -1
 */
int readTerm(
	uchar *buf,
	size_t size,
	struct termios *tio,
	TERM_CTL *ctl,			// üȥ빽¤
	LINE_BUF *lbuf)
{
	int len;

	/* Ϥʤ */
	while (lbuf->last == 0) {
		// Ԥ
		ctl->waitTask = getCurrentTask();
		waitTaskSignal();

		if (isSigint(TaskGetTaskSignal(getCurrentTask())) == YES) {
			return -EINTR;
		}
	}

	// Υ˥⡼
	if (tio->c_lflag & ICANON) {
		if ((uchar) *lbuf->buf == TERM_EOF) {
			*lbuf->buf = '\0';
			lbuf->lineTop = lbuf->crt = lbuf->last = 0;

			return 0;
		}

		len = 0;
		while (lbuf->buf[len++] != '\n') {
			if (lbuf->last <= len) {
				// Ԥ
				ctl->waitTask = getCurrentTask();
				waitTaskSignal();
			}
		}
		len = (len < size) ? len : size;
	}
	// Υ󥫥Υ˥⡼
	else {
		len = (size <= lbuf->last) ? size : lbuf->last;
	}
		
	memcpy(buf, lbuf->buf, len);
	memcpy(lbuf->buf, lbuf->buf + len, lbuf->last - len);
	lbuf->crt -= len;
	lbuf->last -= len;
	lbuf->lineTop -= len;

	return len;
}

/*
 * üХåե˽񤭹
 * return : YES = 桼ɤ߹߲ǽˤʤä or NO = 桼ɤ߹ߤǤʤ
 */
int writeTerm(
	uchar ch,				// ϥ饯
	struct termios *tio,	// termios¤
	TERM_CTL *ctl,			// üȥ빽¤
	LINE_BUF *lbuf)			// Хåե
{
	cc_t *cc = tio->c_cc;

	// ̥ޥɽ
	if (ch == cc[VINTR]) {			/* Send SIGINT. */
		// ץ롼פSIGINT
		procSendSignalPgid(ctl->foregrnd, SIGINT);
		return YES;
	}
	else if (ch == cc[VQUIT]) {		/* Send SIGQUIT. */
		return NO;
	}
	else if (ch == cc[VSTART]) {	/* Send SIGCONT.  */
		return NO;
	}
	else if (ch == cc[VSTOP]) {		/* Send SIGSTOP. */
		return NO;
	}
	else if (ch == cc[VSUSP]) {		/* Send SIGTSTP. */
		// ץ롼פSIGTSTP
		procSendSignalPgid(ctl->foregrnd, SIGTSTP);
		return YES;
	}
	
	// Canonical mode.
	if (tio->c_lflag & ICANON) {
		if (ch == cc[VEOF]) {							/* եνλ */
			eof(lbuf);

			// 
			wakeTermTask(ctl);
			return YES;
		}
		else if ((ch == cc[VEOL]) || (ch == '\n')) {	/* Ԥνλ */
			newLineBuf(lbuf);
			if (tio->c_lflag & ECHO) {
				ctl->newLine();
			}
			// 
			wakeTermTask(ctl);
			return YES;
		}
		else if ((ch == cc[VERASE]) || (ch == 0x7f)) {	/* DEL */
			deleteBuf(lbuf);
			if (tio->c_lflag & ECHO) {
				ctl->delete();
			}
			return NO;
		}
		else if (ch == cc[VKILL]) {						/* á */
			eraseLineBuf(lbuf);
			if (tio->c_lflag & ECHO) {
				ctl->eraseLine();
			}
			return NO;
		}
		else {
			switch(ch) {
			case '\b':									/* ʸõ */
				backSpaceBuf(lbuf);
				if (tio->c_lflag & ECHO) {
					ctl->backSpace();
				}
				break;
			case TRM_UP:
				inputStrBuf(FUNK_UP,3,lbuf);
				if (tio->c_lflag & ECHO) {
					ctl->inputStr();
				}
				break;
			case TRM_DWN:
				inputStrBuf(FUNK_DOWN,3,lbuf);
				if (tio->c_lflag & ECHO) {
					ctl->inputStr();
				}
				break;
			case TRM_LFT:
				leftCursorBuf(lbuf);
				if (tio->c_lflag & ECHO) {
					ctl->leftCursor(1);
				}
				break;
			case TRM_RHT:
				rightCursorBuf(lbuf);
				if (tio->c_lflag & ECHO) {
					ctl->rightCursor(1);
				}
				break;
			default:
				if (ch < MIN_MAPKEY) {
					inputStrBuf(&ch,1,lbuf);
					if (tio->c_lflag & ECHO) {
						ctl->inputStr();
					}
				}
			}
			return NO;
		}
	}
	// Non-canonical mode.
	else {
		switch (ch) {
		case TRM_UP:
			inputStrBuf(FUNK_UP,3,lbuf);
			break;
		case TRM_DWN:
			inputStrBuf(FUNK_DOWN,3,lbuf);
			break;
		case TRM_LFT:
			inputStrBuf(FUNK_LEFT,3,lbuf);
			break;
		case TRM_RHT:
			inputStrBuf(FUNK_RIGHT,3,lbuf);
			break;
		default:
			inputStrBuf(&ch,1,lbuf);
		}

		// 
		wakeTermTask(ctl);
		return YES;
	}
}

//*******************************************************************
void testTerm(
	int irq)
{
	printDebug(80 * 12 + 70, "intr %d ", irq);
}
