/*
 * timer.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/callout.h>
#include <sys/systm.h>
#include <sys/limits.h>
#include <sys/AggregateList.h>
#include <machine/lib.h>
#include <machine/interrupt.h>
#include <dev/8254.h>
#include <dev/MC146818.h>
#include <kern/lock.h>
#include <kern/proc.h>
#include <kern/ProcSignal.h>
#include <kern/vm.h>
#include <kern/kmalloc.h>
#include <kern/time.h>
#include <kern/timer.h>

#include <kern/debug.h>


//#define DEBUG_TIMER 1
#ifdef DEBUG_TIMER
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


//==================================  ===========================================

enum{
	TIMER_MAX = 10,			// Υޡͭ
};

/*
 * ޡإå
 */
typedef struct {
	AggregateList	aggregate;					// ޡ󥯽
	uint			(*getPastTime)();			// ޡߤޤǤθߤηвms
	void			(*setTimer)(const uint);	// ޡms
	int				lock;						// ԥå
} TIMER_HEAD;

/*
 * ޡ¤
 */
typedef struct {
	uint		init;				// ³ޡ
	void		(*handle)(void*);	// ϥɥ
	void		*arg;				// ϥɥ
	List		ownedList;			// ͭ󥯥ꥹ

	// ޡ
	uint		count;				// Ĥ
	List		timerList;			// ޡꥹ
	TIMER_HEAD	*timerHead;			// ³Ƥ륿ޡإå³ƤʤNULL
} TIMER;

//================================== PRIVATE =============================================

/*
 * ޡΥ饤ե
 * allocTimer  registTimer  doTimer  doTimerˤ륿ॢ  λޤǥޡϳʤ
 *                                              or
 *                                         unregistTimer
 */

/*
 * 󥿡Х륿ޡإå
 */
static TIMER_HEAD intervalTimer;

/*
 * ꥢ륿ޡإå
 */
static TIMER_HEAD realTimer;

/*
 * ƥॿޡͭإå
 */
static TIMER_OWNER systemTimer;

/*
 * ޡإåν
 */
STATIC void initTimerHead()
{
	// 󥿡Х륿ޡإåν
	AggregateListConstructor(&intervalTimer.aggregate);
	intervalTimer.getPastTime = getIntervalTimerPastTime;
	intervalTimer.setTimer = setIntervalTimer;
	intervalTimer.lock = 0;
	
	// ꥢ륿ޡإåν
	AggregateListConstructor(&realTimer.aggregate);
	realTimer.getPastTime = getRealTimerPastTime;
	realTimer.setTimer = setRealTimer;
	realTimer.lock = 0;
}

/*
 * ޡ¤Τ򥿥ޡإå˻ֽ³롣
 *Թ
 */
STATIC void linkTimer(
	const int i_time,			// ޡms
	TIMER *m_timer,				// ³륿ޡ¤
	TIMER_HEAD *m_timerHead)	// ³西ޡإå
{
	uint time = i_time;
	AggregateList *aggregate = &m_timerHead->aggregate;
	IteratorList iterator;

	aggregate->iterator(aggregate, &iterator);
	for (;;) {
		TIMER *timer;
		
		if (iterator.hasNext(&iterator) == BOOL_FALSE) {
			aggregate->insertEnd(aggregate, &m_timer->timerList);
			break;
		}

		timer = iterator.next(&iterator);
		if (time <= timer->count) {
			timer->count -= time;
			aggregate->insertPrev(aggregate, &timer->timerList, &m_timer->timerList);
			break;
		}
		else {
			time -= timer->count;
		}
	}
	m_timer->count = time;
}

/*
 * ޡ¤Τ򥿥ޡإåڤΥ
 *Թ
 */
STATIC void unlinkTimer(
	TIMER *m_timer,				// ڤΥޡ¤
	TIMER_HEAD *m_timerHead)	// ³西ޡإå
{
	AggregateList *aggregate = &m_timerHead->aggregate;
	TIMER *next = aggregate->refNext(aggregate, &m_timer->timerList);

	ASSERT(m_timer != NULL);

	if (next != NULL) {
		next->count += m_timer->count;
	}
	aggregate->removeEntry(aggregate, &m_timer->timerList);
}

/*
 * ֤вᤷޡ򥿥ޡإåڤΥƼ롣
 *ԳաԹ
 */
STATIC void getTimeoutTimer(
	const uint i_time,						// вms
	TIMER_HEAD *m_timerHead,				// ³西ޡإå
	AggregateList *m_aggregateUnlinkList)	// ޡ¤Τ³ñꥹȥإå
{
	uint time = i_time;
	AggregateList *aggregate = &m_timerHead->aggregate;
	TIMER *timer;

	for (timer = aggregate->refHead(aggregate); timer != NULL;) {
		TIMER *next = aggregate->refNext(aggregate, &timer->timerList);

		if (timer->count <= time) {
			aggregate->removeEntry(aggregate, &timer->timerList);
			time -= timer->count;
			timer->count = 0;
			m_aggregateUnlinkList->insertHead(m_aggregateUnlinkList, &timer->timerList);
		}
		else {
			timer->count -= time;
			break;
		}
		
		timer = next;
	}
}

/*
 * ޡ󥯤֤򸺤餹
 *Թ
 */
STATIC void subtractTimeTimerLink(
	const uint time,			//  ms
	TIMER_HEAD *i_timerHead)	// ³西ޡإå
{
	AggregateList *aggregate = &i_timerHead->aggregate;
	TIMER *timer = aggregate->refHead(aggregate);
	
	if (timer == NULL) {
		return;
	}

	ASSERT(time <= timer->count);
	
	timer->count -= time;
}

/*
 * ޡ󥯤饿ޡ֤ꤹ
 */
STATIC void setTimeTimerLink(
	TIMER_HEAD *i_timerHead)	// ³西ޡإå
{
	AggregateList *aggregate = &i_timerHead->aggregate;
	TIMER *timer = aggregate->refHead(aggregate);
	
	if (timer == NULL) {
		return;
	}

	// ƬΥޡ֤ꤹ
	if (timer->count == 0) {
		i_timerHead->setTimer(1);
	}
	else {
		i_timerHead->setTimer(timer->count);
	}
}

/*
 * ޡϿե饰ꤹ
 */
STATIC INLINE void setRegist(
	TIMER *m_timer,
	TIMER_HEAD *timerHead)
{
	m_timer->timerHead = timerHead;
}

/*
 * ޡϿե饰
 */
STATIC INLINE void resetRegist(
	TIMER *m_timer)
{
	m_timer->timerHead = NULL;
}

/*
 * ޡ󥯤ϿƤ뤫
 *Ի
 * return : YES or NO
 */
STATIC INLINE int isRegistTimer(
	const TIMER *timer)
{
	return (timer->timerHead != NULL) ? YES : NO;
}

/*
 *ԳաԹ
 * ֤вᤷޡϥɥ¹Ԥ롣
 */
STATIC void doTimer(
	const uint i_time,			// в
	TIMER_HEAD *m_timerHead)	// ³إå
{
	AggregateList aggregateUnlilnkTimer;	// ñꥹ
	IteratorList iterator;
	int eflags;

	AggregateSingleListConstructor(&aggregateUnlilnkTimer);

	eflags = enterCli();
	enter_spinlock(&m_timerHead->lock);
	{
		// ॢȥޡμ
		getTimeoutTimer(i_time, m_timerHead, &aggregateUnlilnkTimer);
	}
	exit_spinlock(&m_timerHead->lock);
	exitCli(eflags);

	// ޡϥɥμ¹
	aggregateUnlilnkTimer.iterator(&aggregateUnlilnkTimer, &iterator);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		TIMER *timer = iterator.next(&iterator);

		timer->handle(timer->arg);
	}

	eflags = enterCli();
	enter_spinlock(&m_timerHead->lock);
	{
		TIMER *timer;

		while ((timer = aggregateUnlilnkTimer.getHead(&aggregateUnlilnkTimer)) != NULL) {
			// ³ޡͤк³
			if (0 < timer->init) {
				linkTimer(timer->init, timer, m_timerHead);
				setRegist(timer, m_timerHead);
			}
			else {
				resetRegist(timer);
			}
		}

		// ǥХޡ֤
		setTimeTimerLink(m_timerHead);
	}
	exit_spinlock(&m_timerHead->lock);
	exitCli(eflags);
}

/*
 * ޡ¤Τ򥿥ޡإåϿ
 *Թ
 * return : error number
 */
STATIC void registTimer(
	const uint i_time,	// ޡms
	TIMER  *m_timer)
{
	TIMER_HEAD *timerHead;
	int eflags;

	// ޡإå
	if (i_time < REAL_TIMER_RATE) {
		timerHead = &intervalTimer;
	}
	else {
		timerHead = &realTimer;
	}

	eflags = enterCli();
	enter_spinlock(&timerHead->lock);
	{
		uint pastTime;

		// ޡηв֤
		pastTime = timerHead->getPastTime();

		// ޡ󥯤³
		subtractTimeTimerLink(pastTime, timerHead);
		linkTimer(i_time, m_timer, timerHead);
		setRegist(m_timer, timerHead);

		// ǥХޡ֤
		setTimeTimerLink(timerHead);
	}
	exit_spinlock(&timerHead->lock);
	exitCli(eflags);
}

/*
 * ޡ¤Τ򥿥ޡإå
 */
STATIC void unregistTimer(
	TIMER  *m_timer)
{
	ASSERT(m_timer != NULL);
	
	if (isRegistTimer(m_timer) == YES) {
		TIMER_HEAD *timerHead = m_timer->timerHead;
		int eflags;

		eflags = enterCli();
		enter_spinlock(&timerHead->lock);
		{
			unlinkTimer(m_timer, m_timer->timerHead);
			resetRegist(m_timer);
		}
		exit_spinlock(&timerHead->lock);
		exitCli(eflags);
	}
}

/*
 * ޡͭإåν
 */
STATIC INLINE void initOwnerHead(
	TIMER_OWNER *m_owner)
{
	AggregateSingleListConstructor(&m_owner->aggregate);
	m_owner->count = 0;
	m_owner->lock = 0;
}

/*
 * ͭإåѥޡõ
 * return : NOERR or -ENOENT
 */
STATIC int findDelTimer(
	TIMER_OWNER *i_owner,		// ͭإå
	TIMER **o_timer)
{
	AggregateList *aggregate = &i_owner->aggregate;
	IteratorList iterator;

	aggregate->iterator(aggregate, &iterator);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		TIMER *timer = iterator.next(&iterator);
		
		if (isRegistTimer(timer) == NO) {
			*o_timer = timer;
			return NOERR;
		}
	}

	return -ENOENT;
}

/*
 * ͭ󥯤˥ޡƤƤʤп˥ޡơޡ롣
 * return : error number
 */
STATIC int allocTimer(
	const uint initTime,	// ַв˺Ƥꤹ륿ޡ
	void (*handle)(void*),	// ַв˼¹Ԥ
	void *arg,				// ַв˼¹Ԥΰ
	TIMER_OWNER *m_owner,	// ޡͭ
	TIMER **o_timer)
{
	TIMER *timer;

	ASSERT(handle != NULL);

	// ޡõƤʤгƤ
	if (findDelTimer(m_owner, &timer) == -ENOENT) {
		AggregateList *aggregate = &m_owner->aggregate;

		if (TIMER_MAX <= m_owner->count) {
			WARNING(0);
			return -ENOMEM;
		}

		timer = kmalloc(sizeof(*timer));
		if (timer == NULL){
			return -ENOMEM;
		}
		listConstructor(&timer->ownedList, timer);
		aggregate->insertHead(aggregate, &timer->ownedList);
		++m_owner->count;
	}

	listConstructor(&timer->timerList, timer);
	timer->init = initTime;
	timer->handle = handle;
	timer->arg = arg;

	*o_timer = timer;

	return NOERR;
}

/*
 * 顼ϥɥ
 *Գ
 */
STATIC void handleAlarm(void *proc)
{
	procSendSignal(proc, SIGALRM);
}

//================================== PUBLIC ==============================================

/*
 *Գ
 * 󥿡Х륿ޡμ¹
 */
void doIntervalTimer(
	const uint time)		// вᤷ(ms)
{
	doTimer(time, &intervalTimer);
}

/*
 *Գ
 * ꥢ륯åޡμ¹
 */
void doRealTimer(
	const uint time)		// вᤷ(ms)
{
	doTimer(time, &realTimer);
}

/*
 * ƥॿޡƤ
 * return : error number
 */
int setSystemTimer(
	const uint i_time,		// ޡ
	const uint initTime,	// ַв˺ꤹ륿ޡ
	void (*handle)(void*),	// ַв˼¹Ԥ
	void *arg,				// ַв˼¹Ԥΰ
	void **o_timer)
{
	int error;

	enter_spinlock(&systemTimer.lock);
	{
		error = allocTimer(initTime, handle, arg, &systemTimer, (TIMER**) o_timer);
	}
	exit_spinlock(&systemTimer.lock);
	
	if (error == NOERR) {
		registTimer(i_time, *o_timer);
		return NOERR;
	}
	else {
		return error;
	}
}

/*
 *Թ
 * ץޡƤ
 * return : error number
 */
int setTaskTimer(
	const uint i_time,		// ޡ
	const uint initTime,	// ַв˺Ƥꤹ륿ޡ
	void (*handle)(void*),	// ַв˼¹Ԥ
	void *arg,				// ַв˼¹Ԥΰ
	void **o_timer)
{
	int error = allocTimer(initTime, handle, arg, getTaskTimer(getCurrentTask()), (TIMER**)o_timer);
	if (error != NOERR) {
		return error;
	}
	registTimer(i_time, *o_timer);

	return NOERR;
}

/*
 * ޡ
 * return : error number
 */
void freeTimer(
	void *m_timer)
{
	// Ͽ
	unregistTimer(m_timer);
}

/*
 *Թ
 * ץͭޡ
 */
void releaseProcessTimer(
	TIMER_OWNER *m_owner)
{
	AggregateList *aggregate = &m_owner->aggregate;
	TIMER *timer;

	while ((timer = aggregate->getHead(aggregate)) != NULL) {
		// Ͽ
		unregistTimer(timer);

		kfree(timer);
	}
}

/*
 * ֤
 * return : ms
 */
uint getInitTime(
	void *i_timer)
{
	TIMER *timer = i_timer;
	return timer->init;
}

/*
 * Ĥ֤
 *Ի
 * return : Ĥms
 */
uint getRemainTime(
	void *i_timer)
{
	TIMER *timer = i_timer;
	TIMER_HEAD *timerHead = timer->timerHead;
	uint time;
	int eflags;
	
	ASSERT(timer != NULL);

	time = 0;
	if (isRegistTimer(timer) == YES) {
		eflags = enterCli();
		enter_spinlock(&timerHead->lock);
		{
			if (isRegistTimer(timer) == YES) {
				AggregateList *aggregate = &timerHead->aggregate;
				IteratorList iterator;

				aggregate->iterator(aggregate, &iterator);
				while (iterator.hasNext(&iterator) == BOOL_TRUE) {
					TIMER *next = iterator.next(&iterator);

					// Ĥ֤û
					time += next->count;
					if (next == timer) {
						break;
					}
				}
			}
		}
		exit_spinlock(&timerHead->lock);
		exitCli(eflags);
	}

	return time;
}

/*
 * ͭإåϥɥõ
 *Ի
 * return : NOERR or -ENOENT
 */
int findTimerByHandle(
	TIMER_OWNER *i_owner,		// ͭإå
	void (*handle)(void*),
	void **o_timer)
{
	AggregateList *aggregate = &i_owner->aggregate;
	IteratorList iterator;

	aggregate->iterator(aggregate, &iterator);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		TIMER *timer = iterator.next(&iterator);
		
		if (timer->handle == handle) {
			*o_timer = timer;
			return NOERR;
		}
	}

	return -ENOENT;
}

/*
 * timeout --
 *	Execute a function after a specified length of time.
 *
 * untimeout --
 *	Cancel previous timeout function call.
 *
 * callout_handle_init --
 *	Initialize a handle so that using it with untimeout is benign.
 *
 *	See AT&T BCI Driver Reference Manual for specification.  This
 *	implementation differs from that one in that although an 
 *	identification value is returned from timeout, the original
 *	arguments to timeout as well as the identifier are used to
 *	identify entries for untimeout.
 *
 *Գ
 *ա¾ץƤФ礢
 */
void callout_handle_init(struct callout_handle *callout)
{
	callout->callout = NULL;
}

struct callout_handle timeout(
	timeout_t *ftn,
	void *arg, 
	int to_ticks)		// ms
{
	void *timer;
	struct callout_handle callout;

	if (to_ticks == 0) {
		callout.callout = NULL;
		return callout;
	}

	if (setSystemTimer(to_ticks, 0, ftn, arg, &timer) == NOERR) {
		callout.callout = timer;
	}
	else {
		printk("Failed timeout()!\n");
		callout.callout = NULL;
	}

	return callout;
}

/*
 *Գ
 */
void untimeout(
	timeout_t *ftn, 
	void *arg, 
	struct callout_handle callout)
{
	if (callout.callout == NULL){
		return;
	}

	freeTimer(callout.callout);
}

void callout_init(void *c)
{
	// Ȥꤢ⤷ʤ
}

void callout_reset(void *c, int to_ticks, void (*ftn)(void *), void *arg)
{
	// Ȥꤢ⤷ʤ
}

/*
 * return : 0 : ȥåcallout̵ or 1 : ȥåcalloutͭ
 */
int callout_stop(void *c)
{
	// Ȥꤢ⤷ʤ
	return 0;
}

/***************************************************************************************************
 * ¾ޡ
 ***************************************************************************************************/

//================================== PUBLIC =============================================

/*
 * micro seccond ޡ
 */
void micro_timer(
	uint count)		// ¬(s)
{
	uint64 end;

	end = rdtsc();
	end += calcClockFromMicroSecond(count);
	while (rdtsc() < end);
}

/*
 * mili seccond timer
 */
void mili_timer(
	uint count)		// ¬(ms)
{
	uint64 end;

	end = rdtsc();
	end += calcClockFromMiliSecond(count);
	while (rdtsc() < end);
}

/***************************************************************************************************
 * 
 ***************************************************************************************************/

/*
 * ޡν
 *ե֡Ȼ˸ƤӽФ
 */
void initTimer(void)
{
	// ƥॿޡإåν
	initTimerHead();

	// ƥॿޡͭإåν
	initOwnerHead(&systemTimer);
}

/*
 * ץޡν
 *եץ˸ƤӽФ
 */
void initProcessTimer(
	void *m_timer)
{
	initOwnerHead(m_timer);
}

//--------------------------------------------------------------------------------------------------
// ƥॳ
//--------------------------------------------------------------------------------------------------

int sys_alarm(
	uint seconds)
{
	void *timer;

	// 顼ॿޡˤв
	if (findTimerByHandle(getTaskTimer(getCurrentTask()), handleAlarm, &timer) == NOERR) {
		freeTimer(timer);
	}

	// ޡ
	if (0 < seconds){
		return setTaskTimer(seconds * 1000, 0, handleAlarm, getCurrentProc(), &timer);
	}
	else {
		return 0;
	}
}

void sysMicroDelay(
	const uint micro)
{
	micro_timer(micro);
}

//*************************************************************************************************

void test_timer()
{
	AggregateList *aggregate = &intervalTimer.aggregate;
	IteratorList iterator;

	printk("test_timer ");
	aggregate->iterator(aggregate, &iterator);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		TIMER *timer = iterator.next(&iterator);
		printk("%d ", timer->init);
	}
	printk("\n");
}
