/*
 * vfs.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ԣģ
 *	Ƥ벾δؿ롣
 *	ФƼ롣
 *	opendir()λͤlookup()бˤ롣
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/mount.h>
#include <kern/kmalloc.h>
#include <kern/lock.h>
#include <lib/lib_path.h>
#include <kern/devfs.h>
#include <kern/fs.h>
#include <kern/Thread.h>
#include <kern/time.h>
#include <kern/vfs.h>

#include <kern/debug.h>


//#define DEBUG_VFS 1
#ifdef DEBUG_VFS
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


/******************************************************************************************************
 *
 * ۥե륷ƥ
 *
 *  ޥȻȥǥ쥯ȥѹե륪ץե¹Ի˥ȥ꡼ɲäinodeObjǤ롣
 *  ȥ꡼󥯤κǸϺǽΥȥ꡼Ϣ뤹롣
 * 
 * ¥եθ           ¥ե
 * lookup()                           deleteObj()
 * creat()
 *                                   
 * ¥ե򥪡ץ             ¥ե򥯥
 * open()                             close()
 *                                   
 * ۥȥ꡼κ             ۥȥ꡼
 * initVent()                         deleteVent()
 *                                   
 * ۥե륷ƥ³       ۥե륷ƥफ
 * addVent()
 *                                   
 *                            
 * linkVent()                         unlinkVent()
 *******************************************************************************************************/

/*
 * ۥȥ꡼
 * Note!
 *  fsinodeΰ֤ϥåȹ¤Τȥ󥯤ƤΤա
 */
typedef struct VENTRY{
	FS				*fs;		// ե륷ƥ
	void			*inodeObj;	// inode object ݥ
	struct VENTRY	*next;		// 饦ɥ󥯥ꥹȥͥ
	struct VENTRY	*prev;		// 饦ɥ󥯥ꥹȥץӥ塼
	struct VENTRY	*parent;	// ƥȥ꡼
	struct VENTRY	*child;		// ΥΡɤΥȥåץȥ꡼
	struct VENTRY	*mount;		// ޥȥȥ꡼
	struct VENTRY	*mounted;	// ޥȸȥ꡼
	void			*din;		// ǥХǥץ
	int				type;		// ե륿
	ushort			linkCount;	// ץ󡢥ޥȡȥǥ쥯ȥȥå
	uchar			del;		// ׵᥵YES or NO
	uchar			len;		// Name size.
	char			name[0];	// ѥ͡
} VENTRY;

//================================== PRIVATE ============================================

static FS fsInfoTop;							// ե륷ƥϿ󥯥ȥå
static char rootEntArea[sizeof(VENTRY) + 1];
static VENTRY *rootEnt = (VENTRY*)&rootEntArea;	// 롼ȥȥ꡼

//------------------ ۥեåɤιȥå ------------

static  MODIFY_WAIT_LOCK fsModifyLock;			// ۥե륷ƥࡦե륷ƥॳå

/*
 * ۥե륷ƥλȳ
 */
STATIC INLINE void refVfs()
{
	modifyWaitRef(&fsModifyLock);
}

/*
 * ۥե륷ƥλȽλ
 */
STATIC INLINE void refEndVfs()
{
	modifyWaitRefEnd(&fsModifyLock);
}

/*
 * ۥե륷ƥι
 */
STATIC INLINE void modifyVfs()
{
	modifyWaitLock(&fsModifyLock);
}

/*
 * ۥե륷ƥιλ
 */
STATIC INLINE void modifyEndVfs()
{
	modifyWaitUnlock(&fsModifyLock);
}

//------------------ ѥ -------------------------------

/*
 * copyPathBack()Ѿؿ
 */
STATIC int isRoot(const VENTRY *vent)
{
	return (vent == rootEnt) ? YES : NO;
}

STATIC int isMount(const VENTRY *vent)
{
	return (vent->mount != NULL) ? YES : NO;
}

STATIC int isParent(const VENTRY *vent)
{
	return (vent->inodeObj != NULL) ? YES : NO;
}

/*
 *ԻաԺƵ
 * ѥäƥХåե˥ԡ롣
 * return : ԡ
 */
STATIC int copyPathBack(
	const VENTRY *vent,					// 饹ȥȥ꡼
	const size_t size,					// ԡ祵
	int (*condition)(const VENTRY *),	// ̤¾
	char *m_path)						// ԡХåե
{
	size_t cpSize;

	if (condition(vent) == YES) {
		cpSize = (size < vent->len) ? size : vent->len;
		memcpy(m_path, vent->name, cpSize);
		return cpSize;
	}
	else{
		int len;

		len = copyPathBack(vent->parent, size, condition, m_path);
		if (size <= len) {
			return len;
		}
		m_path[len++] = '/';
		cpSize = (size - len < vent->len) ? size - len : vent->len;
		memcpy(&m_path[len], vent->name, cpSize);

		return len + cpSize;
	}
}

/*
 *ԻաԺƵ
 * ѥƥǥ쥯ȥޤäƥХåե˥ԡ롣
 * Υȥ꡼롼Ȥޤϥޥȥ롼ȥȥ꡼ʤ鼫ȥ꡼Υѥꤹ롣
 * return : ԡ
 */
STATIC int copyParentPath(
	const VENTRY *vent,		// 饹ȥȥ꡼
	const size_t size,		// ԡ祵
	char *m_path)			// ԡХåե
{
	int len = copyPathBack(vent->parent, size, &isParent, m_path);

	if (vent != vent->parent) {
		size_t cpSize;

		if (size <= len) {
			return len;
		}
		m_path[len++] = '/';
		cpSize = (size - len < vent->len) ? size - len : vent->len;
		memcpy(&m_path[len], vent->name, cpSize);
		len += cpSize;
	}
	
	return len;
}

/*
 *Ի
 * ƥȥ꡼
 * return : ƥȥ꡼
 */
STATIC VENTRY *getParentVent(const VENTRY *vent)
{
	if (vent->parent->inodeObj != NULL) {
		return vent->parent;
	}
	else {
		return getParentVent(vent->parent);
	}
}

/*
 * pathȥåץǥ쥯ȥ롣
 * parameters : ѥɥ쥹ݥ
 * return : Ȳۥǥ쥯ȥꥨȥ꡼
 */
STATIC VENTRY *getTopVdir(const char *path, const char **o_path)
{
	VENTRY *vent;
	char *p = (char*)path;

	if(*p == '/'){
		vent = rootEnt;
		while(*++p == '/');
		*o_path = p;
	}
	else{
		vent = getCurrentDir();
		*o_path = p;
	}

	return (vent->mount != NULL) ? vent->mount : vent;
}

//------------------ ¥եκ -------------------------------

/*
 *Թ
 * ۥȥ꡼¥֥ȡʥե롦ǥ쥯ȥˤ롣
 */
STATIC void deleteObjFromVent(
	const VENTRY *vent,		// ȥ꡼
	const int type)			// ȥ꡼
{
	char path[PATH_MAX + 1];
	VENTRY *parent;
	int len;

	parent = getParentVent(vent);
	len = copyParentPath(vent, PATH_MAX + 1, path);
	if (PATH_MAX < len){
		return;
	}
	path[len] = '\0';
	parent->fs->delete(path, parent->din, parent->inodeObj, type);
}

/*
 *Թ
 * ¥֥Ȥκե饰ΩƤ
 */
STATIC void unlinkObj(VENTRY *m_vent)
{
	m_vent->del = YES;
}

//------------------ ۥȥ꡼κ ---------------------

/*
 *Թ
 * ȥ꡼ꤹ롣
 */
STATIC void initVent(
	void *devDsc,
	FS *fs,
	void *inodeObj,
	const int type,
	VENTRY *m_vent)
{
	m_vent->din = devDsc;
	m_vent->fs = fs;
	m_vent->inodeObj = inodeObj;
	m_vent->type = type;
	m_vent->del = NO;
}

/*
 *Թ
 * ۥȥ꡼롣
 * return : error number
 */
STATIC int deleteVent(VENTRY *m_vent)		// ȥ꡼
{
	if (0 < m_vent->linkCount) {
		return -EBUSY;
	}
	if (m_vent->child != NULL) {
		return -EBUSY;
	}
	if (m_vent->mount != NULL) {
		return -EBUSY;
	}
	if (m_vent->mounted != NULL) {
		VENTRY *mounted = m_vent->mounted;
		int error;

		m_vent->fs->umount(m_vent->din, m_vent->inodeObj);

		mounted->mount = NULL;
		error = deleteVent(mounted);
		if (error != NOERR) {
			return error;
		}
	}

	//ե򥯥
	if (m_vent->inodeObj != NULL) {
		m_vent->fs->close(m_vent->din, m_vent->inodeObj);
	}
	
	//ե
	if (m_vent->del == YES) {
		deleteObjFromVent(m_vent, m_vent->type);
	}

	//ƥȥ꡼λҥ󥯤᤹
	if (m_vent->parent != NULL) {
		VENTRY *parent = m_vent->parent;

		if (m_vent->next == m_vent) {
			parent->child = NULL;
		}
		else {
			if (parent->child == m_vent) {
				parent->child = m_vent->next;
			}
			m_vent->prev->next = m_vent->next;
			m_vent->next->prev = m_vent->prev;
		}
	}

	kfree(m_vent);

	return NOERR;
}

//------------------ ۥե륷ƥ³ǡ ---------------

/*
 *ԻաԼ
 * Ρ⤫饨ȥ꡼õ
 * return : error number
 */
STATIC int searchVentNode(
	const char *path,
	VENTRY *parent,			// ƥȥ꡼
	VENTRY **o_entry)		// Ĥäȥ꡼
{
	VENTRY *entry;

	if (cmpPath(".", path, 1) == 0){
		entry = parent;
	}
	else if (cmpPath("..", path, 2) == 0){
		entry = parent->parent;
	}
	else {
		// ƱΡɥ󥯤õ
		if (parent->child == NULL){
			return -ENOENT;
		}
		entry = parent->child;
		while (cmpPath(entry->name, path, entry->len) != 0) {
			entry = entry->next;
			if (entry == parent->child){
				return -ENOENT;
			}
		}
	}

	// ޥȥȥ꡼֤
	*o_entry = (entry->mount == NULL) ? entry : entry->mount;

	return NOERR;
}

/*
 *ԹաԼ
 * Ρǥȥ꡼ɲä롣
 * return : error number
 */
STATIC int addVentNode(
	const char *path,
	VENTRY *m_parent,
	VENTRY **o_vent)
{
	VENTRY *vent;
	int error;

	error = searchVentNode(path, m_parent, &vent);
	switch (error) {
	case NOERR:
		// ".."ξϺ
		if (cmpPath("..", path, 2) == 0) {
			deleteVent(m_parent);
		}
		break;
	case -ENOENT:
		vent = kmalloc(sizeof(*vent) + pathLen(path));
		if (vent == NULL) {
			return -ENOMEM;
		}
		memset(vent, 0, sizeof(*vent));

		// 󥯤ɲä
		if (m_parent->child == NULL){
			vent->prev = vent->next = vent;
			m_parent->child = vent;
		}
		else{
			VENTRY *child = m_parent->child;
			vent->prev = child;
			vent->next = child->next;
			child->next->prev = vent;
			child->next = vent;
		}
		vent->parent = m_parent;
		vent->child = NULL;
		vent->len = pathLen(path);
		memcpy(vent->name, path, vent->len);

		break;
	default:
		return error;
	}

	*o_vent = vent;

	return NOERR;
}

/*
 *ԹաԼ
 *ահοƥȥ꡼Ʊե륷ƥǤʤФʤʤ
 * ȥ꡼ѥɲä롣
 * return : error number
 */
STATIC int addVent(
	const char *i_path,		//ƤΥѥ
	VENTRY *m_parent,		//ƥȥ꡼
	VENTRY **o_vent)
{
	const char *path = i_path;
	VENTRY *parent;
	VENTRY *vent;
	int error;

	ASSERT(*path != NULL);

	parent = m_parent;
	for(;;){
		//ȥ꡼ɲ
		error = addVentNode(path, parent, &vent);
		if (error != NOERR) {
			return error;
		}

		path = getNextPath(path);
		if (*path == '\0'){
			break;
		}
		else {
			initVent(parent->din, parent->fs, NULL, S_IFDIR, vent);
		}

		parent = vent;
	}

	*o_vent = vent;

	return error;
}

/*
 *ԻաԼ
 * ѥ饨ȥ꡼򸡺롣
 * return : error number
 */
STATIC int searchVent(
	const char *i_path,			// ѥ
	const VENTRY *i_vent,		// inode֥
	const char **o_parentPath,	// inode֥Ȥƥǥ쥯ȥѥ
	VENTRY **o_parentVent,		// inode֥Ȥƥǥ쥯ȥ겾ۥȥ꡼
	VENTRY **o_vent)			// Ǹβۥȥ꡼
{
	const char *path;
	VENTRY *vent;
	int error;

	path = i_path;
	vent = (VENTRY*) i_vent;
	*o_vent = vent;
	for(;;) {
		// ȥ꡼Ƚꤹ
		if ((vent->inodeObj != NULL) || (vent->mounted != NULL )) {
			*o_parentPath = path;
			*o_parentVent = vent;
			error = NOERR;
		}
		else {
			error = -ENOENT;
		}

		if (*path == '\0') {
			break;
		}
		
		if (vent->type != S_IFDIR) {
			return -ENOTDIR;
		}

		// ҥȥ꡼
		error = searchVentNode(path, vent, &vent);
		if (error != NOERR) {
			return error;
		}

		*o_vent = vent;
		path = getNextPath(path);
	}

	return error;
}

//------------------ ۥȥ꡼Υ󥯡 -------------

/*
 *Ի
 * ۥȥ꡼򥪡ץ󤹤
 */
void linkVent(void *i_vent)
{
	VENTRY *m_vent = i_vent;

	// Ȥꤢ25٤ˤ
	ASSERT(m_vent->linkCount <= 25);

	++m_vent->linkCount;
}

/*
 *Թ
 *դθƤӽФɬΤߤˤʤ뤳ȡƱ֤äƸƤӽФʤȡ
 * ۥȥ꡼򥯥
 */
STATIC void unlinkVent(VENTRY *m_vent)
{
	ASSERT(0 < m_vent->linkCount);

	--m_vent->linkCount;

	if (m_vent->linkCount == 0){
		VENTRY *vent;
		VENTRY *parent;

		// ۥȥ꡼̤äƺ롣
		for(vent = m_vent, parent = vent->parent; deleteVent(vent) == NOERR;) {
			vent = parent;
			parent = vent->parent;
		}
	}
}

//------------------ ܥåбѥθ ----------------------

enum {
	SYMLINK_LOOP_MAX = 10,		// ܥå󥯺ݻ
};

/*
 * ܥå󥯥롼ץåѥơ֥
 */
typedef struct {
	int count;							// ܥå󥯥
	void *din[SYMLINK_LOOP_MAX];		// ǥХinode
	uint inodeNum[SYMLINK_LOOP_MAX];	// indoeֹ
} SYMLINK_LOOP;

/*
 * ܥå󥯥롼ץơ֥ν
 */
STATIC void initSymlinkLoop(SYMLINK_LOOP *m_symlinkLoop)
{
	m_symlinkLoop->count = 0;
}

/*
 * ܥå󥯥롼ץå
 * return : error number
 */
STATIC int symlinkLoop(
	void *devDsc,
	const uint inodeNum,
	SYMLINK_LOOP *m_symlinkLoop)
{
	int i;

	ASSERT(m_symlinkLoop->count <= SYMLINK_LOOP_MAX);
	
	if (m_symlinkLoop->count == SYMLINK_LOOP_MAX) {
		return -ELOOP;
	}
	for (i = 0; i < m_symlinkLoop->count; ++i) {
		if ((m_symlinkLoop->din[i] == devDsc) && (m_symlinkLoop->inodeNum[i] == inodeNum)) {
			return -ELOOP;
		}
	}
	
	m_symlinkLoop->din[m_symlinkLoop->count] = devDsc;
	m_symlinkLoop->inodeNum[m_symlinkLoop->count] = inodeNum;
	++m_symlinkLoop->count;
	
	return NOERR;
}

/*
 *Ի
 * ܥå󥯤Υѥ򥻥åȤ롣
 * return : error number
 */
STATIC int setSymlinkPath(
	void *devDsc,
	const FS *fs,
	const uint inodeNum,
	const char *path,
	const char *symlinkPath,
	char *m_path,					// ѥĹ PATH_MAX + 1 Ǥ뤳
	SYMLINK_LOOP *m_symlinkLoop)
{
	char *pathBuf;
	const char *nextPath;
	void *inodeObj;
	int lenPrev;
	int len;
	int dummy;
	int error;

	error = symlinkLoop(devDsc, inodeNum, m_symlinkLoop);
	if (error != NOERR) {
		return error;
	}

	pathBuf = m_path;

	// ܥå󥯥ѥ
	error = fs->open(devDsc, inodeNum, O_RDONLY, &inodeObj, &dummy);
	if (error != NOERR) {
		return error;
	}
	len = fs->read(devDsc, inodeObj, pathBuf, PATH_MAX, 0);
	if (fs->close(devDsc, inodeObj) != NOERR) {
		ASSERT(0);
	}
	if (len < 0) {
		return len;
	}
	
	if (*pathBuf == '/') {
		// ХѥʤΤ˥ѥղäʤ
		lenPrev = 0;
	}
	else {
		// ܥåޤǤΥѥղ
		lenPrev = symlinkPath - path;
		if (PATH_MAX < lenPrev + len) {
			return -ENAMETOOLONG;
		}
		memmove(pathBuf + lenPrev, pathBuf, len);
		memcpy(pathBuf, path, lenPrev);
	}
	pathBuf += lenPrev + len;

	// ܥå󥯸Υѥ
	nextPath = getNextPath(symlinkPath);
	if (*nextPath != '\0') {
		int lenNext;

		nextPath -= 1;			// '/'ޤ᤹
		lenNext = strnlen(nextPath, PATH_MAX + 1);
		if (PATH_MAX < lenPrev + len + lenNext) {
			return -ENAMETOOLONG;
		}
		memcpy(pathBuf, nextPath, lenNext);
		pathBuf += lenNext;
	}
	*pathBuf = '\0';

	return NOERR;
}

/*
 *Ի
 * ¥ե򸡺
 * return : error number or SYMLINK
 */
STATIC int lookupObj(
	const char *i_path, 	// ѥ
	void *devDsc,
	const FS *fs,
	void *i_inodeObj,
	char *m_path,			// ܥåȯˤĤѥѥХåե
	SYMLINK_LOOP *m_symlinkLoop,
	uint *o_inodeNum)		// inodeֹ
{
	uint inodeNum;
	const char *symlinkPath;
	int error;

	if(*i_path=='\0'){
		return -ENOENT;
	}

	error = fs->lookup(i_path, devDsc, i_inodeObj, &inodeNum, &symlinkPath);
	switch (error) {
	case NOERR:
		break;
	case SYMLINK:		// ܥå
		// ѥ
		error = setSymlinkPath(devDsc, fs, inodeNum, i_path, symlinkPath, m_path, m_symlinkLoop);
		if (error != NOERR) {
			return error;
		}
		return SYMLINK;
	default:
		return error;
	}

	*o_inodeNum = inodeNum;

	return error;
}

/*
 *Ի
 * ¥ե򸡺
 * ܥå󥯤⸡
 * return : error number or SYMLINK
 */
STATIC int lookupObjSymlink(
	const char *i_path, 	// ѥ
	void *devDsc,
	const FS *fs,
	void *i_inodeObj,
	char *m_path,			// ܥåȯˤĤѥѥХåե
	SYMLINK_LOOP *m_symlinkLoop,
	uint *o_inodeNum)		// inodeֹ
{
	uint inodeNum;
	const char *symlinkPath;
	int error;

	if(*i_path=='\0'){
		return -ENOENT;
	}

	error = fs->lookup(i_path, devDsc, i_inodeObj, &inodeNum, &symlinkPath);
	switch (error) {
	case NOERR:
		break;
	case SYMLINK:		// ܥå
		// ѥκǸΥȥ꡼ʤｪλ
		if (*getNextPath(symlinkPath) == '\0') {
			error = NOERR;
		}
		else {
			error = setSymlinkPath(devDsc, fs, inodeNum, i_path, symlinkPath, m_path, m_symlinkLoop);
			if (error != NOERR) {
				return error;
			}
			return SYMLINK;
		}
		break;
	default:
		return error;
	}

	*o_inodeNum = inodeNum;

	return error;
}

/*
 * lookupPathѽϥѥ᡼
 */
typedef struct {
	VENTRY *vent;				// ۥȥ꡼
	void *din;					// ǥХǥץ
	FS *fs;						// ե륷ƥ๽¤
	uint inodeNum;				// inodeֹ
	const char *parentPath;		// ƥǥ쥯ȥѥ
	VENTRY *parentVent;			// ƥǥ쥯ȥinode֥
} LOOKUP_OUT;

enum {
	EXIST_VENT = 1,		// ۥȥ꡼
};

/*
 *Ի
 * ۥȥ꡼ȼ¥եõܥåб
 * Ǥ˲ۥȥ꡼Хȥѥ᡼˲ۥȥ꡼򥻥åȤEXIST_VENT֤
 */
STATIC int lookupPath(
	const char *i_path,
	int lookupFunc(const char *, void*, const FS *, void *, char *, SYMLINK_LOOP *, uint *),
	LOOKUP_OUT *m_lookupOut)
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	char linkPath[PATH_MAX + 1];
	VENTRY *vent;
	SYMLINK_LOOP symlinkLoop;
	uint inodeNum;
	int error;

	initSymlinkLoop(&symlinkLoop);

	//ȥǥ쥯ȥ롣
	vent = getTopVdir(i_path, &path);

	for (;;) {
		error = searchVent(path, vent, &parentPath, &parentVent, &vent);
		if (error == NOERR) {				// ۥեˤ
			m_lookupOut->vent = vent;
			return EXIST_VENT;
		}
		else if (error == -ENOENT) {		// ۥե̵
			error = lookupFunc(parentPath, parentVent->din, parentVent->fs, parentVent->inodeObj, linkPath, &symlinkLoop, &inodeNum);
			if (error == NOERR) {
				// m_lookupOut
				break;
			}
			else if (error == SYMLINK) {	// ܥå
				path = linkPath;
				if (*path == '/') {
					// Хѥξ롼Ȳۥȥ꡼
					vent = getTopVdir(path, &path);
				}
				// searchVent()
			}
			else if (error == -ENOENT){
				// m_lookupOut
				break;
			}
			else {
				return error;
			}
		}
		else {
			return error;
		}
	}

	m_lookupOut->din = parentVent->din;
	m_lookupOut->fs = parentVent->fs;
	m_lookupOut->inodeNum = inodeNum;
	m_lookupOut->parentPath = parentPath;
	m_lookupOut->parentVent = parentVent;

	return error;
}

//------------------ 롼Υåѡ ----------------------

/*
 *Ի
 * ۥեˤвۥȥ꡼̵м¥ե򥪡ץ󤷤Ʋۥե롣
 * return : error number
 */
STATIC int getVent(
	const char *i_path,
	const int flag,			// openե饰
	int lookupFunc(const char *, void*, const FS *, void *, char *, SYMLINK_LOOP *, uint *),
	VENTRY **o_vent)		// inode
{
	LOOKUP_OUT lookupOut;
	int error;

	error = lookupPath(i_path, lookupFunc, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		*o_vent = lookupOut.vent;
		error = NOERR;
		break;
	case NOERR: {		// ۥե̵
		void *inodeObj;
		int type;
		VENTRY *vent;

		error = lookupOut.fs->open(lookupOut.din, lookupOut.inodeNum, flag, &inodeObj ,&type);
		if (error != NOERR){
			break;
		}
		vent = kmalloc(sizeof(*vent));
		if (vent == NULL) {
			lookupOut.fs->close(lookupOut.din, inodeObj);
			error = -ENOENT;
			break;
		}
		memset(vent, 0, sizeof(*vent));
		initVent(lookupOut.din, lookupOut.fs, inodeObj, type, vent);
		*o_vent = vent;
		break;
	}
	default:
		break;
	}

	return error;
}

//------------------ ե륷ƥΥޥ -------------

/*
 *Ի
 * ޥȥǥ쥯ȥμ
 * return : ۥȥ꡼
 */
STATIC INLINE VENTRY *getMountedVent(VENTRY *vent)
{
	return (vent->mounted != NULL) ? vent->mounted : vent;
}

/*
 * struct statfs˳Ǽ롣
 * return : error number
 */
STATIC int getFsstat(VENTRY *i_vent, struct statfs *statfs)
{
	VENTRY *vent;
	int error;
	DEV_STAT devstat;

	vent = (i_vent->mount == NULL) ? i_vent : i_vent->mount;

	if ((error = vent->fs->statfs(vent->din, statfs)) < 0){
		return error;
	}

	statDev(getDevInfo(i_vent->din), &devstat);
	if (devstat.minor > 0){
		sprintk(statfs->f_mntfromname, "/dev/%s%d", devstat.name, devstat.minor);
	}
	else{
		sprintk(statfs->f_mntfromname, "/dev/%s", devstat.name);
	}

	copyPathBack(vent, MNAMELEN, &isMount, statfs->f_mntonname);

	return NOERR;
}

/*
 *ԻաԺƵ
 * ǥ쥯ȥޥȥȥ꡼θ
 * return ޥȿ
 */
STATIC int searchMountDir(VENTRY *vent, struct statfs *m_statfs, uint lenStatfs)
{
	int rest, count;
	VENTRY *vent_begin;

	if (lenStatfs <= 0){
		return 0;
	}
	vent_begin = vent;
	count = lenStatfs;

	do{
		// ޥȤ
		if (vent->mount != NULL){
			while (vent->mount != NULL){
				vent = vent->mount;
			}
			if (m_statfs != NULL){
				if (getFsstat(vent, m_statfs) == NOERR){
					m_statfs += 1;
					count -= 1;
				}
			}
		}

		if (vent->child != NULL){
			rest = searchMountDir(vent->child, m_statfs, count);
			m_statfs += rest;
			count -= rest;
		}
	}while ((vent = vent->next) != vent_begin);

	return lenStatfs - count;
}

STATIC void umountFs(VENTRY *ventry)
{
	VENTRY *vent = ventry;

	do {
		if (vent->child != NULL) {
			umountFs(vent->child);
		}
		if (vent->mount != NULL) {
			vent->mount->fs->umount(vent->mount->din, vent->mount->inodeObj);
		}
	} while ((vent = vent->next) != ventry);
}

/*
 *Թ
 * ޥ襨ȥ꡼饢ޥȤ
 * return : error number
 */
STATIC int umountFromVfs(
	VENTRY *vent)		// ޥ襨ȥ꡼
{
	VENTRY *mountVent;

	if (vent->mount == NULL){
		return -ENOENT;
	}
	mountVent = vent->mount;

	if (mountVent->child != NULL) {
		return -EBUSY;
	}

	return NOERR;
}

/*
 * ۥե륷ƥ򸡺ƥޥȤƤե륷ƥ򥢥ޥȤ롣
 */
void umountAllFs()
{
	umountFs(rootEnt);
}

/*
 * return : error number
 */
STATIC int chmod(
	const VENTRY *vent,
	const mode_t mode)
{
	void *proc = getCurrentProc();
	struct stat stat;
	int error;

	// øΥå
	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != NOERR) {
		return error;
	}
	if ((getUid(proc) != 0) && (getUid(proc) != stat.st_uid)) {
		return -EPERM;
	}

	return vent->fs->chattr(vent->din, vent->inodeObj, mode, stat.st_uid, stat.st_gid, stat.st_atime, stat.st_mtime);
}

/*
 * return : error number
 */
STATIC int chuid(
	const VENTRY *vent,
	const uid_t owner)
{
	void *proc = getCurrentProc();
	struct stat stat;
	int error;

	ASSERT(owner != -1);

	// øΥå
	if (getUid(proc) != 0) {
		return -EPERM;
	}

	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != NOERR) {
		return error;
	}

	return vent->fs->chattr(vent->din, vent->inodeObj, stat.st_mode, owner, stat.st_gid, stat.st_atime, stat.st_mtime);
}

/*
 * return : error number
 */
STATIC int chgrp(
	const VENTRY *vent,
	const gid_t group)
{
	void *proc = getCurrentProc();
	struct stat stat;
	int error;

	ASSERT(group != -1);

	// øΥå
	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != NOERR) {
		return error;
	}
	if (((getUid(proc) != 0) && (getUid(proc) != stat.st_uid)) || (group != getGid(proc))) {
		return -EPERM;
	}

	return vent->fs->chattr(vent->din, vent->inodeObj, stat.st_mode, stat.st_uid, group, stat.st_atime, stat.st_mtime);
}

/*
 * return : error number
 */
STATIC inline int chutime(
	const VENTRY *vent,
	const uint atime,
	const uint mtime)
{
	void *proc = getCurrentProc();
	struct stat stat;
	int error;

	// øΥå
	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != NOERR) {
		return error;
	}
	if ((getUid(proc) != 0) && (getUid(proc) != stat.st_uid)) {
		return -EPERM;
	}

	return vent->fs->chattr(vent->din, vent->inodeObj, stat.st_mode, stat.st_uid, stat.st_gid, atime, mtime);
}

//------------------ ե륷ƥϿ -------------------

/*
 * ե륷ƥϿ롣
 * return : 0 or Error=-1
 */
int regist_fs(FS *info)
{
	FS *p;

	p = &fsInfoTop;
	for (; p->next != NULL; p = p->next) {
		switch (strcmp(info->name, p->next->name)) {
		case 0:
			return -EEXIST;
		case -1:
			goto EXIT;
		}
	}
EXIT:
	info->next = p->next;
	p->next = info;

	return 0;
}

/*
 * Ͽ줿ե륷ƥõ
 * return : FS address or NULL
 */
FS *search_fs(const char *name)
{
	FS *p;

	for (p = fsInfoTop.next; p != NULL; p = p->next) {
		switch (strcmp(name, p->name)) {
		case 0:
			return p;
		case -1:
			return NULL;
		}
	}

	return NULL;
}

//------------------ ۡ¥ե -------------------

/*
 *Թ
 * Υե륷ƥΥ롼inodeޥȤ롣
 * return : error number
 */
int mountToVfs(
	FS *fs,
	void *devDsc,
	void *rootInodeObj,
	void *m_dstVent)
{
	VENTRY *dstVent = m_dstVent;
	VENTRY *mountVent;

	// ۥե륷ƥι
	modifyVfs();

	// Ǥ˥ޥȤƤ뤫
	if (dstVent->mount != NULL){
		return -EBUSY;
	}

	// ޥȥȥ꡼κ
	mountVent = kmalloc(sizeof(VENTRY) + dstVent->len);
	if (mountVent == NULL){
		return -ENOMEM;
	}
	memset(mountVent, 0, sizeof(VENTRY));

	// ޥȥȥ꡼ν
	mountVent->next		= dstVent->next;
	mountVent->prev		= dstVent->prev;
	mountVent->inodeObj	= rootInodeObj;
	mountVent->parent	= dstVent->parent;
	mountVent->mounted	= dstVent;
	mountVent->fs		= fs;
	mountVent->din		= devDsc;
	mountVent->type		= S_IFDIR;
	mountVent->len		= dstVent->len;
	memcpy(mountVent->name, dstVent->name, dstVent->len);
	linkVent(mountVent);

	// ޥȤ
	dstVent->mount = mountVent;

	// ۥե륷ƥιλ
	modifyEndVfs();

	return NOERR;
}

/*
 * return : error number
 */
int umountPath(const char *i_path)
{
	LOOKUP_OUT lookupOut;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = lookupPath(i_path, lookupObj, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		if (lookupOut.vent->type == S_IFDIR){
			error = umountFromVfs(lookupOut.vent);
			unlinkVent(lookupOut.vent);
		}
		else {
			error = -ENOENT;
		}
		break;
	case NOERR: {		// ۥե̵
		error = -ENOENT;
		break;
	}
	default:
		break;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 *Թ
 * ¥եκץ
 * return : error number
 */
int creatObj(
	const char *i_path,
	const mode_t mode)
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	VENTRY *vent;
	uint inodeNum;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	//ȥǥ쥯ȥ롣
	vent = getTopVdir(i_path, &path);

	error = searchVent(path, vent, &parentPath, &parentVent, &vent);
	switch (error){
	case NOERR:		// ۥեˤ
		error = -EEXIST;
		break;
	case -ENOENT:	// ۥե̵
		error = parentVent->fs->creat(parentPath, parentVent->din, parentVent->inodeObj, mode, &inodeNum);
		break;
	default:
		break;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return NOERR;
}

/*
 * return : error number
 */
int mkdirPath(
	const char *i_path,
	const mode_t mode)
{
	LOOKUP_OUT lookupOut;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = lookupPath(i_path, lookupObj, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		//
	case NOERR:			// ¥եˤ
		error = -EEXIST;
		break;
	case -ENOENT:			// ¥ե̵
		error = lookupOut.fs->mkdir(lookupOut.parentPath, lookupOut.din, lookupOut.parentVent->inodeObj, mode);
		break;
	default:
		break;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 *Թ
 * ܥå󥯤κ
 * return : error number
 */
int creatSymlink(
	const char *i_path,		// ܥå󥯥ѥ
	const mode_t mode,		// ܥå󥯥ե⡼
	char *linkPath)			// 󥯥ѥ
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	VENTRY *vent;
	uint inodeNum;
	void *inodeObj;
	int len;
	VENTRY *dummy1;
	int dummy2;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	//ȥǥ쥯ȥ롣
	vent = getTopVdir(i_path, &path);

	error = searchVent(path, vent, &parentPath, &parentVent, &dummy1);
	switch (error){
	case NOERR:		// ۥեˤ
		error = -EEXIST;
		goto ERR;
	case -ENOENT:	// ۥե̵
		break;
	default:
		goto ERR;
	}

	error = parentVent->fs->creat(parentPath, parentVent->din, parentVent->inodeObj, mode, &inodeNum);
	if (error != NOERR) {
		goto ERR;
	}
	error = parentVent->fs->open(parentVent->din, inodeNum, O_WRONLY, &inodeObj ,&dummy2);
	if (error != NOERR){
		goto ERR;
	}
	len = strnlen(linkPath, PATH_MAX + 1);
	error = parentVent->fs->write(parentVent->din, inodeObj, linkPath, len, 0);
	if (0 < error) {
		error = NOERR;
	}
	parentVent->fs->close(parentVent->din, inodeObj);
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * ۥȥ꡼򥪡ץ󤹤
 * ۥȥ꡼ˤʤкƥץ󤹤롣
 * return : error number
 */
int openPath(
	const char *i_path,
	const int flag,
	void **o_vent)
{
	LOOKUP_OUT lookupOut;
	VENTRY *vent;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = lookupPath(i_path, lookupObj, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		vent = lookupOut.vent;
		error = NOERR;
		break;
	case NOERR: {		// ۥե̵
		void *inodeObj;
		int type;

		error = lookupOut.fs->open(lookupOut.din, lookupOut.inodeNum, flag, &inodeObj ,&type);
		if (error != NOERR){
			goto ERR;
		}

		// ۥǥ쥯ȥκ
		error = addVent(lookupOut.parentPath, lookupOut.parentVent, &vent);
		if (error != NOERR){
			lookupOut.fs->close(lookupOut.din, inodeObj);
			goto ERR;
		}

		initVent(lookupOut.din, lookupOut.fs, inodeObj, type, vent);
		break;
	}
	default:
		goto ERR;
	}

	linkVent(vent);
	*o_vent = vent;
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * ¥ե򥪡ץ󤷤Ʋۥȥ꡼֤
 * return : error number
 */
int openObj(
	const char *i_path,
	const int flag,			// openե饰
	void **o_vent)			// inode
{
	int error;
	
	// ۥե륷ƥλȳ
	refVfs();

	error = getVent(i_path, flag, lookupObj, (VENTRY**) o_vent);

	// ۥե륷ƥλȽλ
	refEndVfs();

	return error;
}

/*
 * return : error number
 */
int opendirVent(
	void *i_vent,
	uint *o_block,
	int *o_index)
{
	VENTRY *vent = i_vent;
	int error;

	// ۥե륷ƥλȳ
	refVfs();

	error = vent->fs->opendir(vent->din, vent->inodeObj, o_block, o_index);

	// ۥե륷ƥλȽλ
	refEndVfs();
	
	return error;
}

/*
 * return : error number
 */
int readdirVent(
	void *i_vent,
	uint *m_block,
	int *m_index,
	char *m_name)
{
	VENTRY *vent = i_vent;

	return vent->fs->readdir(vent->din, vent->inodeObj, m_block, m_index, m_name);
}

/*
 * ۥȥ꡼򥯥롣
 * return : error number
 */
void closeVent(void *i_vent)
{
	VENTRY *vent = i_vent;

	ASSERT(vent->parent != NULL);

	// ۥե륷ƥι
	modifyVfs();

	unlinkVent(vent);

	// ۥե륷ƥιλ
	modifyEndVfs();
}

/*
 * getVent()Ǽۥȥ꡼롣
 */
void releaseVent(void *i_vent)
{
	VENTRY *vent = i_vent;

	if (vent->parent == NULL) {
		vent->fs->close(vent->din, vent->inodeObj);
		kfree(vent);
	}
}

/*
 * ¥ե롣
 * return : error number
 */
int deleteObj(
	const char *i_path,
	const int type)
{
	LOOKUP_OUT lookupOut;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = lookupPath(i_path, lookupObjSymlink, &lookupOut);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		unlinkObj(lookupOut.vent);
		error = -EBUSY;
		break;
	case NOERR:			// ۥե̵
		error = lookupOut.fs->delete(lookupOut.parentPath, lookupOut.din, lookupOut.parentVent->inodeObj, type);
		break;
	default:
		break;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * return : error number
 */
int renamePath(
	const char *oldPath,
	const char *newPath)
{
	LOOKUP_OUT lookupOutOld;
	LOOKUP_OUT lookupOutNew;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	// renameե򸡺
	error = lookupPath(oldPath, lookupObjSymlink, &lookupOutOld);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		error = -EBUSY;
		goto ERR;
	case NOERR:			// ¥եˤ
		break;
	default:
		goto ERR;
	}

	// renameե򸡺
	error = lookupPath(newPath, lookupObjSymlink, &lookupOutNew);
	switch (error) {
	case EXIST_VENT:	// ۥեˤ
		//
	case NOERR:			// ¥եˤ
		//
	case -ENOENT:		// ¥ե̵
		break;
	default:
		goto ERR;
	}

	if (lookupOutOld.din == lookupOutNew.din) {
		error = lookupOutOld.fs->rename(lookupOutOld.din, 
			lookupOutOld.parentVent->inodeObj, lookupOutOld.parentPath, 
			lookupOutNew.parentVent->inodeObj, lookupOutNew.parentPath);
	}
	else {
		error = -EXDEV;
	}
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * ۥȥ꡼Υǡɤ
 * return : read bytes or error number
 */
int readVent(
	void *i_vent,
	const size_t size,		// read
	const size_t offset,	// readϥեå
	void *m_buf)			// readХåե
{
	VENTRY *vent = i_vent;

	return vent->fs->read(vent->din, vent->inodeObj, m_buf, size, offset);
}

/*
 * ۥȥ꡼Υǡ˽񤭹
 * return : read bytes or error number
 */
int writeVent(
	void *i_vent,
	const size_t size,		// write
	const size_t offset,	// writeϥեå
	void *m_buf)			// writeХåե
{
	VENTRY *vent = i_vent;

	return vent->fs->write(vent->din, vent->inodeObj, m_buf, size, offset);
}

/*
 * ѥե륹ơȤ
 * return : error number
 */
int statPath(
	const char *i_path,
	struct stat *m_stat)
{
	VENTRY *vent;
	int error;

	// ۥե륷ƥλȳ
	refVfs();

	error = getVent(i_path, O_RDONLY, lookupObj, &vent);
	if (error != NOERR) {
		goto ERR;
	}
	error = vent->fs->stat(vent->din, vent->inodeObj, m_stat);
	releaseVent(vent);
ERR:
	// ۥե륷ƥλȽλ
	refEndVfs();
	return error;
}

/*
 * ѥե륹ơȤ
 * return : error number
 */
int statPathSymlink(
	const char *i_path,
	struct stat *m_stat)
{
	VENTRY *vent;
	int error;

	// ۥե륷ƥλȳ
	refVfs();

	error = getVent(i_path, O_RDONLY, lookupObjSymlink, &vent);
	if (error != NOERR) {
		goto ERR;
	}
	error = vent->fs->stat(vent->din, vent->inodeObj, m_stat);
	releaseVent(vent);
ERR:
	// ۥե륷ƥλȽλ
	refEndVfs();
	
	return error;
}

/*
 * return : error number
 */
int statVent(
	void *i_vent,
	struct stat *m_stat)
{
	VENTRY *vent = i_vent;
	
	return vent->fs->stat(vent->din, vent->inodeObj, m_stat);
}

/*
 * return : error number
 */
int statfsPath(
	const char *i_path,
	struct statfs *m_statfs)
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	VENTRY *vent;
	int error;

	// ۥե륷ƥλȳ
	refVfs();

	//ȥǥ쥯ȥ롣
	vent = getTopVdir(i_path, &path);

	/* ǥ쥯ȥõ */
	error = searchVent(path, vent, &parentPath, &parentVent, &vent);
	switch (error){
	case NOERR:		// ۥեˤ
		if (vent->mounted == NULL) {
			VENTRY *p;

			for (p = vent; p != rootEnt; p = p->parent) {
				if (p->din == vent->din){
					break;
				}
			}
			memcpy(m_statfs->f_mntonname, p->name, p->len);			/* directory on which mounted */
			m_statfs->f_mntonname[p->len] = '\0';
		}
		else{
			memcpy(m_statfs->f_mntonname, vent->name, vent->len);	/* directory on which mounted */
			m_statfs->f_mntonname[vent->len] = '\0';
		}
		break;
	case -ENOENT:	// ۥե̵
		goto ERR;
	default:
		goto ERR;
	}
	
	error = getFsstat(vent, m_statfs);

	// ۥե륷ƥλȽλ
	refEndVfs();
ERR:
	return error;
}

/*
 * return : error number
 */
int readlinkPath(
	const char *i_path,
	const size_t bufSize,
	char *readBuf)
{
	VENTRY *vent;
	struct stat stat;
	size_t size;
	int error;

	// ۥե륷ƥλȳ
	refVfs();

	error = getVent(i_path, O_RDONLY, lookupObjSymlink, &vent);
	if (error != NOERR) {
		goto  ERR;
	}

	/* ե륿פǧ */
	error = vent->fs->stat(vent->din, vent->inodeObj, &stat);
	if (error != 0){
		goto ERR2;
	}
	if ((stat.st_mode & S_IFMT) != S_IFLNK){
		error = -EINVAL;
		goto ERR2;
	}
	
	/* 󥯥ѥɤ */
	size = (stat.st_size < bufSize)? stat.st_size : bufSize;
	error = vent->fs->read(vent->din, vent->inodeObj, readBuf, size, 0);
ERR2:	
	releaseVent(vent);
ERR:
	// ۥե륷ƥλȽλ
	refEndVfs();

	return error;
}

/*
 * return : error number
 */
int chmodPath(
	const char *i_path,
	const mode_t mode)
{
	VENTRY *vent;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = getVent(i_path, O_RDWR, lookupObj, &vent);
	if (error != NOERR) {
		goto ERR;
	}
	error = chmod(vent, mode);
	releaseVent(vent);
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * return : error number
 */
int chmodVent(
	void *i_vent,
	const mode_t mode)
{
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = chmod(i_vent, mode);

	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * return : error number
 */
int chownPath(
	const char *i_path,
	const uid_t owner,
	const gid_t group)
{
	VENTRY *vent;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = getVent(i_path, O_RDWR, lookupObjSymlink, &vent);
	if (error != NOERR) {
		goto ERR;
	}

	if (owner != -1) {
		error = chuid(vent, owner);
		goto ERR2;
	}
	if (group != -1) {
		error = chgrp(vent, group);
	}
ERR2:
	releaseVent(vent);
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * return : error number
 */
int chownVent(
	void *i_vent,
	const uid_t owner,
	const gid_t group)
{
	VENTRY *vent = i_vent;
	int error = NOERR;

	// ۥե륷ƥι
	modifyVfs();

	if (owner != -1) {
		error = chuid(vent, owner);
		goto ERR;
	}
	if (group != -1) {
		error = chgrp(vent, group);
	}
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * return : error number
 */
int utimesPath(
	const char *i_path,
	const struct timeval *times)
{
	VENTRY *vent;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = getVent(i_path, O_RDWR, lookupObj, &vent);
	if (error != NOERR) {
		goto ERR;
	}

	if (times != NULL) {
		error = chutime(vent, times[0].tv_sec, times[1].tv_sec);
	}
	else {
		error = chutime(vent, sys_time(NULL), sys_time(NULL));
	}

	releaseVent(vent);
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * return : error number
 */
int linkPath(
	const char *fromPath,
	const char *toPath)
{
	const char *path;
	const char *parentPath;
	VENTRY *parentVent;
	VENTRY *fromVent;
	VENTRY *vent;
	int error;

	// ۥե륷ƥι
	modifyVfs();

	error = getVent(fromPath, O_RDONLY, lookupObj, &fromVent);
	if (error != NOERR) {
		goto ERR;
	}
	
	//ȥǥ쥯ȥ롣
	vent = getTopVdir(toPath, &path);

	/* toPathΥե򸡺롣 */
	error = searchVent(path, vent, &parentPath, &parentVent, &vent);
	switch (error){
	case NOERR:		// ۥեˤ
		error = -EEXIST;
		break;
	case -ENOENT:	// ۥե̵
		if (parentVent->din != fromVent->din) {
			error = -EXDEV;
		}
		else {
			error = parentVent->fs->link(parentVent->din, fromVent->inodeObj, parentVent->inodeObj, parentPath);
		}
		break;
	default:
		break;
	}

	releaseVent(fromVent);
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

/*
 * return : error number
 */
int ioctlVent(
	void *i_vent,
	const int cmd,
	caddr_t parm,
	int fflag)
{
	VENTRY *vent = i_vent;

	return vent->fs->ioctl(vent->din, vent->inodeObj, cmd, parm, fflag);
}

/*
 * return : ۥȥ꡼
 */
void *getChdirVentPath(
	const char *i_path,
	int *o_error)
{
	VENTRY *vent;
	VENTRY *mountedVent;

	*o_error = openPath(i_path, O_RDWR, (void**) &vent);
	if (*o_error != NOERR) {
		return NULL;
	}
	mountedVent = getMountedVent(vent);
	if (mountedVent != vent) {
		closeVent(vent);
		linkVent(mountedVent);
	}
	if (mountedVent->type != S_IFDIR){
		closeVent(mountedVent);
		*o_error = -ENOTDIR;
		return NULL;
	}

	return mountedVent;
}

/*
 * return : ۥȥ꡼
 */
void *getChdirVent(
	void *i_vent,
	int *o_error)
{
	VENTRY *vent;

	vent = getMountedVent(i_vent);
	if (vent->type != S_IFDIR) {
		*o_error = -ENOTDIR;
		return NULL;
	}
	linkVent(vent);

	*o_error = NOERR;
	return vent;
}

/*
 * retern : poll condition
 */
int pollVent(
	void *i_vent,
	int events)			// events
{
	VENTRY *vent = i_vent;

	return vent->fs->poll(vent->inodeObj, events);
}

/*
 * ѥ롼ȤޤäƥХåե˥ԡ롣
 * return : ԡ
 */
int copyRootPath(
	const void *vent,		// 饹ȥȥ꡼
	const size_t size,		// ԡ祵
	char *m_path)			// ԡХåե
{
	int len;

	// ۥե륷ƥλȳ
	refVfs();

	len = copyPathBack(vent, size, isRoot, m_path);

	// ۥե륷ƥλȽλ
	refEndVfs();
	
	return len;
}

/*
 * ޥȥȥ꡼򸡺롣
 * return : struct statfsسǼ or ޥȿstruct statfs=NULL
 */
int searchMount(struct statfs *m_statfs, const int i_count)
{
	struct statfs *statfs = m_statfs;
	int count = i_count;
	int rest;

	if (count == 0) {
		return 0;
	}
	
	// ۥե륷ƥλȳ
	refVfs();

	if (statfs != NULL){
		// rootե륷ƥ
		getFsstat(rootEnt, statfs);
		statfs += 1;
	}
	count -= 1;
	rest = searchMountDir(rootEnt->child, statfs, count);

	// ۥե륷ƥλȽλ
	refEndVfs();

	return rest + 1;
}

/*
 * ۥȥ꡼ե륿פ
 * return : ե륿
 */
int getTypeVent(void *i_vent)
{
	VENTRY *vent = i_vent;

	return vent->type;
}

/*
 * ۥȥ꡼inode֥Ȥ
 * return : inode֥
 */
void *getInodeObjVent(void *i_vent)
{
	VENTRY *vent = i_vent;

	return vent->inodeObj;
}

/*
 * 롼Ȳۥȥ꡼롣
 * return : 롼Ȳۥȥ꡼
 */
void *getRootEnt()
{
	return rootEnt;
}

/*
 * ü̾롣
 * return : error number
 */
int getTtynameVent(
	void *i_vent,
	char *m_name)
{
	VENTRY *vent = i_vent;

	if (getDevInodeNum(vent->din) == DEVICE_ROOT_INODE) {
		return getTtyname(vent->inodeObj, m_name);
	}
	else {
		return -ENODEV;
	}
}

//------------------ ۥե륷ƥν -------------------

void initVfs()
{
	/* rootȥ꡼ν */
	rootEnt->fs			= NULL;
//	rootEnt->inodeObj	= NULL;
	rootEnt->inodeObj	= (void*) 1;	//롼ȥե륷ƥޥȻ searchPath()ǥ顼ˤʤα޽
	rootEnt->next		= rootEnt;
	rootEnt->prev		= rootEnt;
	rootEnt->parent		= rootEnt;
	rootEnt->child		= NULL;
	rootEnt->mount		= NULL;
	rootEnt->linkCount	= 2;		// ǥեȥȥǥ쥯ȥʤΤǣꤹ롣
	rootEnt->din		= 0;
	rootEnt->type		= S_IFDIR;
	rootEnt->len		= 1;
	rootEnt->name[0]	= '/';

	// ȥåν
	initModifyWaitLock(&fsModifyLock);
}
