/*	$NetBSD: if_media.c,v 1.1 1997/03/17 02:55:15 thorpej Exp $	*/
/* $FreeBSD: src/sys/net/if_media.c,v 1.9.2.4 2001/07/04 00:12:38 brooks Exp $ */

/*
 * Copyright (c) 1997
 *	Jonathan Stone and Jason R. Thorpe.  All rights reserved.
 *
 * This software is derived from information provided by Matt Thomas.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Jonathan Stone
 *	and Jason R. Thorpe for the NetBSD Project.
 * 4. The names of the authors may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * BSD/OS-compatible network interface media selection.
 *
 * Where it is safe to do so, this code strays slightly from the BSD/OS
 * design.  Software which uses the API (device drivers, basically)
 * shouldn't notice any difference.
 *
 * Many thanks to Matt Thomas for providing the information necessary
 * to implement this interface.
 */


#include <sys/param.h>
#include <sys/systm.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <sys/malloc.h>
#include <net/if.h>
#include <net/if_media.h>


//================================== PRIVATE ============================================


/*
 * Find media entry matching a given ifm word.
 */
static struct ifmedia_entry *ifmedia_match(
	struct ifmedia *ifm,
	int target,
	int mask)
{
	struct ifmedia_entry *match, *next;

	match = NULL;
	mask = ~mask;

	LIST_FOREACH(next, &ifm->ifm_list, ifm_list) {
		if ((next->ifm_media & mask) == (target & mask)) {
			match = next;
		}
	}

	return match;
}


//================================== PUBLIC =============================================


/*
 * Device-independent media ioctl support function.
 */
int ifmedia_ioctl(
	struct ifnet *ifp,
	struct ifreq *ifr,
	struct ifmedia *ifm,
	u_long cmd)
{
	struct ifmedia_entry *match;
	struct ifmediareq *ifmr = (struct ifmediareq *) ifr;
	int error = 0, sticky;

	if (ifp == NULL || ifr == NULL || ifm == NULL){
		return -EINVAL;
	}

	switch (cmd) {
	/*
	 * Set the current media.
	 */
	case  SIOCSIFMEDIA:
	{
		struct ifmedia_entry *oldentry;
		int oldmedia;
		int newmedia = ifr->ifr_media;

		match = ifmedia_match(ifm, newmedia, ifm->ifm_mask);
		if (match == NULL) {
			return -ENXIO;
		}

		/*
		 * If no change, we're done.
		 * XXX Automedia may invole software intervention.
		 *     Keep going in case the the connected media changed.
		 *     Similarly, if best match changed (kernel debugger?).
		 */
		if ((IFM_SUBTYPE(newmedia) != IFM_AUTO) &&
		    (newmedia == ifm->ifm_media) &&
		    (match == ifm->ifm_cur))
			return 0;

		/*
		 * We found a match, now make the driver switch to it.
		 * Make sure to preserve our old media type in case the
		 * driver can't switch.
		 */
		oldentry = ifm->ifm_cur;
		oldmedia = ifm->ifm_media;
		ifm->ifm_cur = match;
		ifm->ifm_media = newmedia;
		error = (*ifm->ifm_change)(ifp);
		if (error) {
			ifm->ifm_cur = oldentry;
			ifm->ifm_media = oldmedia;
		}
		break;
	}
	/*
	 * Get list of available media and current media on interface.
	 */
	case  SIOCGIFMEDIA: 
	{
		struct ifmedia_entry *ep;
		int *kptr, count;
		int usermax;	/* user requested max */

		kptr = NULL;		/* XXX gcc */

		ifmr->ifm_active = ifmr->ifm_current = ifm->ifm_cur ? ifm->ifm_cur->ifm_media : IFM_NONE;
		ifmr->ifm_mask = ifm->ifm_mask;
		ifmr->ifm_status = 0;
		(*ifm->ifm_status)(ifp, ifmr);

		count = 0;
		usermax = 0;

		/*
		 * If there are more interfaces on the list, count
		 * them.  This allows the caller to set ifmr->ifm_count
		 * to 0 on the first call to know how much space to
		 * allocate.
		 */
		LIST_FOREACH(ep, &ifm->ifm_list, ifm_list)
			usermax++;

		/*
		 * Don't allow the user to ask for too many
		 * or a negative number.
		 */
		if (ifmr->ifm_count > usermax)
			ifmr->ifm_count = usermax;
		else if (ifmr->ifm_count < 0)
			return (-EINVAL);

		if (ifmr->ifm_count != 0) {
			kptr = (int *)malloc(ifmr->ifm_count * sizeof(int), M_TEMP, M_WAITOK);

			/*
			 * Get the media words from the interface's list.
			 */
			ep = LIST_FIRST(&ifm->ifm_list);
			for (; ep != NULL && count < ifmr->ifm_count; ep = LIST_NEXT(ep, ifm_list), count++)
				kptr[count] = ep->ifm_media;

			if (ep != NULL)
				error = -E2BIG;	/* oops! */
		}
		else {
			count = usermax;
		}

		/*
		 * We do the copyout on E2BIG, because that's
		 * just our way of telling userland that there
		 * are more.  This is the behavior I've observed
		 * under BSD/OS 3.0
		 */
		sticky = error;
		if ((error == 0 || error == -E2BIG) && ifmr->ifm_count != 0) {
			error = copyout((caddr_t)kptr, (caddr_t)ifmr->ifm_ulist, ifmr->ifm_count * sizeof(int));
		}
		if (error == 0)
			error = sticky;

		if (ifmr->ifm_count != 0)
			free(kptr, M_TEMP);

		ifmr->ifm_count = count;
		break;
	}

	default:
		return (-EINVAL);
	}

	return (error);
}


/*
 * Add a media configuration to the list of supported media
 * for a specific interface instance.
 */
void ifmedia_add(ifm, mword, data, aux)
	struct ifmedia *ifm;
	int mword;
	int data;
	void *aux;
{
	struct ifmedia_entry *entry;

	entry = malloc(sizeof(*entry), M_IFADDR, M_NOWAIT);
	if (entry == NULL)
		panic("ifmedia_add: can't malloc entry");

	entry->ifm_media = mword;
	entry->ifm_data = data;
	entry->ifm_aux = aux;

	LIST_INSERT_HEAD(&ifm->ifm_list, entry, ifm_list);
}


/*
 * Set the default active media. 
 *
 * Called by device-specific code which is assumed to have already
 * selected the default media in hardware.  We do _not_ call the
 * media-change callback.
 */
void ifmedia_set(ifm, target)
	struct ifmedia *ifm; 
	int target;

{
	struct ifmedia_entry *match;

	match = ifmedia_match(ifm, target, ifm->ifm_mask);

	if (match == NULL) {
		printf("ifmedia_set: no match for 0x%x/0x%x\n", target, ~ifm->ifm_mask);
		panic("ifmedia_set");
	}
	ifm->ifm_cur = match;
}


void ifmedia_removeall(ifm)
	struct ifmedia *ifm;
{
	struct ifmedia_entry *entry;

	for (entry = LIST_FIRST(&ifm->ifm_list); entry; entry = LIST_FIRST(&ifm->ifm_list)) {
		LIST_REMOVE(entry, ifm_list);
		free(entry, M_IFADDR);
	}
}


/*
 * Initialize if_media struct for a specific interface instance.
 */
void ifmedia_init(ifm, dontcare_mask, change_callback, status_callback)
	struct ifmedia *ifm;
	int dontcare_mask;
	ifm_change_cb_t change_callback;
	ifm_stat_cb_t status_callback;
{

	LIST_INIT(&ifm->ifm_list);
	ifm->ifm_cur = NULL;
	ifm->ifm_media = 0;
	ifm->ifm_mask = dontcare_mask;		/* IF don't-care bits */
	ifm->ifm_change = change_callback;
	ifm->ifm_status = status_callback;
}
